﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief シェアページを開くのに利用する API 宣言(内部開発者向け)
 */

#pragma once

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/web/web_ShareShowArgPrivate.h>
#include <nn/web/common/web_CommonReturnData.h>

namespace nn { namespace web { namespace detail {

//--------------------------------------------------------------------------
/**
 * @brief シェアページを開く際のパラメータを設定する為のクラスです。
 */
class ShowSharePageArgDetail
{
public:
    /**
     * @brief コールバックURLを設定します。
     *
     * @param[in,out]   pDst            設定対象となるデータを指定してください。
     * @param[in]       pCallbackUrl    コールバック URL を指定してください。
     *
     * @details pCallbackUrl は NULL 終端されている必要があります。
     *          CallbackUrlMaxLength -1 文字以下にする必要があります。
     *          二度呼び出した場合、起動引数は初期化されます。
     */
    static void SetCallbackUrl(uint8_t* pDst, const char* pCallbackUrl) NN_NOEXCEPT;
};

//--------------------------------------------------------------------------
/**
 * @brief シェアページを開いた結果の情報を Cruiser 側で設定する為のクラスです。
 */
class SharePageReturnValueDetail
    : protected SharePageReturnValue
{
public:
    /**
     * @brief 終了理由を設定します。
     *
     * @param[in]   shareExitReason   終了理由を指定してください。
     */
    inline void SetShareExitReason(ShareExitReason shareExitReason) NN_NOEXCEPT;

    /**
     * @brief 終了時開いていたページの URL を設定します。
     *
     * @details     LastUrlMaxLength より長い場合は欠落します。
     *
     * @param[in]   lastUrl 終了時開いていたページの URL を指定してください。
     */
    inline void SetLastUrl(const char* lastUrl) NN_NOEXCEPT;

    /**
    * @brief 画像投稿の成否結果を設定します。
    *
    * @param[in]   sharePostResult 画像投稿の成否結果を指定してください。
    */
    inline void SetSharePostResult(SharePostResult sharePostResult) NN_NOEXCEPT;

    /**
    * @brief 画像投稿した Web サービスサイト名を設定します。
    *
    * @details     PostServiceNameMaxLength より長い場合は欠落します。
    *
    * @param[in]   postServiceName 画像投稿した Web サービスサイト名を指定してください。
    */
    inline void SetPostServiceName(const char* postServiceName) NN_NOEXCEPT;

    /**
    * @brief 画像投稿 ID を設定します。
    *
    * @details     PostIdLength より長い場合は欠落します。
    *
    * @param[in]   postId 画像投稿 ID を指定してください。
    */
    inline void SetPostId(const char* postId) NN_NOEXCEPT;
};

//------------------------------------------------------------------------
void SharePageReturnValueDetail::SetShareExitReason(ShareExitReason shareExitReason) NN_NOEXCEPT
{
    nn::web::common::CommonReturnData::SetExitReason(m_Data, static_cast<nn::web::common::ExitReason>(shareExitReason));
}

//------------------------------------------------------------------------
void SharePageReturnValueDetail::SetLastUrl(const char* lastUrl) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(lastUrl);
    size_t size = std::strlen(lastUrl) + 1;
    NN_SDK_ASSERT(size <= LastUrlMaxLength);

    if (LastUrlMaxLength < size)
    {
        size = LastUrlMaxLength;
    }

    nn::web::common::CommonReturnData::SetData(m_Data, nn::web::common::CommonReturnData::ReturnKind::LastUrl, LastUrlMaxLength, lastUrl);
    nn::web::common::CommonReturnData::SetLastUrlSize(m_Data, size);
}

//------------------------------------------------------------------------
void SharePageReturnValueDetail::SetSharePostResult(SharePostResult sharePostResult) NN_NOEXCEPT
{
    nn::web::common::CommonReturnData::SetPostResult(m_Data, static_cast<nn::web::common::PostResult>(sharePostResult));
}

//------------------------------------------------------------------------
void SharePageReturnValueDetail::SetPostServiceName(const char* postServiceName) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(postServiceName);
    size_t size = std::strlen(postServiceName) + 1;
    NN_SDK_ASSERT(size <= PostServiceNameMaxLength);

    if (PostServiceNameMaxLength < size)
    {
        size = PostServiceNameMaxLength;
    }

    nn::web::common::CommonReturnData::SetData(m_Data, nn::web::common::CommonReturnData::ReturnKind::PostServiceName, PostServiceNameMaxLength, postServiceName);
    nn::web::common::CommonReturnData::SetPostServiceNameSize(m_Data, size);
}

//------------------------------------------------------------------------
void SharePageReturnValueDetail::SetPostId(const char* postId) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(postId);
    size_t size = std::strlen(postId) + 1;
    NN_SDK_ASSERT(size <= PostIdMaxLength);

    if (PostIdMaxLength < size)
    {
        size = PostIdMaxLength;
    }

    nn::web::common::CommonReturnData::SetData(m_Data, nn::web::common::CommonReturnData::ReturnKind::PostId, PostIdMaxLength, postId);
    nn::web::common::CommonReturnData::SetPostIdSize(m_Data, size);
}

//--------------------------------------------------------------------------
/**
 * @brief 2.0 以前の Shim からシェアページを開いた結果の情報を Cruiser 側で設定する為のクラスです。
 */
class SharePageReturnValue2_0
{
public:
    /**
     * @brief NULL 終端を含む最後に開いていたページの URL の最大バイト数です。
     */
    static const size_t LastUrlMaxLength = 4096;

    /**
     * @brief 有効な情報を持たないクラスを作成します。
     */
    SharePageReturnValue2_0() NN_NOEXCEPT;

    /**
     * @brief 終了理由を設定します。
     *
     * @param[in]   shareExitReason   終了理由を指定してください。
     */
    void SetShareExitReason(ShareExitReason shareExitReason) NN_NOEXCEPT;

    /**
     * @brief 終了時開いていたページの URL を設定します。
     *
     * @details     LastUrlMaxLength より長い場合は欠落します。
     *
     * @param[in]   lastUrl 終了時開いていたページの URL を指定してください。
     */
    void SetLastUrl(const char* lastUrl) NN_NOEXCEPT;

private:
    /**
     * @internal
     */
    uint64_t m_ExitReason;
    /**
     * @internal
     */
    char m_LastUrl[LastUrlMaxLength];
    /**
     * @internal
     */
    uint64_t m_LastUrlSize;
};

}}} // namespace nn::web::detail
