﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/nn_Result.h>

namespace nn { namespace rid {

    /**
     * @brief    コントローラー更新の進行状況です。
     */
    struct ControllerUpdateProgress
    {
        enum class State
        {
            DoNothing,        //!< コントローラー更新処理が始まっていない、または開始処理を行っている
            Applying,         //!< コントローラー更新を適用している
            Completed,        //!< コントローラー更新が完了した
            Failed,           //!< コントローラー更新に失敗した
            NeedNoUpdate,     //!< コントローラー更新の必要がなかった
            Cancelled,        //!< コントローラー更新がキャンセルされた
        }state;

        int64_t done;         //!< 終了したステップ数です。state=Applying のときのみ参照可能です。
        int64_t total;        //!< トータルのステップ数です。state=Applying のときのみ参照可能です。
    };

    /**
     * @brief    コントローラー更新を実行する処理を行うクラスです。
     */
    class ControllerUpdater
    {
    public:
        ControllerUpdater() NN_NOEXCEPT;
        ~ControllerUpdater() NN_NOEXCEPT {};

        /**
         * @brief    コントローラー更新を実行します。
         *
         * @details  コントローラー更新に必要な処理をこの関数の中で行います。
         *           コントローラー更新の必要がなかった場合も成功が返ります。
         *
         * @return   処理の結果が返ります。
         * @retval   ResultControllerUpdateCancelled   コントローラーの更新がキャンセルされました。
         */
        Result Execute() NN_NOEXCEPT;

        /**
         * @brief    コントローラー更新をキャンセルします。
         *
         * @details  コントローラー更新の適用中にキャンセルを行うことはありません。
         *           この関数の呼び出しから実際にキャンセルが行われるまで時間がかかる可能性があります。
         */
        void Cancel() NN_NOEXCEPT;

        /**
         * @brief    コントローラー更新の進行状況を取得します。
         *
         * @return   コントローラー更新の進行状況が返ります。
         */
        ControllerUpdateProgress GetProgress() NN_NOEXCEPT;

    private:
        Result ExecuteImpl() NN_NOEXCEPT;
        void UpdateProgress(ControllerUpdateProgress::State state, int64_t done = 0, int64_t total = 0) NN_NOEXCEPT;

        ControllerUpdateProgress m_Progress;
        os::Mutex m_Mutex;
        bool m_IsCancelled;
    };
}}
