﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/migration/detail/migration_Diagnosis.h>
#include <nn/migration/user/migration_UserMigrationProxyBase.h>

namespace nn { namespace migration { namespace user {

template <typename ObjectAllocator, typename Impl>
class ServerProxy
    : public ProxyBase<ObjectAllocator, Impl>
{
public:
    typedef ProxyBase<ObjectAllocator, Impl> ProxyBaseType;
    typedef sf::SharedPointer<ServerProxy<ObjectAllocator, Impl>> CurrentPtr;

    using ProxyBaseType::RequiredWorkMemorySize;

private:
    // 非同期処理のアダプタ
    template <Result(Impl::*ExecuteImpl)(const detail::Cancellable*)>
    struct Task
    {
        CurrentPtr m_pParent;
        explicit Task(CurrentPtr&& parent) NN_NOEXCEPT
            : m_pParent(std::move(parent))
        {
        }
        Result Execute(const detail::Cancellable* pCancellable) NN_NOEXCEPT
        {
            return (m_pParent->GetImplRef().*ExecuteImpl)(pCancellable);
        }
    };

public:
    template <typename... Args>
    explicit ServerProxy(Args&&... args) NN_NOEXCEPT
        : ProxyBaseType(std::forward<Args>(args)...)
    {
        NN_MIGRATION_DETAIL_TRACE("[Server] Object size: %zu (Impl size:  %zu)\n", sizeof(*this), sizeof(Impl));
    }

    Result Initialize(const account::Uid& uid, const UserMigrationServerProfile& profile) NN_NOEXCEPT
    {
        return this->GetImplRef().Initialize(uid, profile);
    }
    Result Resume() NN_NOEXCEPT
    {
        return this->GetImplRef().Resume();
    }

    Result GetUid(sf::Out<account::Uid> pOut) const NN_NOEXCEPT
    {
        *pOut = this->GetImplRef().GetUid();
        NN_RESULT_SUCCESS;
    }

    Result GetServerProfile(sf::Out<UserMigrationServerProfile> pOut) const NN_NOEXCEPT
    {
        this->GetImplRef().GetServerProfile(pOut.GetPointer());
        NN_RESULT_SUCCESS;
    }

    Result PrepareAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOut) NN_NOEXCEPT
    {
        auto p = this->template CreateAsyncContext<Task<&Impl::Prepare>>(CurrentPtr(this, true));
        NN_RESULT_THROW_UNLESS(p, detail::ResultOutOfSessionObject());
        *pOut = p;
        NN_RESULT_SUCCESS;
    }

    Result GetConnectionRequirement(sf::Out<bool> pOut) NN_NOEXCEPT
    {
        *pOut = this->GetImplRef().IsConnectionRequired();
        NN_RESULT_SUCCESS;
    }

    Result WaitConnectionAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOut) NN_NOEXCEPT
    {
        auto p = this->template CreateAsyncContext<Task<&Impl::WaitConnection>>(CurrentPtr(this, true));
        NN_RESULT_THROW_UNLESS(p, detail::ResultOutOfSessionObject());
        *pOut = p;
        NN_RESULT_SUCCESS;
    }

    Result GetClientProfile(sf::Out<UserMigrationClientProfile> pOut)
    {
        this->GetImplRef().GetClientProfile(pOut.GetPointer());
        NN_RESULT_SUCCESS;
    }

    Result AcceptConnectionAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOut) NN_NOEXCEPT
    {
        auto p = this->template CreateAsyncContext<Task<&Impl::AcceptConnection>>(CurrentPtr(this, true));
        NN_RESULT_THROW_UNLESS(p, detail::ResultOutOfSessionObject());
        *pOut = p;
        NN_RESULT_SUCCESS;
    }

    Result DeclineConnectionAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOut) NN_NOEXCEPT
    {
        auto p = this->template CreateAsyncContext<Task<&Impl::DeclineConnection>>(CurrentPtr(this, true));
        NN_RESULT_THROW_UNLESS(p, detail::ResultOutOfSessionObject());
        *pOut = p;
        NN_RESULT_SUCCESS;
    }

    Result ProcessTransferAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOut) NN_NOEXCEPT
    {
        auto p = this->template CreateAsyncContext<Task<&Impl::ProcessTransfer>>(CurrentPtr(this, true));
        NN_RESULT_THROW_UNLESS(p, detail::ResultOutOfSessionObject());
        *pOut = p;
        NN_RESULT_SUCCESS;
    }

    Result CompleteAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOut) NN_NOEXCEPT
    {
        auto p = this->template CreateAsyncContext<Task<&Impl::Complete>>(CurrentPtr(this, true));
        NN_RESULT_THROW_UNLESS(p, detail::ResultOutOfSessionObject());
        *pOut = p;
        NN_RESULT_SUCCESS;
    }

    Result Abort() NN_NOEXCEPT
    {
        return this->GetImplRef().Abort();
    }
};

}}} // ~namesapce nn::migration::user
