﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_SaveDataTransfer.h>
#include <nn/migration/detail/migration_IdcServer.h>
#include <nn/migration/detail/migration_SaveDataTransfer.h>
#include <nn/migration/user/migration_UserMigrationContext.h>
#include <nn/migration/user/migration_UserMigrationStateController.h>

namespace nn { namespace migration { namespace user {

template <typename ConnectionType, typename EncryptionPolicy>
class IdcServer
    : public detail::IdcServer<ConnectionType, EncryptionPolicy>
{
private:
    typedef detail::IdcServer<ConnectionType, EncryptionPolicy> Base;

    struct Resource
    {
        detail::SaveDataExporter::ResourceStorage dataTransferWorkBuffer;
    } m_Resource;

public:
    explicit IdcServer(detail::ThreadResource&& threadResource) NN_NOEXCEPT
        : Base(std::move(threadResource))
    {
    }
    Result ExportUserMigrationServerInfo(const ServerContext& context, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result ImportUserMigrationClientInfo(ServerContext& context, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result MatchUserMigrationClientInfo(const ServerContext& context, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result AcceptUserMigrationClient(const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result DeclineUserMigrationClient(const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result WaitStateSychronizationInInitialization(StateController& stateController, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result WaitStateSychronizationInFinalization(StateController& stateController, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result WaitStateSychronizationFinalized(StateController& stateController, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result ExportMigrationList(const ServerContext& context, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result ExportSaveData(StateController& stateController, ServerContext& context, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT;
};

}}} // ~namespace nn::migration::user


#include <nn/migration/detail/migration_Diagnosis.h>
#include <nn/migration/detail/migration_Settings.h>
#include <nn/migration/user/migration_UserMigrationIdcImpl.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace migration { namespace user {

const int SuspendDelayMilliSeconds = 3 * 1000;

template <typename ConnectionType, typename EncryptionPolicy>
Result IdcServer<ConnectionType, EncryptionPolicy>::ExportUserMigrationServerInfo(const ServerContext& context, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_MIGRATION_DETAIL_TRACE("[Server] Exporting user migration server info\n");
    GetUserMigrationServerInfoServerApi api(context);
    NN_RESULT_DO(Base::WaitUserCommand(connection, api, detail::GetTransferMessagingTimeoutSeconds(), pCancellable));
    NN_RESULT_SUCCESS;
}

template <typename ConnectionType, typename EncryptionPolicy>
Result IdcServer<ConnectionType, EncryptionPolicy>::ImportUserMigrationClientInfo(ServerContext& context, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_MIGRATION_DETAIL_TRACE("[Server] Importing user migration client info\n");
    PutUserMigrationClientInfoServerApi api;
    NN_RESULT_DO(Base::WaitUserCommand(connection, api, detail::GetTransferMessagingTimeoutSeconds(), pCancellable));
    return api.ImportClientInfo(context);
}

template <typename ConnectionType, typename EncryptionPolicy>
Result IdcServer<ConnectionType, EncryptionPolicy>::MatchUserMigrationClientInfo(const ServerContext& context, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_MIGRATION_DETAIL_TRACE("[Server] Matching user migration client info\n");
    PutUserMigrationClientInfoServerApi api;
    NN_RESULT_DO(Base::WaitUserCommand(connection, api, detail::GetTransferMessagingTimeoutSeconds(), pCancellable));
    return api.MatchClientInfo(context);
}

template <typename ConnectionType, typename EncryptionPolicy>
Result IdcServer<ConnectionType, EncryptionPolicy>::AcceptUserMigrationClient(const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_MIGRATION_DETAIL_TRACE("[Server] Accepting connection from client\n");
    WaitUserMigrationStartServerApi api(true);
    NN_RESULT_DO(Base::WaitUserCommand(connection, api, detail::GetTransferMessagingTimeoutSeconds(), pCancellable));
    NN_RESULT_SUCCESS;
}

template <typename ConnectionType, typename EncryptionPolicy>
Result IdcServer<ConnectionType, EncryptionPolicy>::DeclineUserMigrationClient(const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_MIGRATION_DETAIL_TRACE("[Server] Declining connection from client\n");
    WaitUserMigrationStartServerApi api(false);
    NN_RESULT_DO(Base::WaitUserCommand(connection, api, detail::GetTransferMessagingTimeoutSeconds(), pCancellable));
    NN_RESULT_SUCCESS;
}

template <typename ConnectionType, typename EncryptionPolicy>
Result IdcServer<ConnectionType, EncryptionPolicy>::WaitStateSychronizationInInitialization(StateController& stateController, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_MIGRATION_DETAIL_TRACE("[Server] Wait state synchronization for InInitialization\n");
    SynchronizeStateServerApi api(stateController, MigrationState_InInitialization);
    NN_RESULT_DO(Base::WaitUserCommand(connection, api, detail::GetTransferMessagingTimeoutSeconds(), pCancellable));
    auto r = api.GetResult();
    if (ResultSuspendedByExternal::Includes(r))
    {
        os::SleepThread(TimeSpan::FromMilliSeconds(SuspendDelayMilliSeconds));
    }
    NN_RESULT_THROW(r);
}

template <typename ConnectionType, typename EncryptionPolicy>
Result IdcServer<ConnectionType, EncryptionPolicy>::WaitStateSychronizationInFinalization(StateController& stateController, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_MIGRATION_DETAIL_TRACE("[Server] Wait state synchronization for InFinalization\n");
    SynchronizeStateServerApi api(stateController, MigrationState_InFinalization);
    NN_RESULT_DO(Base::WaitUserCommand(connection, api, detail::GetTransferMessagingTimeoutSeconds(), pCancellable));
    auto r = api.GetResult();
    if (ResultSuspendedByExternal::Includes(r))
    {
        os::SleepThread(TimeSpan::FromMilliSeconds(SuspendDelayMilliSeconds));
    }
    NN_RESULT_THROW(r);
}

template <typename ConnectionType, typename EncryptionPolicy>
Result IdcServer<ConnectionType, EncryptionPolicy>::WaitStateSychronizationFinalized(StateController& stateController, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_MIGRATION_DETAIL_TRACE("[Server] Wait state synchronization for Finalized\n");
    SynchronizeStateServerApi api(stateController, MigrationState_Finalized);
    NN_RESULT_DO(Base::WaitUserCommand(connection, api, detail::GetTransferMessagingTimeoutSeconds(), pCancellable));
    auto r = api.GetResult();
    if (ResultSuspendedByExternal::Includes(r))
    {
        os::SleepThread(TimeSpan::FromMilliSeconds(SuspendDelayMilliSeconds));
    }
    NN_RESULT_THROW(r);
}

template <typename ConnectionType, typename EncryptionPolicy>
Result IdcServer<ConnectionType, EncryptionPolicy>::ExportMigrationList(const ServerContext& context, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_MIGRATION_DETAIL_TRACE("[Server] Exporing migration list\n");
    GetMigrationListServerApi api(context);
    NN_RESULT_DO(Base::WaitUserCommand(connection, api, detail::GetTransferSmallDataTimeoutSeconds(), pCancellable));
    NN_RESULT_SUCCESS;
}

template <typename ConnectionType, typename EncryptionPolicy>
Result IdcServer<ConnectionType, EncryptionPolicy>::ExportSaveData(StateController& stateController, ServerContext& context, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_MIGRATION_DETAIL_TRACE("[Server] Exporting save data\n");
    detail::SaveDataExporter exporter(*this, m_Resource.dataTransferWorkBuffer);
    RequestTransferServerApi<detail::SaveDataExporter> api(exporter, stateController, context);
    NN_RESULT_DO(Base::WaitUserCommand(connection, api, detail::GetTransferLargeDataTimeoutSeconds(), pCancellable));
    auto r = api.GetResult();
    if (ResultSuspendedByExternal::Includes(r))
    {
        os::SleepThread(TimeSpan::FromMilliSeconds(SuspendDelayMilliSeconds));
    }
    NN_RESULT_THROW(r);
}


}}} // ~namespace nn::migration::user
