﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/migration/detail/migration_IdcImplBase.h>
#include <nn/migration/detail/migration_Diagnosis.h>
#include <nn/migration/detail/migration_Result.h>
#include <nn/migration/user/migration_UserMigrationContext.h>
#include <nn/migration/user/migration_UserMigrationInternalTypes.h>
#include <nn/migration/user/migration_UserMigrationStateController.h>
#include <nn/migration/migration_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace migration { namespace user {

enum class UserMigrationCommandKind : Bit8
{
    GetUserMigrationServerInfo = 0x10,
    PutUserMigrationClientInfo = 0x11,
    WaitUserMigrationStart = 0x20,
    GetMigrationList = 0x21,
    SynchronizeState = 0x30,
    RequestTransfer = 0x80,
    NotifySuspend = 0xFF,
};
inline const char* GetCommandString(UserMigrationCommandKind command) NN_NOEXCEPT
{
    switch (command)
    {
    case UserMigrationCommandKind::GetUserMigrationServerInfo:
        return "UserMigrationCommandKind::GetUserMigrationServerInfo";
    case UserMigrationCommandKind::PutUserMigrationClientInfo:
        return "UserMigrationCommandKind::PutUserMigrationClientInfo";
    case UserMigrationCommandKind::WaitUserMigrationStart:
        return "UserMigrationCommandKind::WaitUserMigrationStart";
    case UserMigrationCommandKind::GetMigrationList:
        return "UserMigrationCommandKind::GetMigrationList";
    case UserMigrationCommandKind::SynchronizeState:
        return "UserMigrationCommandKind::SynchronizeState";
    case UserMigrationCommandKind::RequestTransfer:
        return "UserMigrationCommandKind::RequestTransfer";
    case UserMigrationCommandKind::NotifySuspend:
        return "UserMigrationCommandKind::NotifySuspend";
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

// ---------------------------------------------------------------------------------------------
// NotifySuspend

struct RequestNotifySuspend
{
    Bit8 commandId;
    Bit8 padding[7];

    void Reset() NN_NOEXCEPT
    {
        commandId = static_cast<Bit8>(UserMigrationCommandKind::NotifySuspend);
        std::memset(padding, 0x00, sizeof(padding));
    }
};
struct ResponseNotifySuspend
{
    Bit8 commandId;
    Bit8 padding[7];

    void Reset() NN_NOEXCEPT
    {
        commandId = static_cast<Bit8>(UserMigrationCommandKind::NotifySuspend);
        std::memset(padding, 0x00, sizeof(padding));
    }
};

class NotifySuspendClientApi final
    : public detail::ClientApiBase<UserMigrationCommandKind, NotifySuspendClientApi>
{
private:
    typedef NotifySuspendClientApi This;
    typedef detail::ClientApiBase<UserMigrationCommandKind, NotifySuspendClientApi> Base;
    typedef RequestNotifySuspend Request;
    typedef ResponseNotifySuspend Response;

    RequestNotifySuspend m_Request;
    Base::CommandHandler m_Handlers[1];

    Result Handler(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(dataSize == sizeof(Response), detail::ResultUnexpectedResponseSize());
        NN_UNUSED(data);
        NN_RESULT_SUCCESS;
    }

public:
    NotifySuspendClientApi() NN_NOEXCEPT
        : Base(*this)
    {
        m_Request.Reset();
        m_Handlers[0].command = UserMigrationCommandKind::NotifySuspend;
        m_Handlers[0].handler = &This::Handler;
        Initialize(m_Request, m_Handlers);
    }
};

class Suspendable
{
private:
    typedef Suspendable This;
    typedef RequestNotifySuspend Request;
    typedef ResponseNotifySuspend Response;

    Response m_Response;
    bool m_IsSuspended;

    template <typename T>
    Result Handler(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(dataSize == sizeof(Request), detail::ResultUnexpectedRequestSize());
        NN_UNUSED(data);

        m_IsSuspended = true;
        m_Response.Reset();
        static_cast<T*>(this)->SetStaticResponse(m_Response);
        NN_RESULT_SUCCESS;
    }

protected:
    template <typename T>
    typename detail::ServerApiBase<UserMigrationCommandKind, T>::CommandHandler GetCommandHandler() const NN_NOEXCEPT
    {
        typename detail::ServerApiBase<UserMigrationCommandKind, T>::CommandHandler handler;
        handler.command = UserMigrationCommandKind::NotifySuspend;
        handler.handler = &This::Handler<T>;
        return handler;
    }

    Suspendable() NN_NOEXCEPT
        : m_IsSuspended(false)
    {
    }
    Result GetResult() const NN_NOEXCEPT
    {
        if (m_IsSuspended)
        {
            NN_RESULT_THROW(ResultSuspendedByExternal());
        }
        NN_RESULT_SUCCESS;
    }
};

// ---------------------------------------------------------------------------------------------
// GetUserMigrationServerInfo

struct RequestGetUserMigrationServerInfo
{
    Bit8 commandId;
    Bit8 padding[7];

    void Reset() NN_NOEXCEPT
    {
        commandId = static_cast<Bit8>(UserMigrationCommandKind::GetUserMigrationServerInfo);
        std::memset(padding, 0x00, sizeof(padding));
    }
};

struct ResponseGetUserMigrationServerInfo
{
    Bit8 commandId;
    Bit8 padding[7];
    Bit8 data[sizeof(ServerInfoForTransfer)]; // TODO Endian

    void Reset(const ServerInfoForTransfer& serverInfo) NN_NOEXCEPT
    {
        commandId = static_cast<Bit8>(UserMigrationCommandKind::GetUserMigrationServerInfo);
        std::memset(padding, 0x00, sizeof(padding));
        std::memcpy(data, &serverInfo, sizeof(data));
    }
    void Get(ServerInfoForTransfer* pOut) const NN_NOEXCEPT
    {
        std::memcpy(pOut, data, sizeof(*pOut));
    }
};

class GetUserMigrationServerInfoClientApi final
    : public detail::ClientApiBase<UserMigrationCommandKind, GetUserMigrationServerInfoClientApi>
{
private:
    typedef GetUserMigrationServerInfoClientApi This;
    typedef detail::ClientApiBase<UserMigrationCommandKind, GetUserMigrationServerInfoClientApi> Base;
    typedef RequestGetUserMigrationServerInfo Request;
    typedef ResponseGetUserMigrationServerInfo Response;

    RequestGetUserMigrationServerInfo m_Request;
    util::optional<Response> m_Response;
    Base::CommandHandler m_Handlers[1];

    Result Handler(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(dataSize == sizeof(Response), detail::ResultUnexpectedResponseSize());
        m_Response.emplace();
        std::memcpy(&m_Response.value(), data, dataSize);
        NN_RESULT_SUCCESS;
    }

public:
    GetUserMigrationServerInfoClientApi() NN_NOEXCEPT
        : Base(*this)
    {
        m_Request.Reset();
        m_Handlers[0].command = UserMigrationCommandKind::GetUserMigrationServerInfo;
        m_Handlers[0].handler = &This::Handler;
        Initialize(m_Request, m_Handlers);
    }
    Result ImportServerInfo(ClientContext& context) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Response);
        ServerInfoForTransfer serverInfo;
        m_Response->Get(&serverInfo);
        return context.ImportServerInfo(serverInfo);
    }
    Result MatchServerInfo(const ClientContext& context) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Response);
        ServerInfoForTransfer serverInfo;
        m_Response->Get(&serverInfo);
        NN_RESULT_THROW_UNLESS(context.MatchServerInfo(serverInfo), detail::ResultUnexpectedMigrationServerInfo());
        NN_RESULT_SUCCESS;
    }
};

class GetUserMigrationServerInfoServerApi final
    : public detail::ServerApiBase<UserMigrationCommandKind, GetUserMigrationServerInfoServerApi>
{
private:
    typedef GetUserMigrationServerInfoServerApi This;
    typedef detail::ServerApiBase<UserMigrationCommandKind, GetUserMigrationServerInfoServerApi> Base;
    typedef RequestGetUserMigrationServerInfo Request;
    typedef ResponseGetUserMigrationServerInfo Response;

    const ServerContext& m_Context;
    util::optional<Response> m_Response;
    Base::CommandHandler m_Handlers[1];

    Result Handler(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(dataSize == sizeof(Request), detail::ResultUnexpectedRequestSize());
        NN_UNUSED(data);

        ServerInfoForTransfer serverInfo;
        m_Context.GetServerInfoForTransfer(&serverInfo);

        m_Response.emplace();
        m_Response->Reset(serverInfo);
        SetStaticResponse(m_Response.value());
        NN_RESULT_SUCCESS;
    }

public:
    explicit GetUserMigrationServerInfoServerApi(const ServerContext& context) NN_NOEXCEPT
        : Base(*this)
        , m_Context(context)
    {
        m_Handlers[0].command = UserMigrationCommandKind::GetUserMigrationServerInfo;
        m_Handlers[0].handler = &This::Handler;
        Initialize(m_Handlers);
    }
};

// ---------------------------------------------------------------------------------------------
// PutUserMigrationClientInfo

struct RequestPutUserMigrationClientInfo
{
    Bit8 commandId;
    Bit8 padding[7];
    Bit8 data[sizeof(ClientInfoForTransfer)]; // TODO Endian

    void Reset(const ClientInfoForTransfer& clientInfo) NN_NOEXCEPT
    {
        commandId = static_cast<Bit8>(UserMigrationCommandKind::PutUserMigrationClientInfo);
        std::memset(padding, 0x00, sizeof(padding));
        std::memcpy(data, &clientInfo, sizeof(data));
    }
    void Get(ClientInfoForTransfer* pOut) const NN_NOEXCEPT
    {
        std::memcpy(pOut, data, sizeof(*pOut));
    }
};

struct ResponsePutUserMigrationClientInfo
{
    Bit8 commandId;
    Bit8 padding[7];

    void Reset() NN_NOEXCEPT
    {
        commandId = static_cast<Bit8>(UserMigrationCommandKind::PutUserMigrationClientInfo);
        std::memset(padding, 0x00, sizeof(padding));
    }
};

class PutUserMigrationClientInfoClientApi final
    : public detail::ClientApiBase<UserMigrationCommandKind, PutUserMigrationClientInfoClientApi>
{
private:
    typedef PutUserMigrationClientInfoClientApi This;
    typedef detail::ClientApiBase<UserMigrationCommandKind, PutUserMigrationClientInfoClientApi> Base;
    typedef RequestPutUserMigrationClientInfo Request;
    typedef ResponsePutUserMigrationClientInfo Response;

    RequestPutUserMigrationClientInfo m_Request;
    Base::CommandHandler m_Handlers[1];

    Result Handler(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(dataSize == sizeof(Response), detail::ResultUnexpectedResponseSize());
        NN_UNUSED(data);
        NN_RESULT_SUCCESS;
    }

public:
    explicit PutUserMigrationClientInfoClientApi(const ClientContext& context) NN_NOEXCEPT
        : Base(*this)
    {
        ClientInfoForTransfer clientInfo;
        context.GetClientInfoForTransfer(&clientInfo);

        m_Request.Reset(clientInfo);
        m_Handlers[0].command = UserMigrationCommandKind::PutUserMigrationClientInfo;
        m_Handlers[0].handler = &This::Handler;
        Initialize(m_Request, m_Handlers);
    }
};

class PutUserMigrationClientInfoServerApi final
    : public detail::ServerApiBase<UserMigrationCommandKind, PutUserMigrationClientInfoServerApi>
{
private:
    typedef PutUserMigrationClientInfoServerApi This;
    typedef detail::ServerApiBase<UserMigrationCommandKind, PutUserMigrationClientInfoServerApi> Base;
    typedef RequestPutUserMigrationClientInfo Request;
    typedef ResponsePutUserMigrationClientInfo Response;

    ClientInfoForTransfer m_ClientInfo;
    util::optional<Response> m_Response;
    Base::CommandHandler m_Handlers[1];

    Result Handler(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        Request request;
        NN_RESULT_THROW_UNLESS(dataSize == sizeof(request), detail::ResultUnexpectedRequestSize());
        std::memcpy(&request, data, dataSize);
        request.Get(&m_ClientInfo);

        m_Response.emplace();
        m_Response->Reset();
        SetStaticResponse(m_Response.value());
        NN_RESULT_SUCCESS;
    }

public:
    PutUserMigrationClientInfoServerApi() NN_NOEXCEPT
        : Base(*this)
    {
        m_Handlers[0].command = UserMigrationCommandKind::PutUserMigrationClientInfo;
        m_Handlers[0].handler = &This::Handler;
        Initialize(m_Handlers);
    }
    Result ImportClientInfo(ServerContext& context) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Response);
        return context.ImportClientInfo(m_ClientInfo);
    }
    Result MatchClientInfo(const ServerContext& context) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Response);
        NN_RESULT_THROW_UNLESS(context.MatchClientInfo(m_ClientInfo), detail::ResultUnexpectedMigrationClientInfo());
        NN_RESULT_SUCCESS;
    }
};

// ---------------------------------------------------------------------------------------------
// WaitUserMigrationStart

struct RequestWaitUserMigrationStart
{
    Bit8 commandId;
    Bit8 padding[7];

    void Reset() NN_NOEXCEPT
    {
        commandId = static_cast<Bit8>(UserMigrationCommandKind::WaitUserMigrationStart);
        std::memset(padding, 0x00, sizeof(padding));
    }
};

struct ResponseWaitUserMigrationStart
{
    Bit8 commandId;
    Bit8 padding[7];
    Bit8 data[1];

    void Reset(bool isAccepted) NN_NOEXCEPT
    {
        commandId = static_cast<Bit8>(UserMigrationCommandKind::WaitUserMigrationStart);
        std::memset(padding, 0x00, sizeof(padding));
        data[0] = (isAccepted ? 0x01 : 0x00);
    }
    bool Get() const NN_NOEXCEPT
    {
        return data[0] != 0x00;
    }
};

class WaitUserMigrationStartClientApi final
    : public detail::ClientApiBase<UserMigrationCommandKind, WaitUserMigrationStartClientApi>
{
private:
    typedef WaitUserMigrationStartClientApi This;
    typedef detail::ClientApiBase<UserMigrationCommandKind, WaitUserMigrationStartClientApi> Base;
    typedef RequestWaitUserMigrationStart Request;
    typedef ResponseWaitUserMigrationStart Response;

    RequestWaitUserMigrationStart m_Request;
    util::optional<Response> m_Response;
    Base::CommandHandler m_Handlers[1];

    Result Handler(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(dataSize == sizeof(Response), detail::ResultUnexpectedResponseSize());
        m_Response.emplace();
        std::memcpy(&m_Response.value(), data, dataSize);
        NN_RESULT_SUCCESS;
    }

public:
    WaitUserMigrationStartClientApi() NN_NOEXCEPT
        : Base(*this)
    {
        m_Request.Reset();
        m_Handlers[0].command = UserMigrationCommandKind::WaitUserMigrationStart;
        m_Handlers[0].handler = &This::Handler;
        Initialize(m_Request, m_Handlers);
    }
    Result GetResult() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Response);
        NN_RESULT_THROW_UNLESS(m_Response->Get(), ResultSuspendedByExternal());
        NN_RESULT_SUCCESS;
    }
};

class WaitUserMigrationStartServerApi final
    : public detail::ServerApiBase<UserMigrationCommandKind, WaitUserMigrationStartServerApi>
{
private:
    typedef WaitUserMigrationStartServerApi This;
    typedef detail::ServerApiBase<UserMigrationCommandKind, WaitUserMigrationStartServerApi> Base;
    typedef RequestWaitUserMigrationStart Request;
    typedef ResponseWaitUserMigrationStart Response;

    bool m_IsAccepted;
    util::optional<Response> m_Response;
    Base::CommandHandler m_Handlers[1];

    Result Handler(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(dataSize == sizeof(Request), detail::ResultUnexpectedRequestSize());
        NN_UNUSED(data);

        m_Response.emplace();
        m_Response->Reset(m_IsAccepted);
        SetStaticResponse(m_Response.value());
        NN_RESULT_SUCCESS;
    }

public:
    explicit WaitUserMigrationStartServerApi(bool isAccepted) NN_NOEXCEPT
        : Base(*this)
        , m_IsAccepted(isAccepted)
    {
        m_Handlers[0].command = UserMigrationCommandKind::WaitUserMigrationStart;
        m_Handlers[0].handler = &This::Handler;
        Initialize(m_Handlers);
    }
};

// ---------------------------------------------------------------------------------------------
// SynchronizeState

struct RequestSynchronizeState
{
    Bit8 commandId;
    Bit8 padding[7];
    Bit8 data[sizeof(MigrationState)]; // TODO Endian

    void Reset(MigrationState state) NN_NOEXCEPT
    {
        commandId = static_cast<Bit8>(UserMigrationCommandKind::SynchronizeState);
        std::memset(padding, 0x00, sizeof(padding));
        std::memcpy(data, &state, sizeof(data));
    }
    bool TryGet(MigrationState* pOut) const NN_NOEXCEPT
    {
        uint32_t raw;
        std::memcpy(&raw, data, sizeof(raw));

        switch (raw)
        {
        case MigrationState_Started:
            *pOut = MigrationState_Started;
            return true;
        case MigrationState_InInitialization:
            *pOut = MigrationState_InInitialization;
            return true;
        case MigrationState_InTransfer:
            *pOut = MigrationState_InTransfer;
            return true;
        case MigrationState_InFinalization:
            *pOut = MigrationState_InFinalization;
            return true;
        case MigrationState_Finalized:
            *pOut = MigrationState_Finalized;
            return true;
        default:
            NN_MIGRATION_DETAIL_WARN("[RequestSynchronizeState] Undefined state is specified for synchronization\n");
            return false;
        }
    }
};

struct ResponseSynchronizeState
{
    Bit8 commandId;
    Bit8 padding[7];

    void Reset() NN_NOEXCEPT
    {
        commandId = static_cast<Bit8>(UserMigrationCommandKind::SynchronizeState);
        std::memset(padding, 0x00, sizeof(padding));
    }
};

class SynchronizeStateClientApi final
    : public detail::ClientApiBase<UserMigrationCommandKind, SynchronizeStateClientApi>
{
private:
    typedef SynchronizeStateClientApi This;
    typedef detail::ClientApiBase<UserMigrationCommandKind, SynchronizeStateClientApi> Base;
    typedef RequestSynchronizeState Request;
    typedef ResponseSynchronizeState Response;

    RequestSynchronizeState m_Request;
    Base::CommandHandler m_Handlers[1];

    Result Handler(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(dataSize == sizeof(Response), detail::ResultUnexpectedResponseSize());
        NN_UNUSED(data);
        NN_RESULT_SUCCESS;
    }

public:
    explicit SynchronizeStateClientApi(MigrationState state) NN_NOEXCEPT
        : Base(*this)
    {
        m_Request.Reset(state);
        m_Handlers[0].command = UserMigrationCommandKind::SynchronizeState;
        m_Handlers[0].handler = &This::Handler;
        Initialize(m_Request, m_Handlers);
    }
};

template <typename T>
class StateSynchronizable
{
private:
    typedef StateSynchronizable<T> This;
    typedef RequestSynchronizeState Request;
    typedef ResponseSynchronizeState Response;

    Response m_Response;
    Result(T::*m_Callback)(MigrationState state);

    Result Handler(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        Request request;
        NN_RESULT_THROW_UNLESS(dataSize == sizeof(request), detail::ResultUnexpectedRequestSize());
        std::memcpy(&request, data, dataSize);

        MigrationState state;
        auto isValid = request.TryGet(&state);
        NN_RESULT_THROW_UNLESS(isValid, detail::ResultUnexpectedSynchronizationRequest());

        auto& obj = *static_cast<T*>(this);
        NN_RESULT_DO((obj.*m_Callback)(state));

        m_Response.Reset();
        obj.SetStaticResponse(m_Response);
        NN_RESULT_SUCCESS;
    }

protected:
    typename detail::ServerApiBase<UserMigrationCommandKind, T>::CommandHandler GetCommandHandler() const NN_NOEXCEPT
    {
        typename detail::ServerApiBase<UserMigrationCommandKind, T>::CommandHandler handler;
        handler.command = UserMigrationCommandKind::SynchronizeState;
        handler.handler = &This::Handler;
        return handler;
    }

    StateSynchronizable(Result(T::*callback)(MigrationState state)) NN_NOEXCEPT
        : m_Callback(callback)
    {
    }
};

class SynchronizeStateServerApi final
    : public detail::ServerApiBase<UserMigrationCommandKind, SynchronizeStateServerApi>
    , Suspendable
    , StateSynchronizable<SynchronizeStateServerApi>
{
    friend class Suspendable;
    friend class StateSynchronizable<SynchronizeStateServerApi>;

private:
    typedef SynchronizeStateServerApi This;
    typedef detail::ServerApiBase<UserMigrationCommandKind, SynchronizeStateServerApi> Base;

    Base::CommandHandler m_Handlers[2];
    StateController& m_StateController;
    MigrationState const m_Expect;

    Result StateSynchronizationCallback(MigrationState state) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(state == m_Expect, detail::ResultUnexpectedSynchronizationRequest());
        switch (state)
        {
        case MigrationState_InInitialization:
            NN_RESULT_DO(m_StateController.SetStateInInitialization());
            break;
        case MigrationState_InTransfer:
            NN_RESULT_DO(m_StateController.SetStateInTransfer());
            break;
        case MigrationState_InFinalization:
            NN_RESULT_DO(m_StateController.SetStateInFinalization());
            break;
        case MigrationState_Finalized:
            NN_RESULT_DO(m_StateController.SetStateFinalized());
            break;
        case MigrationState_Started:
        default:
            NN_UNEXPECTED_DEFAULT;
        }
        NN_RESULT_SUCCESS;
    }

protected:
    using Base::SetStaticResponse;

public:
    SynchronizeStateServerApi(StateController& stateController, MigrationState expect) NN_NOEXCEPT
        : Base(*this)
        , StateSynchronizable<This>(&This::StateSynchronizationCallback)
        , m_StateController(stateController)
        , m_Expect(expect)
    {
        NN_SDK_ASSERT(false
            || m_Expect == MigrationState_InInitialization
            || m_Expect == MigrationState_InTransfer
            || m_Expect == MigrationState_InFinalization
            || m_Expect == MigrationState_Finalized);

        size_t count = 0;
        m_Handlers[count++] = this->Suspendable::GetCommandHandler<This>();
        m_Handlers[count++] = this->StateSynchronizable<This>::GetCommandHandler();
        Initialize(m_Handlers);
    }

    Result GetResult() const NN_NOEXCEPT
    {
        NN_RESULT_DO(Suspendable::GetResult());
        NN_RESULT_SUCCESS;
    }
};


// ---------------------------------------------------------------------------------------------
// GetMigrationList

struct RequestGetMigrationList
{
    Bit8 commandId;
    Bit8 padding[7];

    void Reset() NN_NOEXCEPT
    {
        commandId = static_cast<Bit8>(UserMigrationCommandKind::GetMigrationList);
        std::memset(padding, 0x00, sizeof(padding));
    }
};

struct ResponseHeaderGetMigrationList
{
    Bit8 commandId;
    Bit8 padding[7];
    Bit8 data[sizeof(detail::MigrationListHeader)]; // TODO Endian

    void Reset(const detail::MigrationListHeader& header) NN_NOEXCEPT
    {
        commandId = static_cast<Bit8>(UserMigrationCommandKind::GetMigrationList);
        std::memset(padding, 0x00, sizeof(padding));
        std::memcpy(data, &header, sizeof(data));
    }
    detail::MigrationListHeader Get() const NN_NOEXCEPT
    {
        detail::MigrationListHeader header;
        std::memcpy(&header, data, sizeof(header));
        return header;
    }
};

class GetMigrationListClientApi final
    : public detail::ClientApiBase<UserMigrationCommandKind, GetMigrationListClientApi>
{
private:
    typedef GetMigrationListClientApi This;
    typedef detail::ClientApiBase<UserMigrationCommandKind, GetMigrationListClientApi> Base;
    typedef RequestGetMigrationList Request;
    typedef ResponseHeaderGetMigrationList Response;

    RequestGetMigrationList m_Request;
    util::optional<Response> m_ResponseHeader;
    Base::CommandHandler m_Handlers[1];

    ClientContext& m_Context;
    size_t m_TotalListSize;
    size_t m_ImportedListSize;

    Result InitializeImport(const Response& response) NN_NOEXCEPT
    {
        auto listHeader = response.Get();

        m_TotalListSize = detail::MigrationList::GetMigrationListSize(listHeader);
        NN_RESULT_DO(m_Context.BeginImportMigrationList(listHeader));
        m_ImportedListSize = sizeof(listHeader);

        NN_MIGRATION_DETAIL_TRACE("[GetMigrationListClientApi] Receiving migration list: %zu of %zu\n", m_ImportedListSize, m_TotalListSize);
        NN_RESULT_SUCCESS;
    }
    Result UpdateImport(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_ResponseHeader);
        NN_RESULT_DO(m_Context.UpdateImportMigrationList(data, dataSize));
        m_ImportedListSize += dataSize;

        NN_MIGRATION_DETAIL_TRACE("[GetMigrationListClientApi] Receiving migration list: %zu of %zu\n", m_ImportedListSize, m_TotalListSize);
        NN_RESULT_SUCCESS;
    }
    bool IsRemaining() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_ResponseHeader);
        return m_ImportedListSize < m_TotalListSize;
    }

    Result Handler(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        if (!m_ResponseHeader)
        {
            NN_RESULT_THROW_UNLESS(dataSize >= sizeof(*m_ResponseHeader), detail::ResultUnexpectedResponseSize());
            m_ResponseHeader.emplace();
            std::memcpy(&m_ResponseHeader.value(), data, sizeof(*m_ResponseHeader));

            NN_RESULT_DO(InitializeImport(*m_ResponseHeader));

            auto dataSize1 = dataSize - sizeof(*m_ResponseHeader);
            if (dataSize1 > 0u)
            {
                auto data1 = reinterpret_cast<void*>(reinterpret_cast<uintptr_t>(data) + sizeof(*m_ResponseHeader));
                NN_RESULT_DO(UpdateImport(data1, dataSize1));
            }

            if (IsRemaining())
            {
                NN_RESULT_THROW(idc::ResultConsumeCommandContinue());
            }
            NN_RESULT_SUCCESS;
        }

        NN_RESULT_DO(UpdateImport(data, dataSize));
        if (IsRemaining())
        {
            NN_RESULT_THROW(idc::ResultConsumeCommandContinue());
        }
        NN_RESULT_SUCCESS;
    }

public:
    explicit GetMigrationListClientApi(ClientContext& context) NN_NOEXCEPT
        : Base(*this)
        , m_Context(context)
    {
        m_Request.Reset();
        m_Handlers[0].command = UserMigrationCommandKind::GetMigrationList;
        m_Handlers[0].handler = &This::Handler;
        Initialize(m_Request, m_Handlers);
    }
    Result Adapt() NN_NOEXCEPT
    {
        return m_Context.EndImportMigrationList();
    }
};

class GetMigrationListServerApi final
    : public detail::StreamingServerApiBase<UserMigrationCommandKind, GetMigrationListServerApi>
{
private:
    typedef GetMigrationListServerApi This;
    typedef detail::StreamingServerApiBase<UserMigrationCommandKind, GetMigrationListServerApi> Base;
    typedef RequestGetMigrationList Request;
    typedef ResponseHeaderGetMigrationList Response;

    util::optional<Response> m_Response;
    Base::CommandHandler m_Handlers[1];

    bool m_IsHeaderReplied;
    const ServerContext& m_Context;
    size_t m_TotalListSize;
    size_t m_ExportedListSize;

    void GetResponseHeader(Response* pOut) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!m_IsHeaderReplied);
        detail::MigrationListHeader header;
        auto sz = m_Context.ReadMigrationListForExport(&header, sizeof(header), 0u);
        NN_SDK_ASSERT(sz == sizeof(header));
        NN_UNUSED(sz);
        m_ExportedListSize = sizeof(header);

        pOut->Reset(header);
        NN_MIGRATION_DETAIL_TRACE("[GetMigrationListServerApi] Sending migration list: %zu of %zu\n", m_ExportedListSize, m_TotalListSize);
    }
    size_t UpdateExport(void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsHeaderReplied);
        NN_SDK_ASSERT(IsRemaining());

        auto sz = m_Context.ReadMigrationListForExport(buffer, bufferSize, m_ExportedListSize);
        NN_SDK_ASSERT(sz > 0);
        m_ExportedListSize += sz;

        NN_MIGRATION_DETAIL_TRACE("[GetMigrationListServerApi] Sending migration list: %zu of %zu\n", m_ExportedListSize, m_TotalListSize);
        return sz;
    }
    bool IsRemaining() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsHeaderReplied);
        return m_ExportedListSize < m_TotalListSize;
    }

    Result StreamOut(size_t* pOutActual, void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        if (!m_IsHeaderReplied)
        {
            NN_RESULT_THROW_UNLESS(bufferSize >= sizeof(Response), detail::ResultUnexpectedResponseSize());

            Response responseHeader;
            GetResponseHeader(&responseHeader);

            *pOutActual = sizeof(responseHeader);
            std::memcpy(buffer, &responseHeader, sizeof(responseHeader));

            m_IsHeaderReplied = true;
            if (IsRemaining())
            {
                NN_RESULT_THROW(idc::ResultProduceCommandContinue());
            }
            NN_RESULT_SUCCESS;
        }

        *pOutActual = UpdateExport(buffer, bufferSize);
        if (IsRemaining())
        {
            NN_RESULT_THROW(idc::ResultProduceCommandContinue());
        }
        NN_RESULT_SUCCESS;
    }

    Result Handler(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(dataSize == sizeof(Request), detail::ResultUnexpectedRequestSize());
        NN_UNUSED(data);

        static const auto LeadingBytes = sizeof(Response) - sizeof(detail::MigrationListHeader);
        SetStreamResponse(LeadingBytes + m_Context.GetMigrationListSize(), &This::StreamOut);
        m_TotalListSize = m_Context.GetMigrationListSize();
        NN_RESULT_SUCCESS;
    }

public:
    explicit GetMigrationListServerApi(const ServerContext& context) NN_NOEXCEPT
        : Base(*this)
        , m_IsHeaderReplied(false)
        , m_Context(context)
    {
        m_Handlers[0].command = UserMigrationCommandKind::GetMigrationList;
        m_Handlers[0].handler = &This::Handler;
        Initialize(m_Handlers);
    }
};

// ---------------------------------------------------------------------------------------------
// RequestTransfer

struct RequestRequestTransfer
{
    Bit8 commandId;
    Bit8 padding[7];
    uint32_t index; // TODO Endian

    void Reset(uint32_t _index) NN_NOEXCEPT
    {
        commandId = static_cast<Bit8>(UserMigrationCommandKind::RequestTransfer);
        std::memset(padding, 0x00, sizeof(padding));
        index = _index;
    }
    uint32_t Get() const NN_NOEXCEPT
    {
        return index;
    }
};

struct ResponseHeaderRequestTransfer
{
    Bit8 commandId;
    Bit8 padding[7];
    uint64_t saveDataId; // TODO Endian
    int64_t codedSize; // TODO Endian

    void Reset(const detail::DataInfo& dataInfo) NN_NOEXCEPT
    {
        commandId = static_cast<Bit8>(UserMigrationCommandKind::RequestTransfer);
        std::memset(padding, 0x00, sizeof(padding));
        saveDataId = dataInfo.id;
        codedSize = dataInfo.codedSize;
    }
    fs::SaveDataId GetSaveDataId() const NN_NOEXCEPT
    {
        return saveDataId;
    }
    size_t GetCodedSize() const NN_NOEXCEPT
    {
        return static_cast<size_t>(codedSize);
    }
};

template <typename Importer>
class RequestTransferClientApi final
    : public detail::ClientApiBase<UserMigrationCommandKind, RequestTransferClientApi<Importer>>
{
private:
    typedef RequestTransferClientApi<Importer> This;
    typedef detail::ClientApiBase<UserMigrationCommandKind, RequestTransferClientApi<Importer>> Base;
    typedef RequestRequestTransfer Request;
    typedef ResponseHeaderRequestTransfer Response;

    Request m_Request;
    util::optional<Response> m_ResponseHeader;
    typename Base::CommandHandler m_Handlers[1];

    Importer& m_Importer;

    Result InitializeImport(const Response& response) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(response.GetSaveDataId() == m_Importer.GetSaveDataId(), detail::ResultUnexpectedDataInfoResponded());
        NN_RESULT_THROW_UNLESS(response.GetCodedSize() == m_Importer.GetRemainingSize(), detail::ResultUnexpectedDataInfoResponded());
        NN_RESULT_SUCCESS;
    }
    Result UpdateImport(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        return m_Importer.Update(data, dataSize);
    }
    bool IsRemaining() const NN_NOEXCEPT
    {
        return m_Importer.GetRemainingSize() > 0u;
    }

    Result Handler(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        if (!m_ResponseHeader)
        {
            NN_RESULT_THROW_UNLESS(dataSize >= sizeof(*m_ResponseHeader), detail::ResultUnexpectedResponseSize());
            m_ResponseHeader.emplace();
            std::memcpy(&m_ResponseHeader.value(), data, sizeof(*m_ResponseHeader));

            NN_RESULT_DO(InitializeImport(*m_ResponseHeader));

            auto dataSize1 = dataSize - sizeof(*m_ResponseHeader);
            if (dataSize1 > 0u)
            {
                auto data1 = reinterpret_cast<void*>(reinterpret_cast<uintptr_t>(data) + sizeof(*m_ResponseHeader));
                NN_RESULT_DO(UpdateImport(data1, dataSize1));
            }

            if (IsRemaining())
            {
                NN_RESULT_THROW(idc::ResultConsumeCommandContinue());
            }
            NN_RESULT_SUCCESS;
        }

        NN_RESULT_DO(UpdateImport(data, dataSize));
        if (IsRemaining())
        {
            NN_RESULT_THROW(idc::ResultConsumeCommandContinue());
        }
        NN_RESULT_SUCCESS;
    }

public:
    explicit RequestTransferClientApi(Importer& importer, int index) NN_NOEXCEPT
        : Base(*this)
        , m_Importer(importer)
    {
        m_Request.Reset(index);
        m_Handlers[0].command = UserMigrationCommandKind::RequestTransfer;
        m_Handlers[0].handler = &This::Handler;
        Base::Initialize(m_Request, m_Handlers);
    }
};

template <typename Exporter>
class RequestTransferServerApi final
    : public detail::StreamingServerApiBase<UserMigrationCommandKind, RequestTransferServerApi<Exporter>>
    , Suspendable
    , StateSynchronizable<RequestTransferServerApi<Exporter>>
{
    friend class Suspendable;
    friend class StateSynchronizable<RequestTransferServerApi<Exporter>>;

private:
    typedef RequestTransferServerApi<Exporter> This;
    typedef detail::StreamingServerApiBase<UserMigrationCommandKind, RequestTransferServerApi<Exporter>> Base;
    typedef RequestRequestTransfer Request;
    typedef ResponseHeaderRequestTransfer Response;

    util::optional<Response> m_Response;
    typename Base::CommandHandler m_Handlers[3];
    bool m_IsHeaderReplied;

    StateController& m_StateController;
    const ServerContext& m_Context;
    Exporter& m_Exporter;
    util::optional<detail::DataInfo> m_DataInfo;

    void GetResponseHeader(Response* pOut) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_DataInfo);
        pOut->Reset(*m_DataInfo);
    }
    size_t UpdateExport(void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_DataInfo);
        return m_Exporter.Export(buffer, bufferSize);
    }
    bool IsRemaining() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_DataInfo);
        return m_Exporter.GetRemainingSize() > 0u;
    }

    Result StreamOut(size_t* pOutActual, void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        if (!m_IsHeaderReplied)
        {
            NN_RESULT_THROW_UNLESS(bufferSize >= sizeof(Response), detail::ResultUnexpectedResponseSize());

            Response responseHeader;
            GetResponseHeader(&responseHeader);

            *pOutActual = sizeof(responseHeader);
            std::memcpy(buffer, &responseHeader, sizeof(responseHeader));

            m_IsHeaderReplied = true;
            if (IsRemaining())
            {
                NN_RESULT_THROW(idc::ResultProduceCommandContinue());
            }
            NN_RESULT_SUCCESS;
        }

        *pOutActual = UpdateExport(buffer, bufferSize);
        if (IsRemaining())
        {
            NN_RESULT_THROW(idc::ResultProduceCommandContinue());
        }
        NN_RESULT_SUCCESS;
    }

    Result Handler(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        Request request;
        NN_RESULT_THROW_UNLESS(dataSize == sizeof(request), detail::ResultUnexpectedRequestSize());
        std::memcpy(&request, data, dataSize);

        auto index = request.Get();
        detail::DataInfo dataInfo;
        auto readCount = m_Context.ReadMigrationList(&dataInfo, 1, index);
        NN_RESULT_THROW_UNLESS(readCount > 0, detail::ResultUnexpectedTransferRequest());
        NN_SDK_ASSERT_EQUAL(readCount, 1u);

        NN_RESULT_DO(m_Exporter.Setup(dataInfo));
        static const auto LeadingBytes = sizeof(Response);
        Base::SetStreamResponse(LeadingBytes + m_Exporter.GetRemainingSize(), &This::StreamOut);
        m_DataInfo = dataInfo;
        NN_RESULT_SUCCESS;
    }

    Result StateSynchronizationCallback(MigrationState state) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(state == MigrationState_InFinalization, detail::ResultUnexpectedSynchronizationRequest());
        if (!m_StateController.IsTransferDone())
        {
            NN_RESULT_DO(m_StateController.SetStateInFinalization());
        }
        NN_RESULT_SUCCESS;
    }

public:
    explicit RequestTransferServerApi(Exporter& exporter, StateController& stateController, const ServerContext& context) NN_NOEXCEPT
        : Base(*this)
        , StateSynchronizable<This>(&This::StateSynchronizationCallback)
        , m_IsHeaderReplied(false)
        , m_StateController(stateController)
        , m_Context(context)
        , m_Exporter(exporter)
    {
        size_t count = 0;
        m_Handlers[count++] = this->Suspendable::GetCommandHandler<This>();
        m_Handlers[count++] = this->StateSynchronizable<This>::GetCommandHandler();
        if (!m_StateController.IsTransferDone())
        {
            m_Handlers[count].command = UserMigrationCommandKind::RequestTransfer;
            m_Handlers[count].handler = &This::Handler;
            ++count;
        }
        Base::Initialize(m_Handlers, count);
    }

    Result GetResult() const NN_NOEXCEPT
    {
        NN_RESULT_DO(Suspendable::GetResult());
        NN_RESULT_SUCCESS;
    }
};

}}} // ~namespace nn::migration::user
