﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/migration/detail/migration_Diagnosis.h>
#include <nn/migration/user/migration_UserMigrationProxyBase.h>
#include <nn/util/util_UuidTypes.h>

namespace nn { namespace migration { namespace user {

template <typename ObjectAllocator, typename Impl>
class ClientProxy
    : public ProxyBase<ObjectAllocator, Impl>
{
public:
    typedef ProxyBase<ObjectAllocator, Impl> ProxyBaseType;
    typedef sf::SharedPointer<ClientProxy<ObjectAllocator, Impl>> CurrentPtr;

    using ProxyBaseType::RequiredWorkMemorySize;

private:
    // 非同期処理のアダプタ
    template <Result(Impl::*ExecuteImpl)(const detail::Cancellable*)>
    struct Task0
    {
        CurrentPtr m_pParent;
        explicit Task0(CurrentPtr&& parent) NN_NOEXCEPT
            : m_pParent(std::move(parent))
        {
        }
        Result Execute(const detail::Cancellable* pCancellable) NN_NOEXCEPT
        {
            return (m_pParent->GetImplRef().*ExecuteImpl)(pCancellable);
        }
    };
    template <typename T0, Result(Impl::*ExecuteImpl)(T0, const detail::Cancellable*)>
    struct Task1
    {
        CurrentPtr m_pParent;
        typename std::remove_reference<T0>::type m_T0;
        Task1(CurrentPtr&& parent, const T0& t0) NN_NOEXCEPT
            : m_pParent(std::move(parent))
            , m_T0(t0)
        {
        }
        Result Execute(const detail::Cancellable* pCancellable) NN_NOEXCEPT
        {
            return (m_pParent->GetImplRef().*ExecuteImpl)(m_T0, pCancellable);
        }
    };

public:
    template <typename... Args>
    explicit ClientProxy(Args&&... args) NN_NOEXCEPT
        : ProxyBaseType(std::forward<Args>(args)...)
    {
        NN_MIGRATION_DETAIL_TRACE("[Client] Object size: %zu (Impl size:  %zu)\n", sizeof(*this), sizeof(Impl));
    }

    Result Initialize(const UserMigrationClientProfile& profile) NN_NOEXCEPT
    {
        return this->GetImplRef().Initialize(profile);
    }
    Result Resume() NN_NOEXCEPT
    {
        return this->GetImplRef().Resume();
    }

    Result GetClientProfile(sf::Out<UserMigrationClientProfile> pOut) const NN_NOEXCEPT
    {
        this->GetImplRef().GetClientProfile(pOut.GetPointer());
        NN_RESULT_SUCCESS;
    }

    Result CreateLoginSession(sf::Out<account::SessionId> pOut) const NN_NOEXCEPT
    {
        *pOut = this->GetImplRef().CreateLoginSession();
        NN_RESULT_SUCCESS;
    }

    Result GetNetworkServiceAccountId(sf::Out<account::NetworkServiceAccountId> pOut) const NN_NOEXCEPT
    {
        return this->GetImplRef().GetNetworkServiceAccountId(pOut.GetPointer());
    }

    Result GetUserNickname(sf::Out<account::Nickname> pOut) const NN_NOEXCEPT
    {
        return this->GetImplRef().GetUserNickname(pOut.GetPointer());
    }

    Result GetUserProfileImage(sf::Out<uint32_t> pOutActualSize, sf::OutBuffer buffer) const NN_NOEXCEPT
    {
        size_t actualSize;
        NN_RESULT_DO(this->GetImplRef().GetUserProfileImage(&actualSize, buffer.GetPointerUnsafe(), buffer.GetSize()));
        NN_SDK_ASSERT(actualSize <= std::numeric_limits<uint32_t>::max());
        *pOutActualSize = static_cast<uint32_t>(actualSize);
        NN_RESULT_SUCCESS;
    }

    Result PrepareAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOut) NN_NOEXCEPT
    {
        auto p = this->template CreateAsyncContext<Task0<&Impl::Prepare>>(CurrentPtr(this, true));
        NN_RESULT_THROW_UNLESS(p, detail::ResultOutOfSessionObject());
        *pOut = p;
        NN_RESULT_SUCCESS;
    }

    Result GetConnectionRequirement(sf::Out<bool> pOut) const NN_NOEXCEPT
    {
        *pOut = this->GetImplRef().IsConnectionRequired();
        NN_RESULT_SUCCESS;
    }

    Result ScanServersAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOut) NN_NOEXCEPT
    {
        auto p = this->template CreateAsyncContext<Task0<&Impl::ScanServers>>(CurrentPtr(this, true));
        NN_RESULT_THROW_UNLESS(p, detail::ResultOutOfSessionObject());
        *pOut = p;
        NN_RESULT_SUCCESS;
    }

    Result ListServers(sf::Out<uint32_t> pOutCount, sf::OutArray<UserMigrationServerInfo> servers) const NN_NOEXCEPT
    {
        *pOutCount = static_cast<uint32_t>(this->GetImplRef().ListServers(servers.GetData(), servers.GetLength()));
        NN_RESULT_SUCCESS;
    }

    Result ConnectByServerIdAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOut, const util::Uuid& serverId) NN_NOEXCEPT
    {
        auto p = this->template CreateAsyncContext<Task1<const util::Uuid&, &Impl::Connect>>(CurrentPtr(this, true), serverId);
        NN_RESULT_THROW_UNLESS(p, detail::ResultOutOfSessionObject());
        *pOut = p;
        NN_RESULT_SUCCESS;
    }

    Result GetImmigrantUid(sf::Out<account::Uid> pOut) const NN_NOEXCEPT
    {
        *pOut = this->GetImplRef().GetImmigrantUid();
        NN_RESULT_SUCCESS;
    }
    Result GetStorageShortfall(sf::Out<int64_t> pOut) const NN_NOEXCEPT
    {
        *pOut = this->GetImplRef().GetStorageShortfall();
        NN_RESULT_SUCCESS;
    }
    Result GetTotalTransferInfo(sf::Out<TransferInfo> pOut) const NN_NOEXCEPT
    {
        *pOut = this->GetImplRef().GetTotalTransferInfo();
        NN_RESULT_SUCCESS;
    }
    Result GetCurrentTransferInfo(sf::Out<TransferInfo> pOut) const NN_NOEXCEPT
    {
        *pOut = this->GetImplRef().GetCurrentTransferInfo();
        NN_RESULT_SUCCESS;
    }
    Result GetCurrentRelatedApplications(sf::Out<uint32_t> pOut, sf::OutArray<ncm::ApplicationId> apps) NN_NOEXCEPT
    {
        *pOut = static_cast<uint32_t>(this->GetImplRef().GetCurrentRelatedApplications(apps.GetData(), apps.GetLength()));
        NN_RESULT_SUCCESS;
    }

    Result TransferNextAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOut) NN_NOEXCEPT
    {
        auto p = this->template CreateAsyncContext<Task0<&Impl::TransferNext>>(CurrentPtr(this, true));
        NN_RESULT_THROW_UNLESS(p, detail::ResultOutOfSessionObject());
        *pOut = p;
        NN_RESULT_SUCCESS;
    }

    Result SuspendAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOut) NN_NOEXCEPT
    {
        auto p = this->template CreateAsyncContext<Task0<&Impl::Suspend>>(CurrentPtr(this, true));
        NN_RESULT_THROW_UNLESS(p, detail::ResultOutOfSessionObject());
        *pOut = p;
        NN_RESULT_SUCCESS;
    }

    Result CompleteAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOut) NN_NOEXCEPT
    {
        auto p = this->template CreateAsyncContext<Task0<&Impl::Complete>>(CurrentPtr(this, true));
        NN_RESULT_THROW_UNLESS(p, detail::ResultOutOfSessionObject());
        *pOut = p;
        NN_RESULT_SUCCESS;
    }

    Result Abort() NN_NOEXCEPT
    {
        return this->GetImplRef().Abort();
    }

    // ---------------------------------------------------------------------------------------------
    // デバッグ用の関数

    Result DebugSynchronizeStateInFinalizationAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOut) NN_NOEXCEPT
    {
        auto p = this->template CreateAsyncContext<Task0<&Impl::DebugSynchronizeStateInFinalization>>(CurrentPtr(this, true));
        NN_RESULT_THROW_UNLESS(p, detail::ResultOutOfSessionObject());
        *pOut = p;
        NN_RESULT_SUCCESS;
    }
};

}}} // ~namesapce nn::migration::user
