﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <mutex>

#include <nn/nn_Result.h>
#include <nn/account/account_Types.h>
#include <nn/account/account_TypesForSystemServices.h>
#include <nn/migration/migration_UserMigrationTypes.h>
#include <nn/migration/detail/migration_Cancellable.h>
#include <nn/migration/detail/migration_LoginSession.h>
#include <nn/migration/detail/migration_NetworkConnection.h>
#include <nn/migration/user/migration_UserMigrationConfig.h>
#include <nn/migration/user/migration_UserMigrationContext.h>
#include <nn/migration/user/migration_UserMigrationIdcClient.h>
#include <nn/migration/user/migration_UserMigrationProgressMonitor.h>
#include <nn/migration/user/migration_UserMigrationStateController.h>
#include <nn/os/os_Mutex.h>

namespace nn { namespace migration { namespace user {

template <typename ContextStoragePolicy, typename ConnectionPolicy, typename EncryptionPolicy, typename SubsystemInitializer>
class Client
{
private:
    static const int ResetDelayMilliSeconds = 3 * 1000;

    // アカウントへのログインセッション
    detail::UniqueResource<detail::RemoteUserLoginSession> m_LoginSession;

    // サブシステムの初期化
    SubsystemInitializer m_SubsystemInitializer;

    // ローカル通信の接続管理
    os::Mutex m_ConnectionLock;
    std::atomic<bool> m_IsConnected;
    typename ConnectionPolicy::Connection m_Connection;
    typename ConnectionPolicy::ConnectionCreator m_ConnectionCreator;

    // 対向デバイスとの通信モジュール
    IdcClient<typename ConnectionPolicy::Connection, EncryptionPolicy> m_Idc;

    // 移行の状態管理
    typename ContextStoragePolicy::Storage m_ContextStorage;
    ClientContext m_Context;
    ProgressMonitor m_ProgressMonitor;
    StateController& m_StateController;

    // 移行先候補
    struct ScanResult
    {
        static const size_t ServerInfoCountMax = 4;

        UserMigrationServerInfo servers[ServerInfoCountMax];
        size_t count;
    } m_ScanResult;

    // 動作に必要なメモリ
    struct Resource
    {
        std::aligned_storage<
            sizeof(ncm::ApplicationId[UserMigrationRelatedApplicationCountMax]),
            std::alignment_of<ncm::ApplicationId[UserMigrationRelatedApplicationCountMax]>::value>::type relatedAppStorage;

        union
        {
            mutable std::aligned_storage<1024 * 1024>::type contextWorkBuffer; // 1MiB
        } u;
    } m_Resource;
    NN_STATIC_ASSERT(sizeof(Resource) <= RequiredWorkBufferSizeForUserMigrationOperation);

private:
    void Connect(typename ConnectionPolicy::Connection&& connection) NN_NOEXCEPT;
    void Disconnect(TimeSpan resetDelay) NN_NOEXCEPT;
    void CleanupContext() NN_NOEXCEPT;

public:
    typedef typename std::aligned_storage<sizeof(Resource), std::alignment_of<Resource>::value>::type ResourceStorage;

public:
    Client(StateController& stateController, detail::UniqueResource<detail::RemoteUserLoginSession>&& loginSession, detail::ThreadResource&& idcThreadResource) NN_NOEXCEPT;

    Result Initialize(const UserMigrationClientProfile& profile) NN_NOEXCEPT;
    Result Resume() NN_NOEXCEPT;

    Result EnsureLastMigration() NN_NOEXCEPT;

    void GetClientProfile(UserMigrationClientProfile* pOut) const NN_NOEXCEPT;

    account::SessionId CreateLoginSession() const NN_NOEXCEPT;
    Result GetNetworkServiceAccountId(account::NetworkServiceAccountId* pOut) const NN_NOEXCEPT;
    Result GetUserNickname(account::Nickname* pOut) const NN_NOEXCEPT;
    Result GetUserProfileImage(size_t* pOut, void* buffer, size_t bufferSize) const NN_NOEXCEPT;

    Result Prepare(const detail::Cancellable* pCancellable) NN_NOEXCEPT;

    bool IsConnectionRequired() const NN_NOEXCEPT;
    Result ScanServers(const detail::Cancellable* pCancellable) NN_NOEXCEPT;
    size_t ListServers(UserMigrationServerInfo* pOut, size_t count) const NN_NOEXCEPT;
    Result Connect(const util::Uuid& serverId, const detail::Cancellable* pCancellable) NN_NOEXCEPT;

    account::Uid GetImmigrantUid() const NN_NOEXCEPT;
    int64_t GetStorageShortfall() const NN_NOEXCEPT;
    TransferInfo GetTotalTransferInfo() const NN_NOEXCEPT;
    TransferInfo GetCurrentTransferInfo() const NN_NOEXCEPT;
    size_t GetCurrentRelatedApplications(ncm::ApplicationId* apps, size_t count) const NN_NOEXCEPT;
    Result TransferNext(const detail::Cancellable* pCancellable) NN_NOEXCEPT;

    Result Suspend(const detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result Complete(const detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result Abort() NN_NOEXCEPT;

    // デバッグ用の関数
    Result DebugSynchronizeStateInFinalization(const detail::Cancellable* pCancellable) NN_NOEXCEPT;
};

}}} // ~namesapce nn::migration::user

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/migration/migration_Result.h>
#include <nn/migration/detail/migration_Diagnosis.h>
#include <nn/migration/detail/migration_Result.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_UserResourceManagementApi.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace migration { namespace user {

template <typename ContextStoragePolicy, typename ConnectionPolicy, typename EncryptionPolicy, typename SubsystemInitializer>
inline Client<ContextStoragePolicy, ConnectionPolicy, EncryptionPolicy, SubsystemInitializer>::Client(StateController& stateController, detail::UniqueResource<detail::RemoteUserLoginSession>&& loginSession, detail::ThreadResource&& idcThreadResource) NN_NOEXCEPT
    : m_LoginSession(std::move(loginSession))
    , m_ConnectionLock(false)
    , m_IsConnected(false)
    , m_Idc(std::move(idcThreadResource))
    , m_Context(m_ContextStorage)
    , m_ProgressMonitor(m_Context)
    , m_StateController(stateController)
{
    m_ScanResult.count = 0;
}

#define NN_MIGRATION_USER_DEFINE_CLIENT_METHOD(rtype, name) \
template <typename ContextStoragePolicy, typename ConnectionPolicy, typename EncryptionPolicy, typename SubsystemInitializer> \
inline rtype Client<ContextStoragePolicy, ConnectionPolicy, EncryptionPolicy, SubsystemInitializer>::name NN_NOEXCEPT

NN_MIGRATION_USER_DEFINE_CLIENT_METHOD(
    Result,
    Initialize(const UserMigrationClientProfile& profile))
{
    NN_RESULT_THROW_UNLESS(!m_StateController.IsResumable(), detail::ResultResumableMigrationClientExists());

    m_ContextStorage.Cleanup();
    m_Context.Initialize(&m_Resource.relatedAppStorage, sizeof(m_Resource.relatedAppStorage));
    m_Context.CreateMigrationInfo(profile);

    NN_MIGRATION_DETAIL_TRACE("[Client] Initialized\n");
    NN_RESULT_SUCCESS;
}
NN_MIGRATION_USER_DEFINE_CLIENT_METHOD(
    Result,
    Resume())
{
    m_Context.Initialize(&m_Resource.relatedAppStorage, sizeof(m_Resource.relatedAppStorage));
    if (!m_Context.TryResume())
    {
        // コンテキストの破損やシステムバージョンの不一致を検知した場合
        NN_MIGRATION_DETAIL_THROW(detail::ResultUnresumableMigrationClient());
    }
    NN_SDK_ASSERT(m_StateController.IsResumable());

    // 前回の中断状態に関して整合性を保証する
    NN_MIGRATION_DETAIL_DO(EnsureLastMigration());

    NN_MIGRATION_DETAIL_TRACE("[Client] Resumed for NSA: id=%016llx\n", m_Context.GetNetworkServiceAccountId().id);
    NN_RESULT_SUCCESS;
}

NN_MIGRATION_USER_DEFINE_CLIENT_METHOD(
    Result,
    EnsureLastMigration())
{
    // データ転送中に中断された場合、転送していたデータ実体が存在しつつも進捗に反映されていない場合がある。
    // この場合には、当該セーブデータに関して進捗に反映したうえで再開処理を行う。
    NN_MIGRATION_DETAIL_DO(m_Context.AdjustCurrentTransferInfo());
#if !defined(NN_SDK_BUILD_RELEASE)
    {
        auto total = m_Context.GetTotalTransferInfo();
        auto current = m_Context.GetCurrentTransferInfo();
        NN_MIGRATION_DETAIL_TRACE("[Client] Transfer info recovered: %u of %u (%llu of %llu bytes)\n",
            current.count, total.count, current.sizeInBytes, total.sizeInBytes);
    }
#endif

    // Finalize 状態への遷移後 (UA 確定後) に再開し、 NSA が登録されている場合
    //  - ゲストログインを再度行うために、本体から NSA を削除する必要がある
    //  - friends やその他のデータは削除しない
    //    - 後々同じ NSA が連携されるはずなので、データ損失を防ぐために削除しないほうが良い
    if (m_StateController.IsFinalized())
    {
        account::NetworkServiceAccountAdministrator admin;
        NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(account::GetNetworkServiceAccountAdministrator(&admin, m_Context.GetUid()));
        bool registered;
        NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(admin.IsNetworkServiceAccountRegistered(&registered));
        if (registered)
        {
            // この本体での当該 NSA に関する記録は存在しないはずなので、強制削除で OK
            NN_MIGRATION_DETAIL_DO(admin.DeleteRegistrationInfoLocally());
            NN_MIGRATION_DETAIL_INFO("[Client] Local NSA registration info is deleted\n");
        }
    }

    NN_RESULT_SUCCESS;
}

NN_MIGRATION_USER_DEFINE_CLIENT_METHOD(
    void,
    Connect(typename ConnectionPolicy::Connection&& connection))
{
    std::lock_guard<os::Mutex> lock(m_ConnectionLock);
    NN_SDK_ASSERT(!m_IsConnected);

    m_Connection = std::move(connection);
    m_IsConnected = true;
    NN_MIGRATION_DETAIL_TRACE("[Client] Connected\n");
}
NN_MIGRATION_USER_DEFINE_CLIENT_METHOD(
    void,
    Disconnect(TimeSpan resetDelay))
{
    std::lock_guard<os::Mutex> lock(m_ConnectionLock);
    NN_SDK_ASSERT(m_IsConnected);

    m_Connection = decltype(m_Connection)();
    os::SleepThread(resetDelay);

    m_ConnectionCreator.Reset();
    m_IsConnected = false;
    NN_MIGRATION_DETAIL_TRACE("[Client] Disconnected\n");
}
NN_MIGRATION_USER_DEFINE_CLIENT_METHOD(
    void,
    CleanupContext())
{
    m_StateController.Reset();
    m_ContextStorage.Cleanup();

    NN_MIGRATION_DETAIL_TRACE("[Client] Context removed\n");
}

NN_MIGRATION_USER_DEFINE_CLIENT_METHOD(
    void,
    GetClientProfile(UserMigrationClientProfile* pOut) const)
{
    m_Context.GetClientProfile(pOut);
}
NN_MIGRATION_USER_DEFINE_CLIENT_METHOD(
    account::SessionId,
    CreateLoginSession() const)
{
    return m_LoginSession.Get().Create();
}
NN_MIGRATION_USER_DEFINE_CLIENT_METHOD(
    Result,
    GetNetworkServiceAccountId(account::NetworkServiceAccountId* pOut) const)
{
    return m_LoginSession.Get().GetNetworkServiceAccountId(pOut);
}
NN_MIGRATION_USER_DEFINE_CLIENT_METHOD(
    Result,
    GetUserNickname(account::Nickname* pOut) const)
{
    return m_LoginSession.Get().GetNickname(pOut);
}
NN_MIGRATION_USER_DEFINE_CLIENT_METHOD(
    Result,
    GetUserProfileImage(size_t* pOut, void* buffer, size_t bufferSize) const)
{
    return m_LoginSession.Get().GetProfileImage(pOut, buffer, bufferSize);
}
NN_MIGRATION_USER_DEFINE_CLIENT_METHOD(
    Result,
    Prepare(const detail::Cancellable* pCancellable))
{
    account::NetworkServiceAccountId nsaId;
    NN_MIGRATION_DETAIL_DO(m_LoginSession.Get().GetNetworkServiceAccountId(&nsaId));
    if (m_StateController.IsResumable())
    {
        NN_MIGRATION_DETAIL_THROW_UNLESS(
            m_Context.GetNetworkServiceAccountId() == nsaId,
            ResultUnexpectedNetworkServiceAccountLoggedIn());
    }
    else
    {
        m_Context.SetNetworkServiceAccountId(nsaId);
        NN_MIGRATION_DETAIL_TRACE("[Client] Context created for NSA ID %016llx\n", m_Context.GetNetworkServiceAccountId().id);
    }

    if (!IsConnectionRequired())
    {
        NN_RESULT_SUCCESS;
    }

    nifm::NetworkConnection connection(os::EventClearMode_ManualClear);
    NN_MIGRATION_DETAIL_DO(detail::ConnectToInternet(connection, nifm::RequirementPreset_InternetBestEffort, pCancellable));
    NN_MIGRATION_DETAIL_DO(detail::Authenticator::PrepareNsaIdToken(m_LoginSession.Get(), pCancellable));
    NN_MIGRATION_DETAIL_DO(m_Idc.Initialize(m_LoginSession.Get(), pCancellable));
    NN_RESULT_SUCCESS;
}
NN_MIGRATION_USER_DEFINE_CLIENT_METHOD(
    bool,
    IsConnectionRequired() const)
{
    return !m_StateController.IsFinalized();
}

NN_MIGRATION_USER_DEFINE_CLIENT_METHOD(
    Result,
    ScanServers(const detail::Cancellable* pCancellable))
{
    account::NetworkServiceAccountId nsaId;
    NN_MIGRATION_DETAIL_DO(m_LoginSession.Get().GetNetworkServiceAccountId(&nsaId));

    m_ScanResult.count = 0u;
    size_t scanned;
    NN_MIGRATION_DETAIL_DO(m_ConnectionCreator.ScanServers(
        &scanned, m_ScanResult.servers, std::extent<decltype(m_ScanResult.servers)>::value,
        nsaId, pCancellable));
    m_ScanResult.count = scanned;
    NN_MIGRATION_DETAIL_TRACE("[Client] Scanned server: %d\n", scanned);
    NN_RESULT_SUCCESS;
}
NN_MIGRATION_USER_DEFINE_CLIENT_METHOD(
    size_t,
    ListServers(UserMigrationServerInfo* pOut, size_t count) const)
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);

    auto actualCount = std::min(m_ScanResult.count, count);
    std::memcpy(pOut, m_ScanResult.servers, actualCount * sizeof(*pOut));
    if (actualCount < count)
    {
        std::memset(pOut + actualCount, 0x00, (count - actualCount) * sizeof(*pOut));
    }
    NN_MIGRATION_DETAIL_TRACE("[Client] Listed server: %d\n", actualCount);
    return actualCount;
}
NN_MIGRATION_USER_DEFINE_CLIENT_METHOD(
    Result,
    Connect(const util::Uuid& serverId, const detail::Cancellable* pCancellable))
{
    NN_SDK_REQUIRES_NOT_EQUAL(serverId, util::InvalidUuid);

    NN_MIGRATION_DETAIL_TRACE("[Client] Connecting to server\n");
    account::NetworkServiceAccountId nsaId;
    NN_MIGRATION_DETAIL_DO(m_LoginSession.Get().GetNetworkServiceAccountId(&nsaId));

    typename ConnectionPolicy::Connection connection;
    NN_MIGRATION_DETAIL_DO(m_ConnectionCreator.template Connect<EncryptionPolicy>(&connection, serverId, nsaId, m_Idc, pCancellable));
    Connect(std::move(connection));

    NN_MIGRATION_DETAIL_DO(m_Idc.ExportUserMigrationClientInfo(m_Context, m_Connection, pCancellable));
    NN_MIGRATION_DETAIL_DO(m_Idc.WaitUserMigrationClientAccepted(m_Connection, pCancellable));

    if (m_StateController.IsResumable())
    {
        NN_MIGRATION_DETAIL_DO(m_Idc.MatchUserMigrationServerInfo(m_Context, m_Connection, pCancellable));
        NN_RESULT_SUCCESS;
    }

    NN_MIGRATION_DETAIL_DO(m_Idc.ImportUserMigrationServerInfo(m_Context, m_Connection, pCancellable));
    NN_MIGRATION_DETAIL_DO(m_Idc.ImportMigrationList(m_Context, m_Connection, pCancellable));
    NN_MIGRATION_DETAIL_DO(m_Context.ResetProgressInfo(&m_Resource.u.contextWorkBuffer, sizeof(m_Resource.u.contextWorkBuffer)));
    NN_MIGRATION_DETAIL_DO(m_Idc.SynchronizeStateInInitialization(m_StateController, m_Connection, pCancellable));
    NN_RESULT_SUCCESS;
}

NN_MIGRATION_USER_DEFINE_CLIENT_METHOD(
    account::Uid,
    GetImmigrantUid() const)
{
    return m_Context.GetUid();
}
NN_MIGRATION_USER_DEFINE_CLIENT_METHOD(
    int64_t,
    GetStorageShortfall() const)
{
    return static_cast<int64_t>(m_Context.CalculateCurrentShortfallOfUserSpace(
        &m_Resource.u.contextWorkBuffer, sizeof(m_Resource.u.contextWorkBuffer)));
}
NN_MIGRATION_USER_DEFINE_CLIENT_METHOD(
    TransferInfo,
    GetTotalTransferInfo() const)
{
    return m_Context.GetTotalTransferInfo();
}
NN_MIGRATION_USER_DEFINE_CLIENT_METHOD(
    TransferInfo,
    GetCurrentTransferInfo() const)
{
    return m_ProgressMonitor.GetTransferInfo();
}
NN_MIGRATION_USER_DEFINE_CLIENT_METHOD(
    size_t,
    GetCurrentRelatedApplications(ncm::ApplicationId* apps, size_t count) const)
{
    return m_Context.GetCurrentRelatedApplications(apps, count);
}

NN_MIGRATION_USER_DEFINE_CLIENT_METHOD(
    Result,
    TransferNext(const detail::Cancellable* pCancellable))
{
    NN_MIGRATION_DETAIL_THROW_UNLESS(m_IsConnected, detail::ResultNoConnectedDevice());

    if (!m_StateController.IsInitialized())
    {
        NN_MIGRATION_DETAIL_TRACE("[Client] State <- InTransfer\n");
        NN_MIGRATION_DETAIL_DO(m_StateController.SetStateInTransfer());
    }
    NN_MIGRATION_DETAIL_THROW_UNLESS(m_StateController.IsInitialized(), detail::ResultInvalidProtocol());

    auto index = m_Context.GetCurrentTransferInfo().count;

    detail::DataInfo dataInfo;
    auto count = m_Context.ReadMigrationList(&dataInfo, 1, index);
    NN_ABORT_UNLESS_EQUAL(count, 1u);

    NN_MIGRATION_DETAIL_TRACE("[Client] Importing savedata (#%d)\n", index);
    detail::SaveDataImporter importer(m_Context.GetUid(), dataInfo, m_ProgressMonitor);
    return m_Idc.ImportSaveData(importer, index, m_Connection, pCancellable);
}

NN_MIGRATION_USER_DEFINE_CLIENT_METHOD(
    Result,
    Suspend(const detail::Cancellable* pCancellable))
{
    if (m_IsConnected)
    {
        NN_MIGRATION_DETAIL_DO(m_Idc.Suspend(m_Connection, pCancellable));
        Disconnect(TimeSpan::FromMilliSeconds(ResetDelayMilliSeconds));
    }
    NN_RESULT_SUCCESS;
}

NN_MIGRATION_USER_DEFINE_CLIENT_METHOD(
    Result,
    Complete(const detail::Cancellable* pCancellable))
{
    m_LoginSession.Get().SetUid(m_Context.GetUid());
    if (!m_StateController.IsFinalized())
    {
        NN_MIGRATION_DETAIL_THROW_UNLESS(m_IsConnected, detail::ResultNoConnectedDevice());

        NN_MIGRATION_DETAIL_DO(m_Idc.SynchronizeStateInFinalization(m_StateController, m_Connection, pCancellable));
        m_LoginSession.Get().RegisterUser();
        NN_MIGRATION_DETAIL_DO(m_Idc.SynchronizeStateFinalized(m_StateController, m_Connection, pCancellable));
        Disconnect(TimeSpan::FromMilliSeconds(ResetDelayMilliSeconds));
    }
    NN_MIGRATION_DETAIL_THROW_UNLESS(m_StateController.IsFinalized(), detail::ResultInvalidProtocol());

    NN_MIGRATION_DETAIL_TRACE("[Client] Connecting to the internet\n");
    nifm::NetworkConnection connection(os::EventClearMode_ManualClear);
    NN_MIGRATION_DETAIL_DO(detail::ConnectToInternet(connection, nifm::RequirementPreset_InternetBestEffort, pCancellable));

    NN_MIGRATION_DETAIL_TRACE("[Client] Registering NSA and assets\n");
    NN_MIGRATION_DETAIL_DO(m_LoginSession.Get().RegisterNetworkServiceAccount(pCancellable));

    CleanupContext();
    NN_MIGRATION_DETAIL_TRACE("[Client] Migration done\n");
    NN_RESULT_SUCCESS;
}

NN_MIGRATION_USER_DEFINE_CLIENT_METHOD(
    Result,
    Abort())
{
    NN_MIGRATION_DETAIL_TRACE("[Client] Aborting\n");
    if (m_IsConnected)
    {
        Disconnect(TimeSpan::FromMilliSeconds(ResetDelayMilliSeconds));
    }

    if (!m_StateController.IsInitialized())
    {
        // 転送が行われていないのでコンテキストの削除のみ
        CleanupContext();
        NN_MIGRATION_DETAIL_TRACE("[Client] Aborted simply\n");
        NN_RESULT_SUCCESS;
    }

    // 転送が行われたので、セーブデータと、もしあればユーザーを削除する
    NN_MIGRATION_DETAIL_TRACE("[Client] Connecting to the internet\n");
    nifm::NetworkConnection connection(os::EventClearMode_ManualClear);
    NN_MIGRATION_DETAIL_DO(detail::ConnectToInternet(connection, nifm::RequirementPreset_InternetBestEffort, nullptr));

    NN_MIGRATION_DETAIL_TRACE("[Client] Cleaning up user data\n");
    NN_MIGRATION_DETAIL_DO(detail::ClearUserData(m_Context.GetUid(), nullptr));

    CleanupContext();
    NN_MIGRATION_DETAIL_TRACE("[Client] Aborted with user account deleted\n");
    NN_RESULT_SUCCESS;
}

// ---------------------------------------------------------------------------------------------
// デバッグ用の関数

NN_MIGRATION_USER_DEFINE_CLIENT_METHOD(
    Result,
    DebugSynchronizeStateInFinalization(const detail::Cancellable* pCancellable))
{
    NN_MIGRATION_DETAIL_THROW_UNLESS(m_IsConnected, detail::ResultNoConnectedDevice());
    NN_MIGRATION_DETAIL_THROW_UNLESS(!m_StateController.IsFinalized(), detail::ResultInvalidClientStateTransition());
    NN_MIGRATION_DETAIL_TRACE("[Client] [Debug] SynchronizeStateInFinalization\n");
    NN_MIGRATION_DETAIL_DO(m_Idc.SynchronizeStateInFinalization(m_StateController, m_Connection, pCancellable));
    NN_RESULT_SUCCESS;
}

#undef NN_MIGRATION_USER_DEFINE_CLIENT_METHOD

}}} // ~namesapce nn::migration::user
