﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/migration/migration_UserMigrationTypes.h>
#include <nn/migration/detail/migration_ServiceObjectAllocator.h>
#include <nn/migration/user/migration_Interface.sfdl.h>
#include <nn/migration/user/migration_UserMigrationServiceImpl.h>
#include <nn/sf/sf_ISharedObject.h>


namespace nn { namespace migration {

template <typename ImplConfiguration>
class UserMigrationServiceResource
{
public:
    static const size_t ThreadStackSize = 32 * 1024;
private:
    static const size_t AllocatorHeapSize = 256 * 1024u;
    detail::ServiceObjectAllocator<AllocatorHeapSize> m_ObjectAllocator;
    detail::ThreadResourceAllocator m_ThreadResourceAllocator;

    sf::EmplacedRef<user::IService, user::ServiceImpl<ImplConfiguration>> m_pService;

public:
    explicit UserMigrationServiceResource(UserMigrationRuntimeConfiguration& runtimeConfig) NN_NOEXCEPT
        : m_ThreadResourceAllocator(runtimeConfig.threadPriority, runtimeConfig.threadName)
    {
        m_ThreadResourceAllocator.Initialize<ThreadStackSize>(runtimeConfig.threadStackBuffer, runtimeConfig.threadStackBufferSize);
        m_pService = decltype(m_ObjectAllocator)::Factory::template CreateSharedEmplaced<user::IService, user::ServiceImpl<ImplConfiguration>>(
            &m_ObjectAllocator,
            m_ThreadResourceAllocator, runtimeConfig.loginSessionBuffer, runtimeConfig.loginSessionBufferSize);
        NN_ABORT_UNLESS(m_pService);
    }

    sf::SharedPointer<user::IService> GetServiceImpl() NN_NOEXCEPT
    {
        return sf::SharedPointer<user::IService>(m_pService);
    }
};

}} // ~namespace nn::migration


#include <nn/migration/idc/migration_KeyExchangeCommandEncryptor.h>
#include <nn/migration/idc/migration_MessageEncryptor.h>

#include <nn/migration/detail/migration_ConnectionPolicy.h>
#include <nn/migration/detail/migration_StoragePolicy.h>
#include <nn/migration/detail/migration_SubsystemInitializer.h>
#include <nn/migration/user/migration_UserMigrationClient.h>
#include <nn/migration/user/migration_UserMigrationConfig.h>
#include <nn/migration/user/migration_UserMigrationServer.h>

namespace nn { namespace migration {

struct EncryptionPolicy
{
#if defined(NN_BUILD_TARGET_PLATFORM_WIN)
    typedef idc::DebugKeyExchangeCommandEncryptor KeyEncryptor;
#else
    typedef idc::KeyExchangeCommandEncryptor KeyEncryptor;
#endif
    typedef idc::MessageEncryptor MessageEncryptor;
    static const size_t UserCommandBlockSize = 1024 * 1024;
    static const int KeyStretchingIteration = 1000;
    static const idc::KeyExchangeCommandConfig::Salt KeyStretchingSalt;
};

struct StateSystemSaveDataConfig
{
    static const char VolumeName[];
    static const fs::SystemSaveDataId Id = 0x8000000000000130ull;
    static const int64_t DataSize = 64 * 1024;
    static const int64_t JournalSize = 64 * 1024;
    static const uint32_t Flags = fs::SaveDataFlags_KeepAfterResettingSystemSaveDataWithoutUserSaveData;
};
struct ContextSystemSaveDataConfig
{
    static const char VolumeName[];
    static const fs::SystemSaveDataId Id = 0x8000000000000131ull;
    static const int64_t DataSize = 4 * 1024 * 1024;
    static const int64_t JournalSize = 3 * 1024 * 1024;
    static const uint32_t Flags = fs::SaveDataFlags_KeepAfterResettingSystemSaveDataWithoutUserSaveData;
};

struct DefaultImplConfiguration {
    typedef detail::SystemSaveDataStoragePolicy<StateSystemSaveDataConfig> StateStoragePolicy;
    typedef user::Server<
        detail::SystemSaveDataStoragePolicy<ContextSystemSaveDataConfig>,
        detail::LdnServerConnectionPolicy,
        EncryptionPolicy,
        detail::ServerSubsystemInitializer> ServerImpl;
    typedef user::Client<
        detail::SystemSaveDataStoragePolicy<ContextSystemSaveDataConfig>,
        detail::LdnClientConnectionPolicy,
        EncryptionPolicy,
        detail::ClientSubsystemInitializer> ClientImpl;

    using SocketBuffer = detail::SubsystemInitializer::SocketBuffer;
    static void AttachSocketBuffer(SocketBuffer& buffer) NN_NOEXCEPT
    {
        detail::SubsystemInitializer::AttachSocketBuffer(buffer);
    }
};

typedef UserMigrationServiceResource<DefaultImplConfiguration> DefaultUserMigrationServiceResource;

}} // ~namespace nn::migration
