﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/os/os_SystemEvent.h>

namespace nn { namespace migration { namespace detail {
class IAsyncContext;
}}} // ~namespace nn::migration::detail

namespace nn { namespace migration {

/**
    @brief 非同期処理の実行コンテキスト
    @details
        *this が非同期処理と関連づいている場合に限り、 *this の状態を「有効である」とします。
*/
class AsyncContext
{
    friend void swap(AsyncContext& o0, AsyncContext& o1) NN_NOEXCEPT;

private:
    detail::IAsyncContext* m_Ptr;

public:
    /**
        @brief デフォルト構築子
        @post *this は無効。
    */
    AsyncContext() NN_NOEXCEPT;
    explicit AsyncContext(detail::IAsyncContext* ptr) NN_NOEXCEPT;

    /**
        @brief ムーブ構築子
        @details
            *this が有効で未完了の場合、この関数の呼び出しは失敗します。
    */
    AsyncContext(AsyncContext&& rhs) NN_NOEXCEPT;

    /**
        @brief 解体子
        @details
            *this が有効で未完了の場合、この関数の呼び出しは失敗します。
    */
    ~AsyncContext() NN_NOEXCEPT;

    /**
        @brief ムーブ代入演算子
        @details
            *this が有効で未完了の場合、この関数の呼び出しは失敗します。
    */
    AsyncContext& operator=(AsyncContext&& rhs) NN_NOEXCEPT;

    /**
        @brief 処理の完了が通知されるイベントを取得します。
        @pre *this が有効
        @details
            取得されるイベントはこのインスタンスが有効な限り有効です。
    */
    void GetSystemEvent(os::SystemEvent* pOut) const NN_NOEXCEPT;

    /**
        @brief 処理の中止を要求します。
        @pre *this が有効
        @details
            この関数は中止の要求のみを行い、完了を待ちません。
    */
    void Cancel() NN_NOEXCEPT;

    /**
        @brief 処理が完了しているかを取得します。
        @pre *this が有効
    */
    bool HasDone() const NN_NOEXCEPT;

    /**
        @brief 処理の結果を取得します。

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理が成功しました。
            }
            @handleresult{
                nn::migration::ResultCanceled,
                Cancel() 関数の呼び出しで処理がキャンセルされました。
                この結果を失敗として扱わないでください。
            }
            @handleresult{
                nn::migration::ResultSuspendedByExternal,
                外部の要因で処理が中断されました。
            }
            @handleresult{
                nn::migration::ResultInternetConnectionUnavailable,
                インターネット接続を利用可能ではありません。
            }
        @endretresult

        @pre *this が有効で、完了している。
        @details
            この関数の説明に記載されていない Result の考慮が必要な場合があります。
            AsyncContext を返した関数の API リファレンスを参照してください。
    */
    Result GetResult() const NN_NOEXCEPT;
};

void swap(AsyncContext& o0, AsyncContext& o1) NN_NOEXCEPT;

}} // ~namespace nn::migration
