﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/migration/detail/migration_AsyncExecutionResource.h>
#include <nn/migration/detail/migration_Mutex.h>
#include <nn/migration/idc/migration_CommandTypes.h>
#include <nn/migration/idc/migration_Result.h>
#include <nn/migration/idc/migration_UserCommandEncryptor.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace migration { namespace idc {

template <typename EncryptionPolicy>
struct ThreadedUserCommandMediatorWorkBuffer
{
    Bit8 plainBlockBuffer[EncryptionPolicy::UserCommandBlockSize];
    Bit8 encryptedBlockBuffer[(EncryptionPolicy::UserCommandBlockSize + UserCommandAuthenticationSize) * 2 - 1];
};

/**
* @brief    ユーザーコンテキストの出力（平文）と通信路上の出力（暗号文）を仲介するクラス。
*           生成処理を別スレッドで動かすバージョン。
*/
template <typename EncryptionPolicy, typename UserContextType>
class ThreadedUserCommandProduceMediator
{
public:
    ThreadedUserCommandProduceMediator(
        typename EncryptionPolicy::MessageEncryptor& messageEncryptor,
        UserContextType* pUserContext,
        migration::detail::ThreadResourceType* pResource,
        ThreadedUserCommandMediatorWorkBuffer<EncryptionPolicy>* pWorkBuffer) NN_NOEXCEPT;

    ~ThreadedUserCommandProduceMediator() NN_NOEXCEPT;

    Result Produce(size_t* pOutProducedSize, void* outStream, size_t outStreamSize) NN_NOEXCEPT;

private:
    Result ProduceBody(size_t* pOutProducedSize, void* outStream, size_t outStreamSize) NN_NOEXCEPT;
    void StartProducerThread() NN_NOEXCEPT;
    static void ProducerFunc(void* arg) NN_NOEXCEPT;

    bool WaitSendable(migration::detail::LockGuard* lock) NN_NOEXCEPT;
    bool WaitEncryptable(migration::detail::LockGuard* lock) NN_NOEXCEPT;

    UserCommandEncryptor<typename EncryptionPolicy::MessageEncryptor> m_UserCommandEncryptor;
    UserContextType* m_pUserContext;
    migration::detail::ThreadResourceType* m_pThreadResource;
    ThreadedUserCommandMediatorWorkBuffer<EncryptionPolicy>* m_pWorkBuffer;

    bool m_IsProducingBody;

    size_t m_EncryptedSize;
    size_t m_TotalSentBodySize;

    os::ThreadType m_ThreadType;
    os::Event m_ProduceDoneEvent;
    os::Event m_StopEvent;
    os::ConditionVariable m_EncryptableCondition;
    os::ConditionVariable m_EncryptedCondition;
    migration::detail::Mutex m_SendBufferMutex;

    util::optional<Result> m_ProduceResult;
};

/**
* @brief    ユーザーコンテキストの入力（平文）と通信路上の入力（暗号文）を仲介するクラス。
*           消費処理を別スレッドで動かすバージョン。
*/
template <typename EncryptionPolicy, typename UserContextType, typename ErrorConfig>
class ThreadedUserCommandConsumeMediator
{
public:
    ThreadedUserCommandConsumeMediator(
        typename EncryptionPolicy::MessageEncryptor& messageEncryptor,
        UserContextType* pUserContext,
        migration::detail::ThreadResourceType* pResource,
        ThreadedUserCommandMediatorWorkBuffer<EncryptionPolicy>* pWorkBuffer) NN_NOEXCEPT;

    ~ThreadedUserCommandConsumeMediator() NN_NOEXCEPT;

    Result Consume(void* stream, size_t size) NN_NOEXCEPT;

private:
    Result ReceiveBody(void* stream, size_t size) NN_NOEXCEPT;
    void StartConsumerThread() NN_NOEXCEPT;
    static void ConsumerFunc(void* arg) NN_NOEXCEPT;

    bool WaitReceivable(migration::detail::LockGuard* lock) NN_NOEXCEPT;
    bool WaitDecryptable(migration::detail::LockGuard* lock) NN_NOEXCEPT;

    UserCommandDecryptor<typename EncryptionPolicy::MessageEncryptor> m_UserCommandDecryptor;
    UserContextType* m_pUserContext;
    migration::detail::ThreadResourceType* m_pThreadResource;
    ThreadedUserCommandMediatorWorkBuffer<EncryptionPolicy>* m_pWorkBuffer;

    bool m_IsConsumingBody;

    size_t  m_ReceivedSize;
    size_t  m_TotalReceivedBodySize;

    os::ThreadType m_ThreadType;
    os::Event m_ConsumeDoneEvent;
    os::Event m_StopEvent;
    os::ConditionVariable m_ReceivedCondition;
    os::ConditionVariable m_DecryptedCondition;
    migration::detail::Mutex m_ReceiveBufferMutex;

    util::optional<Result> m_ConsumeResult;
};

}}}

// 以下実装。

#include <nn/crypto.h>
#include <nn/migration/detail/migration_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace migration { namespace idc {

// ThreadedUserCommandProduceMediator

template <typename EncryptionPolicy, typename UserContextType>
ThreadedUserCommandProduceMediator<EncryptionPolicy, UserContextType>::ThreadedUserCommandProduceMediator(
    typename EncryptionPolicy::MessageEncryptor& messageEncryptor,
    UserContextType* pUserContext,
    migration::detail::ThreadResourceType* pThreadResource,
    ThreadedUserCommandMediatorWorkBuffer<EncryptionPolicy>* pWorkBuffer) NN_NOEXCEPT
    : m_UserCommandEncryptor(messageEncryptor)
    , m_pUserContext(pUserContext)
    , m_pThreadResource(pThreadResource)
    , m_pWorkBuffer(pWorkBuffer)
    , m_IsProducingBody(false)
    , m_EncryptedSize(0u)
    , m_TotalSentBodySize(0u)
    , m_ProduceDoneEvent(os::EventClearMode_ManualClear)
    , m_StopEvent(os::EventClearMode_ManualClear)
{
    NN_SDK_ASSERT_NOT_NULL(pUserContext);
    NN_SDK_ASSERT_NOT_NULL(pThreadResource);
    NN_SDK_ASSERT_NOT_NULL(pWorkBuffer);
    m_SendBufferMutex = NN_MIGRATION_DETAIL_MUTEX_INITIALIZER(false);
}

template <typename EncryptionPolicy, typename UserContextType>
ThreadedUserCommandProduceMediator<EncryptionPolicy, UserContextType>::~ThreadedUserCommandProduceMediator() NN_NOEXCEPT
{
    if( m_IsProducingBody )
    {
        m_StopEvent.Signal();
        m_ProduceDoneEvent.Wait();
        os::DestroyThread(&m_ThreadType);
    }
}

template <typename EncryptionPolicy, typename UserContextType>
Result ThreadedUserCommandProduceMediator<EncryptionPolicy, UserContextType>::Produce(size_t* pOutProducedSize, void* outStream, size_t outStreamSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutProducedSize);
    NN_SDK_REQUIRES_NOT_NULL(outStream);
    if( !m_IsProducingBody )
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(outStreamSize, sizeof(UserCommandHeader));
        auto pHeader = reinterpret_cast<UserCommandHeader*>(outStream);
        m_UserCommandEncryptor.Initialize(pHeader, this->m_pUserContext->GetProducableByteSize(), EncryptionPolicy::UserCommandBlockSize);
        *pOutProducedSize = sizeof(UserCommandHeader);

        StartProducerThread();

        NN_RESULT_THROW(ResultProduceCommandContinue());
    }
    return ProduceBody(pOutProducedSize, outStream, outStreamSize);
}

template <typename EncryptionPolicy, typename UserContextType>
Result ThreadedUserCommandProduceMediator<EncryptionPolicy, UserContextType>::ProduceBody(size_t* pOutProducedSize, void* outStream, size_t outStreamSize) NN_NOEXCEPT
{
    migration::detail::LockGuard lock;
    NN_RESULT_THROW_UNLESS(WaitSendable(&lock), *m_ProduceResult);

    *pOutProducedSize = std::min(outStreamSize, m_EncryptedSize);
    std::memcpy(outStream, m_pWorkBuffer->encryptedBlockBuffer, *pOutProducedSize);
    m_TotalSentBodySize += *pOutProducedSize;
    if( m_TotalSentBodySize == m_UserCommandEncryptor.GetTotalOutputSize() )
    {
        NN_RESULT_SUCCESS;
    }
    std::memmove(m_pWorkBuffer->encryptedBlockBuffer, m_pWorkBuffer + *pOutProducedSize, m_EncryptedSize - *pOutProducedSize);
    m_EncryptedSize -= *pOutProducedSize;
    if( sizeof(m_pWorkBuffer->encryptedBlockBuffer) - m_EncryptedSize >= EncryptionPolicy::UserCommandBlockSize + UserCommandAuthenticationSize )
    {
        m_EncryptableCondition.Signal();
    }
    NN_RESULT_THROW(ResultProduceCommandContinue());
}

template <typename EncryptionPolicy, typename UserContextType>
void ThreadedUserCommandProduceMediator<EncryptionPolicy, UserContextType>::StartProducerThread() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_IsProducingBody);
    NN_SDK_ASSERT_EQUAL(m_EncryptedSize, 0u);
    NN_SDK_ASSERT_EQUAL(m_TotalSentBodySize, 0u);

    m_IsProducingBody = true;

    NN_ABORT_UNLESS_RESULT_SUCCESS(os::CreateThread(&m_ThreadType, ThreadedUserCommandProduceMediator::ProducerFunc, this, m_pThreadResource->stack, m_pThreadResource->stackSize, m_pThreadResource->priority));
    os::SetThreadName(&m_ThreadType, m_pThreadResource->name);
    os::StartThread(&m_ThreadType);

    m_EncryptableCondition.Signal();
}

template <typename EncryptionPolicy, typename UserContextType>
void ThreadedUserCommandProduceMediator<EncryptionPolicy, UserContextType>::ProducerFunc(void* arg) NN_NOEXCEPT
{
    auto pMediator = reinterpret_cast<ThreadedUserCommandProduceMediator<EncryptionPolicy, UserContextType>*>(arg);

#define NN_MIGRATION_IDC_PRODUCE_FUNC_EXIT_UNLESS(condition, r) \
    if(!(condition))                                            \
    {                                                           \
        pMediator->m_ProduceResult = util::optional<Result>(r); \
        return;                                                 \
    }                                                           \

    NN_UTIL_SCOPE_EXIT
    {
        pMediator->m_ProduceDoneEvent.Signal();
    };

    size_t bufferedPlainUserCommandSize = 0u;
    while( NN_STATIC_CONDITION(true) )
    {
        size_t plainUserCommandSize = 0u;
        auto result = pMediator->m_pUserContext->Produce(
            &plainUserCommandSize, pMediator->m_pWorkBuffer->plainBlockBuffer + bufferedPlainUserCommandSize, sizeof(pMediator->m_pWorkBuffer->plainBlockBuffer) - bufferedPlainUserCommandSize);
        NN_MIGRATION_IDC_PRODUCE_FUNC_EXIT_UNLESS(result.IsSuccess() || ResultProduceCommandContinue::Includes(result) || ResultProduceCommandPause::Includes(result), result);
        if( ResultProduceCommandPause::Includes(result) )
        {
            NN_SDK_ASSERT_EQUAL(plainUserCommandSize, 0u);
            os::SleepThread(TimeSpan::FromSeconds(1));
            continue;
        }
        NN_SDK_ASSERT(result.IsSuccess() || ResultProduceCommandContinue::Includes(result));
        bufferedPlainUserCommandSize += plainUserCommandSize;
        NN_SDK_ASSERT_LESS_EQUAL(bufferedPlainUserCommandSize, EncryptionPolicy::UserCommandBlockSize + 0u);
        if( ResultProduceCommandContinue::Includes(result) && !(bufferedPlainUserCommandSize >= EncryptionPolicy::UserCommandBlockSize) )
        {
            continue;
        }
        migration::detail::LockGuard lock;
        NN_MIGRATION_IDC_PRODUCE_FUNC_EXIT_UNLESS(pMediator->WaitEncryptable(&lock), nullptr);
        size_t encryptedSize;
        pMediator->m_UserCommandEncryptor.Update(
            &encryptedSize,
            pMediator->m_pWorkBuffer->encryptedBlockBuffer + pMediator->m_EncryptedSize, sizeof(pMediator->m_pWorkBuffer->encryptedBlockBuffer) - pMediator->m_EncryptedSize,
            pMediator->m_pWorkBuffer->plainBlockBuffer, bufferedPlainUserCommandSize);
        pMediator->m_EncryptedSize += encryptedSize;
        bufferedPlainUserCommandSize = 0u;
        pMediator->m_EncryptedCondition.Signal();
        if( result.IsSuccess() )
        {
            break;
        }
    }
}

template <typename EncryptionPolicy, typename UserContextType>
bool ThreadedUserCommandProduceMediator<EncryptionPolicy, UserContextType>::WaitSendable(migration::detail::LockGuard* pOutLock) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutLock);
    NN_SDK_REQUIRES(!static_cast<bool>(*pOutLock));
    migration::detail::LockGuard lock(m_SendBufferMutex);
    while( !(m_EncryptedSize > 0 || m_ProduceDoneEvent.TryWait()) )
    {
        m_EncryptedCondition.TimedWait(m_SendBufferMutex.mutex, TimeSpan::FromSeconds(1));
    }
    if( !(m_EncryptedSize > 0) )
    {
        return false;

    }
    NN_SDK_ASSERT_GREATER(m_EncryptedSize, 0u);
    *pOutLock = std::move(lock);
    return true;
}

template <typename EncryptionPolicy, typename UserContextType>
bool ThreadedUserCommandProduceMediator<EncryptionPolicy, UserContextType>::WaitEncryptable(migration::detail::LockGuard* pOutLock) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutLock);
    NN_SDK_REQUIRES(!static_cast<bool>(*pOutLock));
    migration::detail::LockGuard lock(m_SendBufferMutex);
    while( !(sizeof(m_pWorkBuffer->encryptedBlockBuffer) - m_EncryptedSize >= EncryptionPolicy::UserCommandBlockSize + UserCommandAuthenticationSize || m_StopEvent.TryWait()) )
    {
        m_EncryptableCondition.TimedWait(m_SendBufferMutex.mutex, TimeSpan::FromSeconds(1));
    }
    if( m_StopEvent.TryWait() )
    {
        return false;
    }
    NN_SDK_ASSERT_GREATER_EQUAL(sizeof(m_pWorkBuffer->encryptedBlockBuffer) - EncryptionPolicy::UserCommandBlockSize + UserCommandAuthenticationSize, 0u);
    *pOutLock = std::move(lock);
    return true;
}

// ThreadedUserCommandConsumeMediator

template <typename EncryptionPolicy, typename UserContextType, typename ErrorConfig>
ThreadedUserCommandConsumeMediator<EncryptionPolicy, UserContextType, ErrorConfig>::ThreadedUserCommandConsumeMediator(
    typename EncryptionPolicy::MessageEncryptor& messageEncryptor,
    UserContextType* pUserContext,
    migration::detail::ThreadResourceType* pThreadResource,
    ThreadedUserCommandMediatorWorkBuffer<EncryptionPolicy>* pWorkBuffer) NN_NOEXCEPT
    : m_UserCommandDecryptor(messageEncryptor)
    , m_pUserContext(pUserContext)
    , m_pThreadResource(pThreadResource)
    , m_pWorkBuffer(pWorkBuffer)
    , m_IsConsumingBody(false)
    , m_ReceivedSize(0u)
    , m_TotalReceivedBodySize(0u)
    , m_ConsumeDoneEvent(os::EventClearMode_ManualClear)
    , m_StopEvent(os::EventClearMode_ManualClear)
{
    NN_SDK_ASSERT_NOT_NULL(pUserContext);
    NN_SDK_ASSERT_NOT_NULL(pThreadResource);
    NN_SDK_ASSERT_NOT_NULL(pWorkBuffer);
    m_ReceiveBufferMutex = NN_MIGRATION_DETAIL_MUTEX_INITIALIZER(false);
}

template <typename EncryptionPolicy, typename UserContextType, typename ErrorConfig>
ThreadedUserCommandConsumeMediator<EncryptionPolicy, UserContextType, ErrorConfig>::~ThreadedUserCommandConsumeMediator() NN_NOEXCEPT
{
    if( m_IsConsumingBody )
    {
        m_StopEvent.Signal();
        m_ConsumeDoneEvent.Wait();
        os::DestroyThread(&m_ThreadType);
    }
}

template <typename EncryptionPolicy, typename UserContextType, typename ErrorConfig>
Result ThreadedUserCommandConsumeMediator<EncryptionPolicy, UserContextType, ErrorConfig>::Consume(void* stream, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(stream);
    if( !m_IsConsumingBody )
    {
        // ヘッダ分受信するまでバッファリング。
        size_t copySize = 0u;
        {
            migration::detail::LockGuard lock(m_ReceiveBufferMutex);
            copySize = std::min(sizeof(UserCommandHeader) - m_ReceivedSize, size);
            std::memcpy(m_pWorkBuffer->encryptedBlockBuffer + m_ReceivedSize, stream, copySize);
            m_ReceivedSize += copySize;
            if( !(m_ReceivedSize >= sizeof(UserCommandHeader)) )
            {
                NN_RESULT_THROW(ResultConsumeCommandContinue());
            }
            NN_SDK_ASSERT_EQUAL(m_ReceivedSize, sizeof(UserCommandHeader));
            auto pHeader = reinterpret_cast<const UserCommandHeader*>(m_pWorkBuffer->encryptedBlockBuffer);
            NN_RESULT_THROW_UNLESS(m_UserCommandDecryptor.Initialize(*pHeader), typename ErrorConfig::ResultInvalidHeader());
            NN_RESULT_THROW_UNLESS(m_UserCommandDecryptor.GetBlockSize() == EncryptionPolicy::UserCommandBlockSize, typename ErrorConfig::ResultInvalidBlockSize());
            m_ReceivedSize = 0u;

            m_pUserContext->SetByteSizeToConsume(m_UserCommandDecryptor.GetCommandSize());
        }

        StartConsumerThread();

        size_t remainSize = size - copySize;
        if( remainSize > 0u )
        {
            auto remainStream = reinterpret_cast<Bit8*>(stream) + copySize;
            return ReceiveBody(remainStream, remainSize);
        }
        NN_RESULT_THROW(ResultConsumeCommandContinue());
    }
    return ReceiveBody(stream, size);
}

template <typename EncryptionPolicy, typename UserContextType, typename ErrorConfig>
Result ThreadedUserCommandConsumeMediator<EncryptionPolicy, UserContextType, ErrorConfig>::ReceiveBody(void* stream, size_t size) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_TotalReceivedBodySize + size <= m_UserCommandDecryptor.GetExpectedTotalInputSize(), typename ErrorConfig::ResultInvalidSize());

    m_TotalReceivedBodySize += size;

    size_t bufferedSize = 0;
    while( bufferedSize < size )
    {
        migration::detail::LockGuard lock;
        NN_RESULT_THROW_UNLESS(WaitReceivable(&lock), *m_ConsumeResult);

        // バッファリング用領域にデータをコピー。
        auto copySize = std::min(sizeof(m_pWorkBuffer->encryptedBlockBuffer) - m_ReceivedSize, size - bufferedSize);
        std::memcpy(m_pWorkBuffer->encryptedBlockBuffer + m_ReceivedSize, reinterpret_cast<Bit8*>(stream) + bufferedSize, copySize);
        bufferedSize += copySize;
        m_ReceivedSize += copySize;
        if( m_UserCommandDecryptor.GetAcceptableUpdateSize(m_ReceivedSize) > 0u )
        {
            m_ReceivedCondition.Signal();
        }
    }
    NN_SDK_ASSERT_EQUAL(bufferedSize, size);

    if( m_TotalReceivedBodySize == m_UserCommandDecryptor.GetExpectedTotalInputSize() )
    {
        // データを全て受信した場合は消費スレッドの終了を待って結果を返す。
        m_ConsumeDoneEvent.Wait();
        NN_SDK_ASSERT(m_ConsumeResult);
        return *m_ConsumeResult;
    }
    NN_RESULT_THROW(ResultConsumeCommandContinue());
}

template <typename EncryptionPolicy, typename UserContextType, typename ErrorConfig>
void ThreadedUserCommandConsumeMediator<EncryptionPolicy, UserContextType, ErrorConfig>::StartConsumerThread() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_IsConsumingBody);
    NN_SDK_ASSERT_EQUAL(m_ReceivedSize, 0u);

    m_IsConsumingBody = true;
    m_TotalReceivedBodySize = 0u;

    NN_ABORT_UNLESS_RESULT_SUCCESS(os::CreateThread(&m_ThreadType, ThreadedUserCommandConsumeMediator::ConsumerFunc, this, m_pThreadResource->stack, m_pThreadResource->stackSize, m_pThreadResource->priority));
    os::SetThreadName(&m_ThreadType, m_pThreadResource->name);
    os::StartThread(&m_ThreadType);
}

template <typename EncryptionPolicy, typename UserContextType, typename ErrorConfig>
bool ThreadedUserCommandConsumeMediator<EncryptionPolicy, UserContextType, ErrorConfig>::WaitReceivable(migration::detail::LockGuard* pOutLock) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutLock);
    NN_SDK_REQUIRES(!static_cast<bool>(*pOutLock));
    migration::detail::LockGuard lock(m_ReceiveBufferMutex);
    while( !(m_ReceivedSize < sizeof(m_pWorkBuffer->encryptedBlockBuffer) || m_ConsumeDoneEvent.TryWait()) )
    {
        m_DecryptedCondition.TimedWait(m_ReceiveBufferMutex.mutex, TimeSpan::FromSeconds(1));
    }
    if( m_ConsumeDoneEvent.TryWait() )
    {
        return false;
    }
    NN_SDK_ASSERT_LESS(m_ReceivedSize, sizeof(m_pWorkBuffer->encryptedBlockBuffer));
    *pOutLock = std::move(lock);
    return true;
}

template <typename EncryptionPolicy, typename UserContextType, typename ErrorConfig>
bool ThreadedUserCommandConsumeMediator<EncryptionPolicy, UserContextType, ErrorConfig>::WaitDecryptable(migration::detail::LockGuard* pOutLock) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutLock);
    NN_SDK_REQUIRES(!static_cast<bool>(*pOutLock));
    migration::detail::LockGuard lock(m_ReceiveBufferMutex);
    while( !(m_UserCommandDecryptor.GetAcceptableUpdateSize(m_ReceivedSize) > 0u || m_StopEvent.TryWait()) )
    {
        m_ReceivedCondition.TimedWait(m_ReceiveBufferMutex.mutex, nn::TimeSpan::FromSeconds(1));
    }
    if( m_StopEvent.TryWait() )
    {
        return false;
    }
    NN_SDK_ASSERT_GREATER(m_UserCommandDecryptor.GetAcceptableUpdateSize(m_ReceivedSize), 0u);
    *pOutLock = std::move(lock);
    return true;
}

template <typename EncryptionPolicy, typename UserContextType, typename ErrorConfig>
void ThreadedUserCommandConsumeMediator<EncryptionPolicy, UserContextType, ErrorConfig>::ConsumerFunc(void* arg) NN_NOEXCEPT
{
    auto pMediator = reinterpret_cast<ThreadedUserCommandConsumeMediator<EncryptionPolicy, UserContextType, ErrorConfig>*>(arg);

#define NN_MIGRATION_IDC_CONSUME_FUNC_EXIT_UNLESS(condition, r) \
    if(!(condition))                                            \
    {                                                           \
        pMediator->m_ConsumeResult = util::optional<Result>(r); \
        return;                                                 \
    }                                                           \

    NN_UTIL_SCOPE_EXIT
    {
        pMediator->m_ConsumeDoneEvent.Signal();
    };

    while( NN_STATIC_CONDITION(true) )
    {
        size_t plainSize;
        {
            migration::detail::LockGuard lock;
            NN_MIGRATION_IDC_CONSUME_FUNC_EXIT_UNLESS(pMediator->WaitDecryptable(&lock), nullptr);
            auto decryptableSize = std::min(EncryptionPolicy::UserCommandBlockSize + UserCommandAuthenticationSize, pMediator->m_UserCommandDecryptor.GetAcceptableUpdateSize(pMediator->m_ReceivedSize));
            NN_SDK_ASSERT_GREATER(decryptableSize, 0u);
            NN_MIGRATION_IDC_CONSUME_FUNC_EXIT_UNLESS(
                pMediator->m_UserCommandDecryptor.Update(
                    &plainSize,
                    pMediator->m_pWorkBuffer->plainBlockBuffer, sizeof(pMediator->m_pWorkBuffer->plainBlockBuffer),
                    pMediator->m_pWorkBuffer->encryptedBlockBuffer, decryptableSize),
                typename ErrorConfig::ResultInvalidData());
            pMediator->m_ReceivedSize -= decryptableSize;
            std::memmove(pMediator->m_pWorkBuffer->encryptedBlockBuffer, pMediator->m_pWorkBuffer->encryptedBlockBuffer + decryptableSize, pMediator->m_ReceivedSize);
            pMediator->m_DecryptedCondition.Signal();
        }

RetryUserConsumeFunc:

        auto result = pMediator->m_pUserContext->Consume(pMediator->m_pWorkBuffer->plainBlockBuffer, plainSize);
        NN_MIGRATION_IDC_CONSUME_FUNC_EXIT_UNLESS(result.IsSuccess() || ResultConsumeCommandContinue::Includes(result) || ResultConsumeCommandPause::Includes(result), result);
        if( ResultConsumeCommandPause::Includes(result) )
        {
            os::SleepThread(nn::TimeSpan::FromSeconds(1));
            goto RetryUserConsumeFunc;
        }
        else if( ResultConsumeCommandContinue::Includes(result) )
        {
            continue;
        }
        NN_SDK_ASSERT(result.IsSuccess());
        NN_MIGRATION_IDC_CONSUME_FUNC_EXIT_UNLESS(pMediator->m_ReceivedSize == 0u, typename ErrorConfig::ResultInvalidSize());
        pMediator->m_ConsumeResult = util::optional<Result>(nn::ResultSuccess());
        return;
    }
#undef NN_MIGRATION_IDC_CONSUME_FUNC_EXIT_UNLESS
}

}}}
