﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace migration { namespace idc {

/**
    @brief  通信プロトコルの暗号化に用いるパラメータの定義。
*/
struct MessageEncryptorConfig
{
    typedef Bit8 P2[16];
    typedef Bit8 Iv1[8];
    typedef uint32_t Counter;
    typedef Bit8 Iv2[sizeof(Iv1) + sizeof(Counter)];
    static const size_t SequenceManagementDataSize = sizeof(Iv2); //!< 暗号化時に付与するシーケンス管理用データのサイズ。
};

/**
    @brief  通信プロトコルの暗号化を行うクラス。
*/
class MessageEncryptor
{
    NN_DISALLOW_COPY(MessageEncryptor);
public:
    MessageEncryptor() NN_NOEXCEPT;
    ~MessageEncryptor() NN_NOEXCEPT;

    void Initialize(const Bit8 passphrase[], size_t passphraseSize, const Bit8 salt[], size_t saltSize, int iteration, MessageEncryptorConfig::Counter counterInitialValue) NN_NOEXCEPT;

    /**
        @brief      暗号化されたメッセージを作成します。暗号化されたメッセージにはシーケンス管理用のデータが付与されます。
        @param[out] pOutEncryptedSize   outEncrypted に書き込まれた暗号化されたメッセージのサイズ。
        @param[out] outEncrypted        暗号化されたメッセージを書き込むバッファ。
        @param[in]  outEncryptedSize    outEncrypted のサイズ。
        @param[out] outMac              MACを書き込むバッファ。
        @param[in]  outMacSize          outMac のサイズ。
        @param[in]  plain               平文。nullptr を指定した場合、outEncrypted にはシーケンス管理用のデータのみが書き込まれます。
        @param[in]  plainSize           plain のサイズ。plain が nullptr の場合は 0 である必要があります。
        @pre
                    - pOutEncryptedSize != null
                    - outEncrypted != null
                    - outEncryptedSize >= plainSize + MessageEncryptorConfig::SequenceManagementDataSize
                    - outMac != null
                    - outMacSize == crypto::Aes128GcmEncryptor::MacSize
                    - (plain != nullptr) || (plain == nullptr && plainSize == 0)
    */
    void Encrypt(
        size_t* pOutEncryptedSize,
        void* outEncrypted, size_t outEncryptedSize,
        void* outMac,  size_t outMacSize,
        const void* plain, size_t plainSize) NN_NOEXCEPT;

    /**
        @brief      暗号化されたメッセージを復号化します。MAC検証もしくはシーケンス管理用データの検証に失敗した場合は false を返します。
        @param[out] pOutPlainSize       outPlain に書き込まれた復号化されたメッセージのサイズ。
        @param[out] outPlain            復号化されたメッセージを書き込むバッファ。
        @param[in]  outPlainSize        outPlain のサイズ。
        @param[in]  encrypted           暗号文。
        @param[in]  encryptedSize       encrypted のサイズ。
        @param[in]  mac                 検証に用いる MAC値。
        @param[in]  macSize             mac のサイズ。
        @pre
                    - pOutPlainSize != null
                    - outPlain != nullptr
                    - outPlainSize >= encryptedSize - MessageEncryptorConfig::SequenceManagementDataSize
                    - encrypted != null
                    - encryptedSize >= MessageEncryptorConfig::SequenceManagementDataSize
                    - mac != null
                    - macSize == crypto::Aes128GcmDecryptor::MacSize
    */
    bool Decrypt(
        size_t* pOutPlainSize,
        void* outPlain, size_t outPlainSize,
        const void* encrypted, size_t encryptedSize,
        const void* mac, size_t macSize) NN_NOEXCEPT;

private:
    void GenerateIv2(MessageEncryptorConfig::Iv2& outValue) NN_NOEXCEPT;

    MessageEncryptorConfig::P2  m_P2;
    MessageEncryptorConfig::Iv1 m_Iv1;
    util::optional<MessageEncryptorConfig::Counter> m_Counter;
};

/**
 @brief （デバッグ用）通信プロトコルの暗号化を行うクラス。
*/
class DebugMessageEncryptor
{
    NN_DISALLOW_COPY(DebugMessageEncryptor);
public:
    DebugMessageEncryptor() NN_NOEXCEPT;

    void Initialize(const Bit8 passphrase[], size_t passphraseSize, const Bit8 salt[], size_t saltSize, int iteration, MessageEncryptorConfig::Counter counterInitialValue) NN_NOEXCEPT;

    void Encrypt(
        size_t* pOutEncryptedSize,
        void* outEncrypted, size_t outEncryptedSize,
        void* outMac,  size_t outMacSize,
        const void* plain, size_t plainSize) NN_NOEXCEPT;

    bool Decrypt(
        size_t* pOutDecryptedCount,
        void* outPlain, size_t outPlainSize,
        const void* encrypted, size_t encryptedSize,
        const void* mac, size_t macSize) NN_NOEXCEPT;

private:
    void GenerateManagementData(Bit8 outManagementData[], size_t outManagementDataSize) NN_NOEXCEPT;

    util::optional<MessageEncryptorConfig::Counter> m_Counter;
};

}}}
