﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/migration/idc/migration_CommandTypes.h>

namespace nn { namespace migration { namespace idc {

inline size_t GetResponseCommandSize(CommandKind command) NN_NOEXCEPT
{
    switch( command )
    {
    case CommandKind::Initiate0:    return sizeof(Initiate0Response);
    case CommandKind::Initiate1:    return sizeof(Initiate1Response);
    case CommandKind::Resume0:      return sizeof(Resume0Response);
    case CommandKind::Resume1:      return sizeof(Resume1Response);
    case CommandKind::Terminate:    return sizeof(TerminateResponse);
    case CommandKind::User:         return sizeof(UserCommandHeader);
    case CommandKind::Error:        return sizeof(ErrorResponse);
    default:                        NN_UNEXPECTED_DEFAULT;
    }
}

inline size_t GetRequestCommandSize(CommandKind command) NN_NOEXCEPT
{
    switch( command )
    {
    case CommandKind::Initiate0:    return sizeof(Initiate0Request);
    case CommandKind::Initiate1:    return sizeof(Initiate1Request);
    case CommandKind::Resume0:      return sizeof(Resume0Request);
    case CommandKind::Resume1:      return sizeof(Resume1Request);
    case CommandKind::Terminate:    return sizeof(TerminateRequest);
    case CommandKind::User:         return sizeof(UserCommandHeader);
    case CommandKind::Error:        // ErrorRequest doesn't exist.
    default:                        NN_UNEXPECTED_DEFAULT;
    }
}

// 鍵交換1

template <typename KeyEncryptor>
void CreateInitiate0Request(
    Initiate0Request* pOutRequest,
    const KeyExchangeCommandConfig::Challenge& clientChallenge,
    KeyEncryptor& encryptor)
NN_NOEXCEPT;

template <typename KeyEncryptor>
bool ParseInitiate0Request(
    KeyExchangeCommandConfig::Challenge& outClientChallenge,
    const Initiate0Request& request,
    KeyEncryptor& encryptor)
NN_NOEXCEPT;

template <typename KeyEncryptor>
void CreateInitiate0Response(
    Initiate0Response* pOutResponse,
    const KeyExchangeCommandConfig::Iv0& iv0,
    const KeyExchangeCommandConfig::Passphrase& passphrase,
    const KeyExchangeCommandConfig::Challenge& clientChallenge,
    const KeyExchangeCommandConfig::Challenge& serverChallenge,
    KeyEncryptor& encryptor)
NN_NOEXCEPT;

template <typename KeyEncryptor>
bool ParseInitiate0Response(
    KeyExchangeCommandConfig::Passphrase& outPassphrase,
    KeyExchangeCommandConfig::Challenge& outServerChallenge,
    const Initiate0Response& response,
    const KeyExchangeCommandConfig::Challenge& clientChallenge,
    KeyEncryptor& encryptor)
NN_NOEXCEPT;

// 鍵交換2

template <typename MsgEncryptor>
void CreateInitiate1Request(
    Initiate1Request* pOutRequest,
    const KeyExchangeCommandConfig::Challenge& serverChallenge,
    MsgEncryptor& encryptor)
NN_NOEXCEPT;

template <typename MsgEncryptor>
bool ParseInitiate1Request(
    const Initiate1Request& request,
    const KeyExchangeCommandConfig::Challenge& serverChallenge,
    MsgEncryptor& encryptor)
NN_NOEXCEPT;

template <typename MsgEncryptor>
void CreateInitiate1Response(
    Initiate1Response* pOutResponse,
    MsgEncryptor& encryptor)
NN_NOEXCEPT;

template <typename MsgEncryptor>
bool ParseInitiate1Response(
    const Initiate1Response& response,
    MsgEncryptor& encryptor)
NN_NOEXCEPT;

// 再開1

template <typename MsgEncryptor>
void CreateResume0Request(
    Resume0Request* pOutRequest,
    const KeyExchangeCommandConfig::Challenge& clientChallenge,
    MsgEncryptor& encryptor)
NN_NOEXCEPT;

template <typename MsgEncryptor>
bool ParseResume0Request(
    KeyExchangeCommandConfig::Challenge& outClientChallenge,
    const Resume0Request& request,
    MsgEncryptor& encryptor)
NN_NOEXCEPT;

template <typename MsgEncryptor>
void CreateResume0Response(
    Resume0Response* pOutResponse,
    const KeyExchangeCommandConfig::Challenge& clientChallenge,
    const KeyExchangeCommandConfig::Challenge& serverChallenge,
    MsgEncryptor& encryptor)
NN_NOEXCEPT;

template <typename MsgEncryptor>
bool ParseResume0Response(
    KeyExchangeCommandConfig::Challenge& outServerChallenge,
    const Resume0Response& response,
    const KeyExchangeCommandConfig::Challenge& clientChallenge,
    MsgEncryptor& encryptor)
NN_NOEXCEPT;

// 再開2

template <typename MsgEncryptor>
void CreateResume1Request(
    Resume1Request* pOutRequest,
    const KeyExchangeCommandConfig::Challenge& serverChallenge,
    MsgEncryptor& encryptor)
NN_NOEXCEPT;

template <typename MsgEncryptor>
bool ParseResume1Reqeust(
    const Resume1Request& request,
    const KeyExchangeCommandConfig::Challenge& serverChallenge,
    MsgEncryptor& encryptor)
NN_NOEXCEPT;

template <typename MsgEncryptor>
void CreateResume1Response(
    Resume1Response* pOutResponse,
    MsgEncryptor& encryptor)
NN_NOEXCEPT;

template <typename MsgEncryptor>
bool ParseResume1Response(
    const Resume1Response& response,
    MsgEncryptor& encryptor)
NN_NOEXCEPT;

// 終了

template <typename MsgEncryptor>
void CreateTerminateRequest(
    TerminateRequest* pOutRequest,
    MsgEncryptor& encryptor)
NN_NOEXCEPT;

template <typename MsgEncryptor>
bool ParseTerminateRequest(
    const TerminateRequest& request,
    MsgEncryptor& encryptor)
NN_NOEXCEPT;

template <typename MsgEncryptor>
void CreateTerminateResponse(
    TerminateResponse* pOutResponse,
    MsgEncryptor& encryptor)
NN_NOEXCEPT;

template <typename MsgEncryptor>
bool ParseTerminateResponse(
    const TerminateResponse& response,
    MsgEncryptor& encryptor)
NN_NOEXCEPT;

// エラー

template <typename MsgEncryptor>
void CreateErrorResponse(
    ErrorResponse* pOutResponse,
    const ErrorInfo& errorInfo,
    MsgEncryptor& encryptor)
NN_NOEXCEPT;

template <typename MsgEncryptor>
bool ParseErrorResponse(
    ErrorInfo* pOutErrorInfo,
    const ErrorResponse& response,
    MsgEncryptor& encryptor)
NN_NOEXCEPT;

}}}

// 以下実装

#include <nn/nn_SdkAssert.h>
#include <nn/crypto.h>
#include <nn/crypto/crypto_Compare.h>
#include <nn/migration/detail/migration_Log.h>
#include <nn/migration/idc/detail/migration_Util.h>
#include <nn/util/util_ScopeExit.h>

namespace nn { namespace migration { namespace idc {

// 鍵交換1

template <typename KeyEncryptor>
void CreateInitiate0Request(
    Initiate0Request* pOutRequest,
    const KeyExchangeCommandConfig::Challenge& clientChallenge,
    KeyEncryptor& encryptor)
    NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutRequest);
    NN_SDK_REQUIRES_NOT_NULL(clientChallenge);

    pOutRequest->commandId = CommandKind::Initiate0;
    std::memset(pOutRequest->padding, 0, sizeof(pOutRequest->padding));
    std::memcpy(pOutRequest->data, clientChallenge, sizeof(pOutRequest->data));

    encryptor.GenerateMacWithSalt(pOutRequest->mac, sizeof(pOutRequest->mac), clientChallenge);
}

template <typename KeyEncryptor>
bool ParseInitiate0Request(
    KeyExchangeCommandConfig::Challenge& outClientChallenge,
    const Initiate0Request& request,
    KeyEncryptor& encryptor)
    NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outClientChallenge);
    NN_SDK_REQUIRES_EQUAL(request.commandId, CommandKind::Initiate0);

    Bit8 mac[sizeof(request.mac)];
    encryptor.GenerateMacWithSalt(mac, sizeof(mac), request.data);

    if( !crypto::IsSameBytes(request.mac, mac, sizeof(mac)) )
    {
        return false;
    }

    std::memcpy(outClientChallenge, request.data, KeyExchangeCommandConfig::ChallengeSize);

    return true;
}

template <typename KeyEncryptor>
void CreateInitiate0Response(
    Initiate0Response* pOutResponse,
    const KeyExchangeCommandConfig::Iv0& iv0,
    const KeyExchangeCommandConfig::Passphrase& passphrase,
    const KeyExchangeCommandConfig::Challenge& clientChallenge,
    const KeyExchangeCommandConfig::Challenge& serverChallenge,
    KeyEncryptor& encryptor)
NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutResponse);
    NN_SDK_REQUIRES_NOT_NULL(iv0);
    NN_SDK_REQUIRES_NOT_NULL(passphrase);
    NN_SDK_REQUIRES_NOT_NULL(clientChallenge);
    NN_SDK_REQUIRES_NOT_NULL(serverChallenge);

    pOutResponse->commandId = CommandKind::Initiate0;
    std::memset(pOutResponse->padding, 0, sizeof(pOutResponse->padding));
    std::memcpy(pOutResponse->iv, iv0, sizeof(pOutResponse->iv));

    Bit8 plain[sizeof(pOutResponse->data)] = {};
    NN_UTIL_SCOPE_EXIT{ detail::SecureMemoryZero(plain, sizeof(plain)); };
    std::memcpy(plain, passphrase, KeyExchangeCommandConfig::PassphraseSize);
    std::memcpy(plain + KeyExchangeCommandConfig::PassphraseSize, clientChallenge, KeyExchangeCommandConfig::ChallengeSize);
    std::memcpy(plain + KeyExchangeCommandConfig::PassphraseSize + KeyExchangeCommandConfig::ChallengeSize, serverChallenge, KeyExchangeCommandConfig::ChallengeSize);

    encryptor.Encrypt(pOutResponse->data, sizeof(pOutResponse->data), plain, sizeof(plain), pOutResponse->iv, sizeof(pOutResponse->iv));
    encryptor.GenerateMac(pOutResponse->mac, sizeof(pOutResponse->mac), plain, sizeof(plain));
}

template <typename KeyEncryptor>
bool ParseInitiate0Response(
    KeyExchangeCommandConfig::Passphrase& outPassphrase,
    KeyExchangeCommandConfig::Challenge& outServerChallenge,
    const Initiate0Response& response,
    const KeyExchangeCommandConfig::Challenge& clientChallenge,
    KeyEncryptor& encryptor)
NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outPassphrase);
    NN_SDK_REQUIRES_NOT_NULL(outServerChallenge);
    NN_SDK_REQUIRES_EQUAL(response.commandId, CommandKind::Initiate0);
    NN_SDK_REQUIRES_NOT_NULL(clientChallenge);

    Bit8 plain[sizeof(response.data)];
    NN_UTIL_SCOPE_EXIT{ detail::SecureMemoryZero(plain, sizeof(plain)); };
    encryptor.Decrypt(plain, sizeof(plain), response.data, sizeof(response.data), response.iv, sizeof(response.iv));

    Bit8 mac[sizeof(response.mac)];
    encryptor.GenerateMac(mac, sizeof(mac), plain, sizeof(plain));

    if( !crypto::IsSameBytes(mac, response.mac, sizeof(mac)) )
    {
        return false;
    }

    if( !crypto::IsSameBytes(clientChallenge, &plain[KeyExchangeCommandConfig::PassphraseSize], KeyExchangeCommandConfig::ChallengeSize) )
    {
        return false;
    }

    std::memcpy(outPassphrase, plain, KeyExchangeCommandConfig::PassphraseSize);
    std::memcpy(outServerChallenge, plain + KeyExchangeCommandConfig::PassphraseSize + KeyExchangeCommandConfig::ChallengeSize, KeyExchangeCommandConfig::ChallengeSize);

    return true;
}

// 鍵交換2

template <typename MsgEncryptor>
void CreateInitiate1Request(
    Initiate1Request* pOutRequest,
    const KeyExchangeCommandConfig::Challenge& serverChallenge,
    MsgEncryptor& encryptor)
NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutRequest);
    NN_SDK_REQUIRES_NOT_NULL(serverChallenge);

    pOutRequest->commandId = CommandKind::Initiate1;
    std::memset(pOutRequest->padding, 0, sizeof(pOutRequest->padding));
    size_t encryptedSize;
    encryptor.Encrypt(
        &encryptedSize,
        pOutRequest->data, sizeof(pOutRequest->data),
        pOutRequest->mac, sizeof(pOutRequest->mac),
        serverChallenge, KeyExchangeCommandConfig::ChallengeSize);
    NN_SDK_ASSERT_EQUAL(encryptedSize, sizeof(pOutRequest->data));
    NN_UNUSED(encryptedSize);
}

template <typename MsgEncryptor>
bool ParseInitiate1Request(
    const Initiate1Request& request,
    const KeyExchangeCommandConfig::Challenge& serverChallenge,
    MsgEncryptor& encryptor)
NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(request.commandId, CommandKind::Initiate1);
    NN_SDK_REQUIRES_NOT_NULL(serverChallenge);

    size_t decryptedSize;
    Bit8 decrypted[sizeof(request.data) - MessageEncryptorConfig::SequenceManagementDataSize];
    if( !encryptor.Decrypt(
        &decryptedSize,
        decrypted, sizeof(decrypted),
        request.data, sizeof(request.data),
        request.mac, sizeof(request.mac)) )
    {
        return false;
    }
    NN_SDK_ASSERT_EQUAL(decryptedSize, sizeof(decrypted));
    NN_UNUSED(decryptedSize);
    return crypto::IsSameBytes(serverChallenge, decrypted, KeyExchangeCommandConfig::ChallengeSize);
}

template <typename MsgEncryptor>
void CreateInitiate1Response(
    Initiate1Response* pOutResponse,
    MsgEncryptor& encryptor)
NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutResponse);

    pOutResponse->commandId = CommandKind::Initiate1;
    std::memset(pOutResponse->padding, 0, sizeof(pOutResponse->padding));
    size_t encryptedSize;
    encryptor.Encrypt(
        &encryptedSize,
        pOutResponse->data, sizeof(pOutResponse->data),
        pOutResponse->mac, sizeof(pOutResponse->mac),
        nullptr, 0);
    NN_SDK_ASSERT_EQUAL(encryptedSize, sizeof(pOutResponse->data));
    NN_UNUSED(encryptedSize);
}

template <typename MsgEncryptor>
bool ParseInitiate1Response(
    const Initiate1Response& response,
    MsgEncryptor& encryptor)
NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(response.commandId, CommandKind::Initiate1);

    size_t decryptedSize;
    Bit8 decrypted[1]; // dummy
    return encryptor.Decrypt(
        &decryptedSize,
        decrypted, sizeof(decrypted),
        response.data, sizeof(response.data),
        response.mac, sizeof(response.mac));
}

// 再開1

template <typename MsgEncryptor>
void CreateResume0Request(
    Resume0Request* pOutRequest,
    const KeyExchangeCommandConfig::Challenge& clientChallenge,
    MsgEncryptor& encryptor)
NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutRequest);

    pOutRequest->commandId = CommandKind::Resume0;
    std::memset(pOutRequest->padding, 0, sizeof(pOutRequest->padding));
    size_t encryptedSize;
    encryptor.Encrypt(
        &encryptedSize,
        pOutRequest->data, sizeof(pOutRequest->data),
        pOutRequest->mac, sizeof(pOutRequest->mac),
        clientChallenge, KeyExchangeCommandConfig::ChallengeSize);
    NN_SDK_ASSERT_EQUAL(encryptedSize, sizeof(pOutRequest->data));
}

template <typename MsgEncryptor>
bool ParseResume0Request(
    KeyExchangeCommandConfig::Challenge& outClientChallenge,
    const Resume0Request& request,
    MsgEncryptor& encryptor)
NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outClientChallenge);
    NN_SDK_REQUIRES_EQUAL(request.commandId, CommandKind::Resume0);

    size_t decryptedSize;
    return encryptor.Decrypt(
        &decryptedSize,
        outClientChallenge, KeyExchangeCommandConfig::ChallengeSize,
        request.data, sizeof(request.data),
        request.mac, sizeof(request.mac));
}

template <typename MsgEncryptor>
void CreateResume0Response(
    Resume0Response* pOutResponse,
    const KeyExchangeCommandConfig::Challenge& clientChallenge,
    const KeyExchangeCommandConfig::Challenge& serverChallenge,
    MsgEncryptor& encryptor)
NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutResponse);
    NN_SDK_REQUIRES_NOT_NULL(clientChallenge);
    NN_SDK_REQUIRES_NOT_NULL(serverChallenge);

    pOutResponse->commandId = CommandKind::Resume0;
    std::memset(pOutResponse->padding, 0, sizeof(pOutResponse->padding));
    Bit8 plain[KeyExchangeCommandConfig::ChallengeSize * 2];
    std::memcpy(plain, clientChallenge, KeyExchangeCommandConfig::ChallengeSize);
    std::memcpy(plain + KeyExchangeCommandConfig::ChallengeSize, serverChallenge, KeyExchangeCommandConfig::ChallengeSize);
    size_t encryptedSize;
    encryptor.Encrypt(
        &encryptedSize,
        pOutResponse->data, sizeof(pOutResponse->data),
        pOutResponse->mac, sizeof(pOutResponse->mac),
        plain, sizeof(plain));
    NN_SDK_ASSERT_EQUAL(encryptedSize, sizeof(pOutResponse->data));
}

template <typename MsgEncryptor>
bool ParseResume0Response(
    KeyExchangeCommandConfig::Challenge& outServerChallenge,
    const Resume0Response& response,
    const KeyExchangeCommandConfig::Challenge& clientChallenge,
    MsgEncryptor& encryptor)
NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outServerChallenge);
    NN_SDK_REQUIRES_EQUAL(response.commandId, CommandKind::Resume0);

    Bit8 decrypted[KeyExchangeCommandConfig::ChallengeSize * 2];
    size_t decryptedSize;
    if( !encryptor.Decrypt(
        &decryptedSize,
        decrypted, sizeof(decrypted),
        response.data, sizeof(response.data),
        response.mac, sizeof(response.mac)) )
    {
        return false;
    }

    if( !crypto::IsSameBytes(clientChallenge, decrypted, KeyExchangeCommandConfig::ChallengeSize) )
    {
        return false;
    }

    std::memcpy(outServerChallenge, decrypted + KeyExchangeCommandConfig::ChallengeSize, KeyExchangeCommandConfig::ChallengeSize);

    return true;
}

// 再開2

template <typename MsgEncryptor>
void CreateResume1Request(
    Resume1Request* pOutRequest,
    const KeyExchangeCommandConfig::Challenge& serverChallenge,
    MsgEncryptor& encryptor)
NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutRequest);
    NN_SDK_REQUIRES_NOT_NULL(serverChallenge);

    pOutRequest->commandId = CommandKind::Resume1;
    std::memset(pOutRequest->padding, 0, sizeof(pOutRequest->padding));
    size_t encryptedSize;
    encryptor.Encrypt(
        &encryptedSize,
        pOutRequest->data, sizeof(pOutRequest->data),
        pOutRequest->mac, sizeof(pOutRequest->mac),
        serverChallenge, KeyExchangeCommandConfig::ChallengeSize);
}

template <typename MsgEncryptor>
bool ParseResume1Reqeust(
    const Resume1Request& request,
    const KeyExchangeCommandConfig::Challenge& serverChallenge,
    MsgEncryptor& encryptor)
NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(serverChallenge);
    NN_SDK_REQUIRES_EQUAL(request.commandId, CommandKind::Resume1);

    Bit8 decrypted[KeyExchangeCommandConfig::ChallengeSize];
    size_t decryptedSize;
    if( !encryptor.Decrypt(
        &decryptedSize,
        decrypted, sizeof(decrypted),
        request.data, sizeof(request.data),
        request.mac, sizeof(request.mac)) )
    {
        return false;
    }
    NN_SDK_ASSERT_EQUAL(decryptedSize, sizeof(decrypted));

    return crypto::IsSameBytes(serverChallenge, decrypted, KeyExchangeCommandConfig::ChallengeSize);
}

template <typename MsgEncryptor>
void CreateResume1Response(
    Resume1Response* pOutResponse,
    MsgEncryptor& encryptor)
NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutResponse);

    pOutResponse->commandId = CommandKind::Resume1;
    std::memset(pOutResponse->padding, 0, sizeof(pOutResponse->padding));
    size_t encryptedSize;
    encryptor.Encrypt(
        &encryptedSize,
        pOutResponse->data, sizeof(pOutResponse->data),
        pOutResponse->mac, sizeof(pOutResponse->mac),
        nullptr, 0);
    NN_SDK_ASSERT_EQUAL(encryptedSize, sizeof(pOutResponse->data));
}

template <typename MsgEncryptor>
bool ParseResume1Response(
    const Resume1Response& response,
    MsgEncryptor& encryptor)
NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(response.commandId, CommandKind::Resume1);

    size_t decryptedSize;
    Bit8 decrypted[1];
    return encryptor.Decrypt(
        &decryptedSize,
        decrypted, sizeof(decrypted),
        response.data, sizeof(response.data),
        response.mac, sizeof(response.mac));
}

// 終了

template <typename MsgEncryptor>
void CreateTerminateRequest(
    TerminateRequest* pOutRequest,
    MsgEncryptor& encryptor)
NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutRequest);

    pOutRequest->commandId = CommandKind::Terminate;
    std::memset(pOutRequest->padding, 0, sizeof(pOutRequest->padding));
    size_t encryptedSize;
    encryptor.Encrypt(
        &encryptedSize,
        pOutRequest->data, sizeof(pOutRequest->data),
        pOutRequest->mac, sizeof(pOutRequest->mac),
        nullptr, 0);
    NN_SDK_ASSERT_EQUAL(encryptedSize, sizeof(pOutRequest->data));
}

template <typename MsgEncryptor>
bool ParseTerminateRequest(
    const TerminateRequest& request,
    MsgEncryptor& encryptor)
NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(request.commandId, CommandKind::Terminate);

    size_t decryptedSize;
    Bit8 decrypted[1];
    return encryptor.Decrypt(
        &decryptedSize,
        decrypted, sizeof(decrypted),
        request.data, sizeof(request.data),
        request.mac, sizeof(request.mac));
}

template <typename MsgEncryptor>
void CreateTerminateResponse(
    TerminateResponse* pOutResponse,
    MsgEncryptor& encryptor)
NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutResponse);

    pOutResponse->commandId = CommandKind::Terminate;
    std::memset(pOutResponse->padding, 0, sizeof(pOutResponse->padding));
    size_t encryptedSize;
    encryptor.Encrypt(
        &encryptedSize,
        pOutResponse->data, sizeof(pOutResponse->data),
        pOutResponse->mac, sizeof(pOutResponse->mac),
        nullptr, 0);
    NN_SDK_ASSERT_EQUAL(encryptedSize, sizeof(pOutResponse->data));
}

template <typename MsgEncryptor>
bool ParseTerminateResponse(
    const TerminateResponse& response,
    MsgEncryptor& encryptor)
NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(response.commandId, CommandKind::Terminate);

    size_t decryptedSize;
    Bit8 decrypted[1];
    return encryptor.Decrypt(
        &decryptedSize,
        decrypted, sizeof(decrypted),
        response.data, sizeof(response.data),
        response.mac, sizeof(response.mac));
}

// エラー

template <typename MsgEncryptor>
void CreateErrorResponse(
    ErrorResponse* pOutResponse,
    const ErrorInfo& errorInfo,
    MsgEncryptor& encryptor)
NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutResponse);

    pOutResponse->commandId = CommandKind::Error;
    size_t encryptedSize;
    encryptor.Encrypt(
        &encryptedSize,
        pOutResponse->data, sizeof(pOutResponse->data),
        pOutResponse->mac, sizeof(pOutResponse->mac),
        &errorInfo, sizeof(ErrorInfo));
    NN_SDK_ASSERT_EQUAL(encryptedSize, sizeof(pOutResponse->data));
}

template <typename MsgEncryptor>
bool ParseErrorResponse(
    ErrorInfo* pOutErrorInfo,
    const ErrorResponse& response,
    MsgEncryptor& encryptor)
NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutErrorInfo);

    size_t decryptedSize;
    return encryptor.Decrypt(
        &decryptedSize,
        pOutErrorInfo, sizeof(ErrorInfo),
        response.data, sizeof(response.data),
        response.mac, sizeof(response.mac));
}

}}}
