﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/migration/detail/migration_AsyncExecutionResource.h>
#include <nn/migration/idc/migration_ClientContext.h>
#include <nn/migration/idc/migration_CommunicationApi.h>
#include <nn/migration/idc/migration_CommandTypes.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

namespace nn { namespace migration { namespace idc {

class Client
{
public:

    template <typename ClientContextType>
    struct InvokeUserCommandBuffer
    {
        Bit8 communicationBuffer[ClientContextType::EncryptionPolicy::UserCommandBlockSize + UserCommandAuthenticationSize];
        UserCommandMediatorWorkBuffer<typename ClientContextType::EncryptionPolicy> userCommandMediatorWorkBuffer;
    };

    template <typename ClientContextType>
    struct InvokeUserCommandBufferForThreaded
    {
        Bit8 communicationBuffer[ClientContextType::EncryptionPolicy::UserCommandBlockSize + UserCommandAuthenticationSize];
        ThreadedUserCommandMediatorWorkBuffer<typename ClientContextType::EncryptionPolicy> userCommandMediatorWorkBuffer;
    };

    template <typename ConnectionType, typename ClientContextType>
    static Result Initiate(const ConnectionType& connection, ClientContextType& clientContext, int timeoutSeconds,
        void* workBuffer, size_t workBufferSize, const migration::detail::Cancellable* pCancellable) NN_NOEXCEPT
    {
        NN_RESULT_DO(Invoke(CommandKind::Initiate0, connection, clientContext, timeoutSeconds, workBuffer, workBufferSize, pCancellable));
        NN_RESULT_DO(Invoke(CommandKind::Initiate1, connection, clientContext, timeoutSeconds, workBuffer, workBufferSize, pCancellable));
        NN_RESULT_SUCCESS;
    }

    template <typename ConnectionType, typename ClientContextType>
    static Result Resume(const ConnectionType& connection, ClientContextType& clientContext, int timeoutSeconds,
        void* workBuffer, size_t workBufferSize, const migration::detail::Cancellable* pCancellable) NN_NOEXCEPT
    {
        NN_RESULT_DO(Invoke(CommandKind::Resume0, connection, clientContext, timeoutSeconds, workBuffer, workBufferSize, pCancellable));
        NN_RESULT_DO(Invoke(CommandKind::Resume1, connection, clientContext, timeoutSeconds, workBuffer, workBufferSize, pCancellable));
        NN_RESULT_SUCCESS;
    }

    template <typename ConnectionType, typename ClientContextType>
    static Result Terminate(const ConnectionType& connection, ClientContextType& clientContext, int timeoutSeconds,
        void* workBuffer, size_t workBufferSize, const migration::detail::Cancellable* pCancellable) NN_NOEXCEPT
    {
        NN_RESULT_DO(Invoke(CommandKind::Terminate, connection, clientContext, timeoutSeconds, workBuffer, workBufferSize, pCancellable));
        NN_RESULT_SUCCESS;
    }

    // 受信/消費を並列化していない。
    template <typename ConnectionType, typename ClientContextType, typename UserContextType>
    static Result InvokeUserCommand(const ConnectionType& connection, ClientContextType& clientContext, UserContextType& userContext, int timeoutSeconds,
        void* workBuffer, size_t workBufferSize, const migration::detail::Cancellable* pCancellable,
        migration::idc::detail::TransferSpeedMonitor* pSpeedMonitor = nullptr) NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(std::alignment_of<InvokeUserCommandBuffer<ClientContextType>>::value == 1u);
        NN_SDK_REQUIRES_GREATER_EQUAL(timeoutSeconds, 0);
        NN_SDK_REQUIRES_NOT_NULL(workBuffer);
        NN_SDK_REQUIRES_GREATER_EQUAL(workBufferSize, sizeof(InvokeUserCommandBuffer<ClientContextType>));
        NN_UNUSED(workBufferSize);
        auto pBuffer = reinterpret_cast<InvokeUserCommandBuffer<ClientContextType>*>(workBuffer);
        {
            clientContext.SetCommandKind(CommandKind::User);
            auto holder = clientContext.AcquireProducerHolder(&userContext, &pBuffer->userCommandMediatorWorkBuffer);
            NN_RESULT_DO(Send<>(
                connection, holder.GetProcessorRef(), timeoutSeconds,
                pBuffer->communicationBuffer, sizeof(pBuffer->communicationBuffer), pCancellable, pSpeedMonitor));
        }
        {
            auto holder = clientContext.AcquireConsumerHolder(&userContext, &pBuffer->userCommandMediatorWorkBuffer);
            NN_RESULT_DO(Receive<>(
                connection, holder.GetProcessorRef(), timeoutSeconds,
                pBuffer->communicationBuffer, sizeof(pBuffer->communicationBuffer), pCancellable, pSpeedMonitor));
        }
        NN_RESULT_SUCCESS;
    }

    // 受信/消費を並列化。
    template <typename ConnectionType, typename ClientContextType, typename UserContextType>
    static Result InvokeUserCommand(const ConnectionType& connection, ClientContextType& clientContext, UserContextType& userContext, int timeoutSeconds,
        migration::detail::ThreadResourceType* pThreadResource, void* workBuffer, size_t workBufferSize, const migration::detail::Cancellable* pCancellable,
        migration::idc::detail::TransferSpeedMonitor* pSpeedMonitor = nullptr) NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(std::alignment_of<InvokeUserCommandBufferForThreaded<ClientContextType>>::value == 1u);
        NN_SDK_REQUIRES_GREATER_EQUAL(timeoutSeconds, 0);
        NN_SDK_REQUIRES_NOT_NULL(pThreadResource);
        NN_SDK_REQUIRES_NOT_NULL(workBuffer);
        NN_SDK_REQUIRES_GREATER_EQUAL(workBufferSize, sizeof(InvokeUserCommandBufferForThreaded<ClientContextType>));
        NN_UNUSED(workBufferSize);
        auto pBuffer = reinterpret_cast<InvokeUserCommandBufferForThreaded<ClientContextType>*>(workBuffer);
        {
            clientContext.SetCommandKind(CommandKind::User);
            auto holder = clientContext.AcquireThreadedProducerHolder(&userContext, pThreadResource, &pBuffer->userCommandMediatorWorkBuffer);
            NN_RESULT_DO(Send<>(
                connection, holder.GetProcessorRef(), timeoutSeconds,
                pBuffer->communicationBuffer, sizeof(pBuffer->communicationBuffer), pCancellable, pSpeedMonitor));
        }
        {
            auto holder = clientContext.AcquireThreadedConsumerHolder(&userContext, pThreadResource, &pBuffer->userCommandMediatorWorkBuffer);
            NN_RESULT_DO(Receive<>(
                connection, holder.GetProcessorRef(), timeoutSeconds,
                pBuffer->communicationBuffer, sizeof(pBuffer->communicationBuffer), pCancellable, pSpeedMonitor));
        }
        NN_RESULT_SUCCESS;
    }

private:

    template <typename ConnectionType, typename ClientContextType>
    static Result Invoke(CommandKind commandKind, const ConnectionType& connection, ClientContextType& clientContext, int timeoutSeconds,
        void* workBuffer, size_t workBufferSize, const migration::detail::Cancellable* pCancellable) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(timeoutSeconds, 0);
        NN_SDK_REQUIRES_NOT_NULL(workBuffer);

        clientContext.SetCommandKind(commandKind);
        {
            auto producer = clientContext.GetProducer();
            NN_RESULT_DO(Send<>(connection, producer, timeoutSeconds, workBuffer, workBufferSize, pCancellable));
        }
        {
            auto consumer = clientContext.GetConsumer();
            NN_RESULT_DO(Receive<>(connection, consumer, timeoutSeconds, workBuffer, workBufferSize, pCancellable));
        }
        NN_RESULT_SUCCESS;
    }
};

}}}
