﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/migration/idc/migration_CommandTypes.h>
#include <nn/migration/detail/migration_Log.h>
#include <nn/os.h>
#include <nn/util/util_FormatString.h>

namespace nn { namespace migration { namespace idc { namespace detail {

// TORIAEZU: memset ではコンパイラの最適化で消えるので独自に定義
//           memset_s が欲しい。
void SecureMemoryZero(Bit8* addr, size_t size) NN_NOEXCEPT;

void GenerateRandomBytes(void* buffer, size_t bufferSize) NN_NOEXCEPT;

#if defined(NN_SDK_BUILD_DEVELOP) || defined(NN_SDK_BUILD_DEBUG)
const char* GetCommandKindString(CommandKind command) NN_NOEXCEPT;
#endif

template <size_t L>
inline void TraceByteArray(const char* prefix, const Bit8(&bytes)[L]) NN_NOEXCEPT
{
#if defined(NN_SDK_BUILD_DEVELOP) || defined(NN_SDK_BUILD_DEBUG)
    char str[L * 2 + 1];
    for( size_t i = 0; i < L; i++ )
    {
        util::SNPrintf(str + i * 2, 3, "%02x", bytes[i]);
    }
    NN_DETAIL_MIGRATION_TRACE("%s%s\n", prefix, str);
#else
    NN_UNUSED(prefix);
    NN_UNUSED(bytes);
#endif
}

template <size_t L>
inline void TraceByteArray(const char* prefix, const void* bytes) NN_NOEXCEPT
{
#if defined(NN_SDK_BUILD_DEVELOP) || defined(NN_SDK_BUILD_DEBUG)
    char str[L * 2 + 1];
    for( size_t i = 0; i < L; i++ )
    {
        util::SNPrintf(str + i * 2, 3, "%02x", reinterpret_cast<const Bit8*>(bytes)[i]);
    }
    NN_DETAIL_MIGRATION_TRACE("%s%s\n", prefix, str);
#else
    NN_UNUSED(prefix);
    NN_UNUSED(bytes);
#endif
}

class TransferSpeedMonitor
{
public:
    TransferSpeedMonitor() NN_NOEXCEPT;
    TransferSpeedMonitor(size_t bytesPerSec, TimeSpan timeSpan) NN_NOEXCEPT;
    void SetLimit(size_t bytesPerSec, TimeSpan timeSpan) NN_NOEXCEPT;
    void Reset() NN_NOEXCEPT;
    void Pause() NN_NOEXCEPT;
    void Resume() NN_NOEXCEPT;

    void Update(size_t transferredByteCount) NN_NOEXCEPT;
    bool IsLowLimitDetected() const NN_NOEXCEPT;

private:
    bool m_IsSet;
    bool m_IsPaused;
    size_t m_BytePerSecLimit;
    TimeSpan m_TrackDuration;
    TimeSpan m_BelowLimitDuration;
    os::Tick m_LastUpdated;

    TimeSpan m_CachedDuration;
    size_t m_CachedTransferredByteCount;
};

}}}}
