﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/migration/idc/migration_Result.h>
#include <nn/migration/detail/migration_Diagnosis.h>
#include <nn/migration/detail/migration_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace migration { namespace detail {

class Sender
{
private:
    const void* m_Data;
    size_t m_DataSize;
    size_t m_Offset;

public:
    Sender() NN_NOEXCEPT
        : m_Data (nullptr)
        , m_DataSize(0)
        , m_Offset(0)
    {
    }
    size_t GetProducableByteSize() const NN_NOEXCEPT
    {
        return m_DataSize;
    }
    template <typename RequestBody>
    void InitializeSender(const RequestBody& requestBody) NN_NOEXCEPT
    {
        m_Data = &requestBody;
        m_DataSize = sizeof(requestBody);
    }
    Result Produce(size_t* pOutActualSize, void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        auto produceSize = std::min(bufferSize, m_DataSize - m_Offset);
        auto dataHead = reinterpret_cast<const void*>(reinterpret_cast<uintptr_t>(m_Data) + m_Offset);
        std::memcpy(buffer, dataHead, produceSize);
        *pOutActualSize = produceSize;
        m_Offset += produceSize;
        NN_RESULT_THROW_UNLESS(m_Offset == m_DataSize, idc::ResultProduceCommandContinue());
        NN_RESULT_SUCCESS;
    }
};

template <typename CommandKind, typename ResultUnexpectedCommand, typename T>
class Receiver
{
    NN_STATIC_ASSERT(std::alignment_of<CommandKind>::value <= 8);

public:
    struct CommandHandler
    {
        NN_ALIGNAS(8) CommandKind command;
        Result(T::*handler)(const void* data, size_t dataSize);
    };

private:
    T& m_Obj;
    CommandHandler* m_Handlers;
    size_t m_HandlerCount;

    util::optional<CommandHandler> m_Received;

protected:
    explicit Receiver(T& obj) NN_NOEXCEPT
        : m_Obj(obj)
        , m_Handlers(nullptr)
        , m_HandlerCount(0)
    {
    }
    void InitializeReceiver(CommandHandler *handlers, const size_t N) NN_NOEXCEPT
    {
#if !defined(NN_SDK_BUILD_RELEASE)
        for (size_t i = 0; i < N; ++ i)
        {
            for (size_t j = 0; j < i; ++ j)
            {
                NN_SDK_ASSERT_NOT_EQUAL(handlers[i].command, handlers[j].command);
            }
            NN_SDK_ASSERT_NOT_NULL(handlers[i].handler);
        }
#endif
        m_Handlers = handlers;
        m_HandlerCount = N;
    }

public:
    Result Consume(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(dataSize > 0);
        if (!m_Received)
        {
            auto command = *reinterpret_cast<const CommandKind*>(data);
            for (size_t i = 0; i < m_HandlerCount; ++ i)
            {
                if (m_Handlers[i].command == command)
                {
                    NN_MIGRATION_DETAIL_TRACE("[Receiver] Received command=%s\n", GetCommandString(m_Handlers[i].command));
                    m_Received = m_Handlers[i];
                    break;
                }
            }
            NN_RESULT_THROW_UNLESS(m_Received, ResultUnexpectedCommand());
        }
        return (m_Obj.*(m_Received->handler))(data, dataSize);
    }
    void SetByteSizeToConsume(size_t) NN_NOEXCEPT
    {
    }
};

template <typename CommandKind, typename T>
class ClientApiBase
    : protected Sender
    , protected Receiver<CommandKind, ResultUnexpectedResponseKind, T>
{
private:
    typedef Receiver<CommandKind, ResultUnexpectedResponseKind, T> ReceiverType;

public:
    using typename ReceiverType::CommandHandler;

protected:
    explicit ClientApiBase(T& obj) NN_NOEXCEPT
        : Sender()
        , ReceiverType(obj)
    {
    }
    template <typename RequestBody, size_t N>
    void Initialize(const RequestBody& requestBody, CommandHandler(&handlers)[N]) NN_NOEXCEPT
    {
        Sender::InitializeSender(requestBody);
        ReceiverType::InitializeReceiver(handlers, N);
    }

public:
    using Sender::GetProducableByteSize;
    using Sender::Produce;
    using ReceiverType::Consume;
    using ReceiverType::SetByteSizeToConsume;
};

template <typename CommandKind, typename T>
class ServerApiBase
    : protected Receiver<CommandKind, ResultUnexpectedRequestKind, T>
{
private:
    typedef Receiver<CommandKind, ResultUnexpectedRequestKind, T> ReceiverType;

    util::optional<Sender> m_Sender;

public:
    using typename ReceiverType::CommandHandler;

protected:
    explicit ServerApiBase(T& obj) NN_NOEXCEPT
        : ReceiverType(obj)
    {
    }
    template <size_t N>
    void Initialize(CommandHandler(&handlers)[N]) NN_NOEXCEPT
    {
        ReceiverType::InitializeReceiver(handlers, N);
    }
    void Initialize(CommandHandler *handlers, const size_t N) NN_NOEXCEPT
    {
        ReceiverType::InitializeReceiver(handlers, N);
    }
    template <typename ResponseBody>
    void SetStaticResponse(const ResponseBody& responseBody) NN_NOEXCEPT
    {
        m_Sender.emplace();
        m_Sender->InitializeSender(responseBody);
    }

public:
    using ReceiverType::Consume;
    using ReceiverType::SetByteSizeToConsume;
    size_t GetProducableByteSize() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Sender);
        return m_Sender->GetProducableByteSize();
    }
    Result Produce(size_t* pOutActualSize, void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Sender);
        return m_Sender->Produce(pOutActualSize, buffer, bufferSize);
    }
};

template <typename CommandKind, typename T>
class StreamingServerApiBase
    : public ServerApiBase<CommandKind, T>
{
private:
    typedef ServerApiBase<CommandKind, T> Base;

    T& m_Obj;

    util::optional<Result(T::*)(size_t*, void*, size_t)> m_StreamOut;
    size_t m_StreamOutSize;

public:
    using typename Base::CommandHandler;

protected:
    using Base::Initialize;

    explicit StreamingServerApiBase(T& obj) NN_NOEXCEPT
        : Base(obj)
        , m_Obj(obj)
    {
    }
    void SetStreamResponse(size_t streamOutSize, Result (T::*streamOut)(size_t*, void*, size_t)) NN_NOEXCEPT
    {
        m_StreamOut = streamOut;
        m_StreamOutSize = streamOutSize;
    }

public:
    size_t GetProducableByteSize() const NN_NOEXCEPT
    {
        if (m_StreamOut)
        {
            return m_StreamOutSize;
        }
        return Base::GetProducableByteSize();
    }
    Result Produce(size_t* pOutActualSize, void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        if (m_StreamOut)
        {
            return (m_Obj.*(*m_StreamOut))(pOutActualSize, buffer, bufferSize);
        }
        return Base::Produce(pOutActualSize, buffer, bufferSize);
    }
};

}}} // ~namespace nn::migration::detail
