﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/migration/detail/migration_Authenticator.h>
#include <nn/migration/detail/migration_IdcImplBase.h>
#include <nn/migration/detail/migration_Diagnosis.h>
#include <nn/migration/detail/migration_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace migration { namespace detail {

enum class BasicCommandKind : Bit8
{
    GetAuthenticationCode = 0x00,
    VerifyAuthenticationDigest = 0x01,
};
inline const char* GetCommandString(BasicCommandKind command) NN_NOEXCEPT
{
    switch (command)
    {
    case BasicCommandKind::GetAuthenticationCode:
        return "BasicCommandKind::GetAuthenticationCode";
    case BasicCommandKind::VerifyAuthenticationDigest:
        return "BasicCommandKind::VerifyAuthenticationDigest";
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

// ---------------------------------------------------------------------------------------------
// GetAuthenticationCode

struct RequestGetAuthenticationCode
{
    Bit8 commandId;
    Bit8 padding[7];

    void Reset() NN_NOEXCEPT
    {
        commandId = static_cast<Bit8>(BasicCommandKind::GetAuthenticationCode);
        std::memset(padding, 0x00, sizeof(padding));
    }
};

struct ResponseGetAuthenticationCode
{
    Bit8 commandId;
    Bit8 padding[7];
    Bit8 data[sizeof(reinterpret_cast<OnetimeToken*>(0)->data)];

    void Reset(const OnetimeToken& ott) NN_NOEXCEPT
    {
        commandId = static_cast<Bit8>(BasicCommandKind::GetAuthenticationCode);
        std::memset(padding, 0x00, sizeof(padding));
        std::memcpy(data, ott.data, sizeof(data));
    }
    void Get(OnetimeToken* pOut) const NN_NOEXCEPT
    {
        std::memcpy(pOut->data, data, sizeof(pOut->data));
    }
};

class GetAuthenticationCodeClientApi final
    : public ClientApiBase<BasicCommandKind, GetAuthenticationCodeClientApi>
{
private:
    typedef GetAuthenticationCodeClientApi This;
    typedef ClientApiBase<BasicCommandKind, GetAuthenticationCodeClientApi> Base;
    typedef RequestGetAuthenticationCode Request;
    typedef ResponseGetAuthenticationCode Response;

    RequestGetAuthenticationCode m_Request;
    util::optional<Response> m_Response;
    Base::CommandHandler m_Handlers[1];

    Result Handler(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(dataSize == sizeof(*m_Response), ResultUnexpectedResponseSize());
        m_Response.emplace();
        std::memcpy(&m_Response.value(), data, dataSize);
        NN_RESULT_SUCCESS;
    }

public:
    GetAuthenticationCodeClientApi() NN_NOEXCEPT
        : Base(*this)
    {
        m_Request.Reset();
        m_Handlers[0].command = BasicCommandKind::GetAuthenticationCode;
        m_Handlers[0].handler = &This::Handler;
        Initialize(m_Request, m_Handlers);
    }
    void GetOnetimeToken(OnetimeToken* pOut) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Response);
        m_Response->Get(pOut);
    }
};

class GetAuthenticationCodeServerApi final
    : public ServerApiBase<BasicCommandKind, GetAuthenticationCodeServerApi>
{
private:
    typedef GetAuthenticationCodeServerApi This;
    typedef ServerApiBase<BasicCommandKind, GetAuthenticationCodeServerApi> Base;
    typedef RequestGetAuthenticationCode Request;
    typedef ResponseGetAuthenticationCode Response;

    const OnetimeToken& m_Ott;
    util::optional<Response> m_Response;
    Base::CommandHandler m_Handlers[1];

    Result Handler(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(dataSize == sizeof(Request), ResultUnexpectedRequestSize());
        NN_UNUSED(data);

        m_Response.emplace();
        m_Response->Reset(m_Ott);
        SetStaticResponse(m_Response.value());
        NN_RESULT_SUCCESS;
    }

public:
    explicit GetAuthenticationCodeServerApi(const OnetimeToken& ott) NN_NOEXCEPT
        : Base(*this)
        , m_Ott(ott)
    {
        m_Handlers[0].command = BasicCommandKind::GetAuthenticationCode;
        m_Handlers[0].handler = &This::Handler;
        Initialize(m_Handlers);
    }
};

// ---------------------------------------------------------------------------------------------
// VerifyAuthenticationDigest

struct RequestVerifyAuthenticationDigest
{
    Bit8 commandId;
    Bit8 padding[7];
    Bit8 data[sizeof(reinterpret_cast<FsKeySeedDigest*>(0)->data)];

    void Reset(const FsKeySeedDigest& digest) NN_NOEXCEPT
    {
        commandId = static_cast<Bit8>(BasicCommandKind::VerifyAuthenticationDigest);
        std::memset(padding, 0x00, sizeof(padding));
        std::memcpy(data, digest.data, sizeof(data));
    }
    void Get(FsKeySeedDigest* pOut) const NN_NOEXCEPT
    {
        std::memcpy(pOut->data, data, sizeof(pOut->data));
    }
};

struct ResponseVerifyAuthenticationDigest
{
    Bit8 commandId;
    Bit8 padding[7];
    Bit8 data[1];

    void Reset(bool isValid) NN_NOEXCEPT
    {
        commandId = static_cast<Bit8>(BasicCommandKind::VerifyAuthenticationDigest);
        std::memset(padding, 0x00, sizeof(padding));
        data[0] = isValid ? 0x01 : 0x00;
    }
    bool Get() const NN_NOEXCEPT
    {
        return data[0] != 0;
    }
};

class VerifyAuthenticationDigestClientApi final
    : public ClientApiBase<BasicCommandKind, VerifyAuthenticationDigestClientApi>
{
private:
    typedef VerifyAuthenticationDigestClientApi This;
    typedef ClientApiBase<BasicCommandKind, VerifyAuthenticationDigestClientApi> Base;
    typedef RequestVerifyAuthenticationDigest Request;
    typedef ResponseVerifyAuthenticationDigest Response;

    RequestVerifyAuthenticationDigest m_Request;
    util::optional<Response> m_Response;
    Base::CommandHandler m_Handlers[1];

    Result Handler(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(dataSize == sizeof(*m_Response), ResultUnexpectedResponseSize());
        m_Response.emplace();
        std::memcpy(&m_Response.value(), data, dataSize);
        NN_RESULT_SUCCESS;
    }

public:
    explicit VerifyAuthenticationDigestClientApi(const FsKeySeedDigest& digest) NN_NOEXCEPT
        : Base(*this)
    {
        m_Request.Reset(digest);
        m_Handlers[0].command = BasicCommandKind::VerifyAuthenticationDigest;
        m_Handlers[0].handler = &This::Handler;
        Initialize(m_Request, m_Handlers);
    }
    Result GetResult() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Response);
        NN_RESULT_THROW_UNLESS(m_Response->Get(), ResultDataUnexpectedFsKeySeedDigest());
        NN_RESULT_SUCCESS;
    }
};

class VerifyAuthenticationDigestServerApi final
    : public ServerApiBase<BasicCommandKind, VerifyAuthenticationDigestServerApi>
{
private:
    typedef VerifyAuthenticationDigestServerApi This;
    typedef ServerApiBase<BasicCommandKind, VerifyAuthenticationDigestServerApi> Base;
    typedef RequestVerifyAuthenticationDigest Request;
    typedef ResponseVerifyAuthenticationDigest Response;

    const FsKeySeedDigest& m_Digest;
    util::optional<Response> m_Response;
    Base::CommandHandler m_Handlers[1];

    bool m_Result;

    Result Handler(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        Request request;
        NN_RESULT_THROW_UNLESS(dataSize == sizeof(request), ResultUnexpectedRequestSize());
        std::memcpy(&request, data, dataSize);

        FsKeySeedDigest received;
        request.Get(&received);
        m_Result = (std::memcmp(m_Digest.data, received.data, sizeof(m_Digest.data)) == 0);

        m_Response.emplace();
        m_Response->Reset(m_Result);
        SetStaticResponse(m_Response.value());
        NN_RESULT_SUCCESS;
    }

public:
    explicit VerifyAuthenticationDigestServerApi(const FsKeySeedDigest& digest) NN_NOEXCEPT
        : Base(*this)
        , m_Digest(digest)
    {
        m_Handlers[0].command = BasicCommandKind::VerifyAuthenticationDigest;
        m_Handlers[0].handler = &This::Handler;
        Initialize(m_Handlers);
    }
    Result GetResult() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Response);
        NN_RESULT_THROW_UNLESS(m_Result, ResultDataUnexpectedFsKeySeedDigest());
        NN_RESULT_SUCCESS;
    }
};

}}} // ~namespace nn::migration::detail
