﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ldn/ldn_PrivateTypes.h>
#include <nn/ldn/ldn_Types.h>

namespace nn { namespace ldn
{
    //! @name 高度な API
    //! @{

   /**
    * @brief         周囲のネットワークを探索します。
    *
    * @param[out]    outBuffer             スキャン結果の出力先です。
    * @param[out]    pOutCount             @a outBuffer に出力されたスキャン結果の数です。
    * @param[in]     bufferCount           @a outBuffer に格納できるスキャン結果の数です。
    * @param[in]     filter                スキャンの条件です
    * @param[in]     channel               スキャン対象のチャンネルです。
    *
    * @retresult
    *  @handleresult{ResultSuccess}
    *  @handleresult{ResultInvalidState}
    *  @handleresult{ResultWifiOff}
    *  @handleresult{ResultSleep}
    *  @handleresult{ResultDeviceOccupied}
    * @endretresult
    *
    * @pre
    * - GetState() != @ref State_None
    * - @a outBuffer != nullptr
    * - @a pOutCount != nullptr
    * - 0 < @a bufferCount
    * - @a channel == 1  || @a channel == 6  || @a channel == 11 ||
    *   @a channel == 36 || @a channel == 40 || @a channel == 44 || @a channel == 48 ||
    *   @a channel == @ref AutoChannel
    *
    * @details
    * Scan 関数とは以下の 2 点が異なります。
    * - ScanFilter に ScanFilterFlag_Bssid を使用できる
    * - 製品環境でも無線チャンネルを指定できる
    */
    Result ScanPrivate(NetworkInfo* outBuffer, int* pOutCount, int bufferCount,
                       const ScanFilter& filter, int channel) NN_NOEXCEPT;

   /**
    * @brief         アクセスポイントとしてネットワークを構築します。
    *
    * @param[in]     network                構築するネットワークの設定情報です。
    * @param[in]     securityConfig         構築するネットワークのセキュリティ設定です。
    * @param[in]     securityParam          構築するネットワークのセキュリティ・パラメータです。
    * @param[in]     user                   ネットワークを構築するユーザの情報です。
    * @param[in]     addressEntryCount      静的に割り当てる IPv4 アドレスの数です。
    * @param[in]     addressEntries         静的に割り当てる IPv4 アドレスです。
    *
    * @retresult
    *  @handleresult{ResultSuccess}
    *  @handleresult{ResultInvalidState}
    *  @handleresult{ResultNodeCountLimitation}
    *  @handleresult{ResultWifiOff}
    *  @handleresult{ResultSleep}
    *  @handleresult{ResultDeviceOccupied}
    * @endretresult
    *
    * @pre
    * - GetState() != @ref State_None
    * - 1 <= @a network.nodeCountMax && @a network.nodeCountMax <= @ref NodeCountMax
    * - @a network.channel == 1 || @a network.channel == 6 || @a network.channel = 11 ||
    *   @a network.channel == 36 || @a network.channel == 40 || @a network.channel == 44 ||
    *   @a network.channel == 48 || @a network.channel == @ref AutoChannel
    * - @a network.intentId.localCommunicationId がアプリケーション管理データに登録されていること
    * - @a security.securityMode == @ref SecurityMode_Debug ||
    *   @a security.securityMode == @ref SecurityMode_Product
    * - @ref PassphraseSizeMin <= @a security.passphraseSize &&
    *   @a security.passphraseSize <= @ref PassphraseSizeMax
    * -  0 <= @a addressEntryCount && @a addressEntryCount <= @ref AddressEntryCountMax
    * - addressEntryCount == 0 || addressEntries != nullptr
    *
    * @post
    * - GetState() == @ref State_AccessPointCreated
    *
    * @details
    * アクセスポイントとしてネットワークを構築し、周囲のノードから探索・接続できる状態にします。
    * @ref CreateNetwork 関数よりも詳細な設定が可能です。
    * @ref State_AccessPoint 以外の状態でこの関数を実行すると @ref ResultInvalidState を返します。
    *
    * @a network, @a securityConfig, @a user の指定については @ref CreateNetwork 関数と同様です。
    * @ref NetworkConfig, @ref SecurityConfig, @ref UserConfig のリファレンスを参照してください。
    * 通常、ノードの最大接続台数としては @ref NodeCountMax 以下の数値を自由に設定できますが、
    * 特定の条件下ではより厳しい制限を課されることがあります。
    * その制限を越える最大接続台数を指定すると、この関数は
    * @ref ResultNodeCountLimitation を返して失敗します。
    * ノードの最大接続台数が制限される条件については LDN ライブラリのマニュアルを参照してください。
    *
    * @a securityParam は一度ネットワークを構築した後、
    * アクセスポイントやステーションといった役割の交代や、再起動後の合流が必要な場合に使用します。
    * 詳細については @ref SecurityParameter と
    * @ref GetSecurityParameter 関数のリファレンスを参照してください。
    * 全て 0 で埋められた @ref SecurityParameter を引数に与えた場合の挙動は、
    * @a securityParam を引数にもたない @ref CreateNetwork 関数と同じです。
    *
    * @a addressEntryCount と @a addressEntries は静的 IPv4 アドレス割当に使用します。
    * 詳細については @ref AddressEntry のリファレンスを参照してください。
    * 静的 IPv4 アドレス割当が不要な場合には @a addressEntryCount に 0 を、
    * @a addressEntries に nullptr を指定します。
    *
    * ネットワークの構築には数百ミリ秒程度必要で、
    * その間この関数は処理を返しませんので注意してください。
    */
    Result CreateNetworkPrivate(
        const NetworkConfig&            network,
        const SecurityConfig&           securityConfig,
        const SecurityParameter&        securityParam,
        const UserConfig&               user,
        int                             addressEntryCount,
        const AddressEntry*             addressEntries
    ) NN_NOEXCEPT;

   /**
    * @brief         ステーションとして指定されたネットワークに接続します。
    *
    * @param[in]     network                接続先のネットワークの情報です。
    * @param[in]     securityConfig         接続に必要なセキュリティパラメータです。
    * @param[in]     securityParam          接続先のネットワークのセキュリティ・パラメータです。
    * @param[in]     user                   ネットワークに参加するユーザの情報です。
    * @param[in]     version                ローカル通信バージョンです。
    * @param[in]     option                 挙動を指定するオプションです。
    *
    * @retresult
    *  @handleresult{ResultSuccess}
    *  @handleresult{ResultInvalidState}
    *  @handleresult{ResultNodeCountLimitation}
    *  @handleresult{ResultWifiOff}
    *  @handleresult{ResultSleep}
    *  @handleresult{ResultDeviceOccupied}
    *  @handleresult{ResultNetworkNotFound}
    *  @handleresult{ResultConnectionTimeout}
    *  @handleresult{ResultConnectionRejected}
    *  @handleresult{ResultLowerVersion}
    *  @handleresult{ResultHigherVersion}
    * @endretresult
    *
    * @pre
    * - GetState() != @ref State_None
    * - @a network.channel == 1 || @a network.channel == 6 || @a network.channel = 11 ||
    *   @a network.channel == 36 || @a network.channel == 40 || @a network.channel == 44 ||
    *   @a network.channel == 48
    * - @a network.intentId.localCommunicationId がアプリケーション管理データに登録されていること
    * - @ref LocalCommunicationVersionMin <= @a network.localCommunicationVersion &&
    *   @a network.localCommunicationVersion <= @ref LocalCommunicationVersionMax
    * - @a security.securityMode == @ref SecurityMode_Debug ||
    *   @a security.securityMode == @ref SecurityMode_Product
    *
    * @post
    * - GetState() == @ref State_StationConnected
    *
    * @details
    * 指定されたネットワークにステーションとして接続します。
    * @ref Connect 関数よりも詳細な設定が可能です。
    * @ref State_Station 以外の状態でこの関数を実行すると @ref ResultInvalidState を返します。
    *
    * @a network, @a securityConfig, @a userの設定方法については
    * @ref NetworkConfig, @ref SecurityConfig, @ref UserConfig のリファレンスを参照してください。
    * ただし、securityConfig.mode に @ref SecurityMode_Any を指定できないことに注意してください。
    * @ref SecurityMode_Any は @ref Connect 関数でのみ使用できる定数です。
    * また、LDN ライブラリでは特定の条件下において、
    * ノードの最大接続台数が大きいネットワークに接続しようとすると、
    * @ref ResultNodeCountLimitation を返して失敗します。
    * ノードの最大接続台数が制限される条件については LDN ライブラリのマニュアルを参照してください。
    *
    * @a securityParameter は一度ネットワークを構築した後、
    * アクセスポイントやステーションといった役割の交代や、再起動後の合流が必要な場合に使用します。
    * 詳細については @ref SecurityParameter と
    * @ref GetSecurityParameter 関数のリファレンスを参照してください。
    *
    * version はローカル通信バージョンです。
    * アプリケーションへのパッチの適用などで通信の仕様が変わる場合に、
    * ローカル通信バージョンをインクリメントしておくことで、通信の互換性がないことを表明できます。
    * ステーションはローカル通信バージョンが異なるアクセスポイントには接続できず、
    * @ref Connect 関数がアクセスポイントと自身のローカル通信バージョンを比較して
    * @ref ResultLowerVersion と @ref ResultHigherVersion のいずれかのエラーを返します。
    * また、このとき @a option に @ref ConnectOption_ShowErrorMessage が指定されていれば、
    * 画面にエラーメッセージを表示します。
    * ローカル通信バージョンには @ref LocalCommunicationVersionMin 以上、
    * @ref LocalCommunicationVersionMax 以下の値を設定してください。
    */
    Result ConnectPrivate(
        const NetworkConfig&        network,
        const SecurityConfig&       securityConfig,
        const SecurityParameter&    securityParam,
        const UserConfig&           user,
        int                         version,
        ConnectOption               option
    ) NN_NOEXCEPT;

   /**
    * @brief         ステーションとして指定されたネットワークに接続します。
    *
    * @param[in]     network                接続先のネットワークの情報です。
    * @param[in]     securityConfig         接続に必要なセキュリティパラメータです。
    * @param[in]     securityParam          接続先のネットワークのセキュリティ・パラメータです。
    * @param[in]     user                   ネットワークに参加するユーザの情報です。
    * @param[in]     version                ローカル通信バージョンです。
    *
    * @retresult
    *  @handleresult{ResultSuccess}
    *  @handleresult{ResultInvalidState}
    *  @handleresult{ResultNodeCountLimitation}
    *  @handleresult{ResultWifiOff}
    *  @handleresult{ResultSleep}
    *  @handleresult{ResultDeviceOccupied}
    *  @handleresult{ResultNetworkNotFound}
    *  @handleresult{ResultConnectionTimeout}
    *  @handleresult{ResultConnectionRejected}
    *  @handleresult{ResultLowerVersion}
    *  @handleresult{ResultHigherVersion}
    * @endretresult
    *
    * @pre
    * - GetState() != @ref State_None
    * - @a network.channel == 1 || @a network.channel == 6 || @a network.channel = 11 ||
    *   @a network.channel == 36 || @a network.channel == 40 || @a network.channel == 44 ||
    *   @a network.channel == 48
    * - @a network.intentId.localCommunicationId がアプリケーション管理データに登録されていること
    * - @ref LocalCommunicationVersionMin <= @a network.localCommunicationVersion &&
    *   @a network.localCommunicationVersion <= @ref LocalCommunicationVersionMax
    * - @a security.securityMode == @ref SecurityMode_Debug ||
    *   @a security.securityMode == @ref SecurityMode_Product
    *
    * @post
    * - GetState() == @ref State_StationConnected
    *
    * @details
    * この関数は option に @ref ConnectOption_Default
    * が指定されたと仮定してネットワークへの接続を試行します。
    * 詳細については option を指定できる @ref ConnectPrivate 関数のリファレンスを参照してください。
    *
    * @note
    * この関数は将来の SDK で仕様の変更が予定されています。
    * 仕様変更後は option に ConnectOption_None が指定された扱いになります。
    * 新しい SDK に移行する予定がある場合、仕様の変更後も問題なく動作するように
    * ResultIncompatibleVersion に対するエラーハンドリングを追加しておくか、
    * 引数に option を指定できる @ref ConnectPrivate 関数を使用してください。
    */
    NN_DEPRECATED Result ConnectPrivate(
        const NetworkConfig&        network,
        const SecurityConfig&       securityConfig,
        const SecurityParameter&    securityParam,
        const UserConfig&           user,
        int                         version
    ) NN_NOEXCEPT;

   /**
    * @brief        現在参加しているネットワークのセキュリティ・パラメータを取得します。
    *
    * @param[out]   pOutSecurityParameter  セキュリティ・パラメータの出力先です。
    *
    * @retresult
    *  @handleresult{ResultSuccess}
    *  @handleresult{ResultInvalidState}
    *  @handleresult{ResultWifiOff}
    *  @handleresult{ResultSleep}
    *  @handleresult{ResultDeviceOccupied}
    * @endretresult
    *
    * @pre
    * - GetState() != @ref State_None
    *
    * @details
    * 現在参加しているネットワークのセキュリティ・パラメータを取得します。
    * セキュリティ・パラメータの詳細については
    * @ref SecurityParameter のリファレンスを参照してください。
    *
    * アクセスポイントは @ref CreateNetwork 関数でネットワークを構築した後、
    * ステーションは @ref Connect 関数でネットワークに接続した後にこの関数を実行できます。
    * @ref State_StationConnected, @ref State_AccessPointCreated
    * 以外の状態でこの関数を実行した場合は @ref ResultInvalidState を返します。
    */
    Result GetSecurityParameter(SecurityParameter* pOutSecurityParameter) NN_NOEXCEPT;

   /**
    * @brief        現在参加しているネットワークの設定情報を取得します。
    *
    * @param[out]   pOutNetworkConfig       ネットワークの設定情報の出力先です。
    *
    * @retresult
    *  @handleresult{ResultSuccess}
    *  @handleresult{ResultInvalidState}
    *  @handleresult{ResultWifiOff}
    *  @handleresult{ResultSleep}
    *  @handleresult{ResultDeviceOccupied}
    * @endretresult
    *
    * @pre
    * - GetState() != @ref State_None
    *
    * @details
    * 現在参加しているネットワークの設定情報を取得します。
    * ネットワーク設定の詳細については @ref NetworkConfig のリファレンスを参照してください。
    *
    * アクセスポイントは @ref CreateNetwork 関数でネットワークを構築した後、
    * ステーションは @ref Connect 関数でネットワークに接続した後にこの関数を実行できます。
    * @ref State_StationConnected, @ref State_AccessPointCreated
    * 以外の状態でこの関数を実行した場合は @ref ResultInvalidState を返します。
    */
    Result GetNetworkConfig(NetworkConfig* pOutNetworkConfig) NN_NOEXCEPT;

    /**
     * @brief         ステーションの接続を制限するフィルタにエントリを追加します。
     *
     * @param[in]     station             対象のステーションの MAC アドレスです。
     *
     * @retresult
     *  @handleresult{ResultSuccess}
     *  @handleresult{ResultInvalidState}
     *  @handleresult{ResultWifiOff}
     *  @handleresult{ResultSleep}
     *  @handleresult{ResultDeviceOccupied}
     * @endretresult
     *
     * @pre
     * - GetState() != @ref State_None
     *
     * @details
     * この関数は、対象となるステーションを MAC アドレスで指定できる以外は、
     * @ref AddAcceptFilterEntry(const NodeInfo& station) 関数と同じです。
     */
    Result AddAcceptFilterEntry(MacAddress station) NN_NOEXCEPT;

    //! @}

}} // end of namespace nn::ldn
