﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <type_traits>
#include <nn/nn_Common.h>

namespace nn { namespace ldn { namespace detail { namespace impl
{
    template <size_t N>
    struct BitTypeByteOrderReverser;

    template <>
    struct BitTypeByteOrderReverser<1>
    {
        typedef Bit8 BitType;
        static BitType Calc(BitType value) NN_NOEXCEPT
        {
            return value;
        }
    };

    template <>
    struct BitTypeByteOrderReverser<2>
    {
        typedef Bit16 BitType;
        static BitType Calc(BitType value) NN_NOEXCEPT
        {
            BitType x = value;
            x = ((x & UINT16_C(0x00FF)) << 8) | ((x & UINT16_C(0xFF00)) >> 8);
            return x;
        }
    };

    template <>
    struct BitTypeByteOrderReverser<4>
    {
        typedef Bit32 BitType;
        static BitType Calc(BitType value) NN_NOEXCEPT
        {
            BitType x = value;
            x = ((x & UINT32_C(0x00FF00FF)) <<  8) | ((x & UINT32_C(0xFF00FF00)) >>  8);
            x = ((x & UINT32_C(0x0000FFFF)) << 16) | ((x & UINT32_C(0xFFFF0000)) >> 16);
            return x;
        }
    };

    template <>
    struct BitTypeByteOrderReverser<8>
    {
        typedef Bit64 BitType;
        static BitType Calc(BitType value) NN_NOEXCEPT
        {
            BitType x = value;
            x = ((x & UINT64_C(0x00FF00FF00FF00FF)) <<  8) |
                ((x & UINT64_C(0xFF00FF00FF00FF00)) >>  8);
            x = ((x & UINT64_C(0x0000FFFF0000FFFF)) << 16) |
                ((x & UINT64_C(0xFFFF0000FFFF0000)) >> 16);
            x = ((x & UINT64_C(0x00000000FFFFFFFF)) << 32) |
                ((x & UINT64_C(0xFFFFFFFF00000000)) >> 32);
            return x;
        }
    };

    template <typename T, typename Enable = void>
    struct IntegralTypeByteOrderReverser;

    template <typename T>
    struct IntegralTypeByteOrderReverser<
        T, typename std::enable_if<std::is_integral<T>::value>::type>
    {
        static T Calc(T value) NN_NOEXCEPT
        {
            return static_cast<T>(BitTypeByteOrderReverser<sizeof(T)>::Calc(
                static_cast<typename BitTypeByteOrderReverser<sizeof(T)>::BitType>(value)));
        }
    };

    // T が integral の場合には単純にバイト列を反転します。
    template <typename T>
    inline T ReverseByteOrderDispatcher(T value, std::true_type) NN_NOEXCEPT
    {
        return IntegralTypeByteOrderReverser<T>::Calc(value);
    }

    // T が integral でない場合にはそれぞれの型を引数にとる ReverseByteOrderImpl() に任せます。
    template <typename T>
    inline T ReverseByteOrderDispatcher(T value, std::false_type) NN_NOEXCEPT
    {
        return ReverseByteOrderImpl(value);
    }

}}}} // namespace nn::ldn::detail::impl

namespace nn { namespace ldn { namespace detail
{

    /**
     * @brief         バイトオーダを反転します。
     *
     * @tparam        T         対象の値の型です。
     *
     * @param[in]     value     対象の値です。
     *
     * @return        バイトオーダを反転した値です。
     */
    template <typename T>
    inline T ReverseByteOrder(T value) NN_NOEXCEPT
    {
        return impl::ReverseByteOrderDispatcher(value, std::is_integral<T>());
    }

    /**
     * @brief         ホストバイトオーダからネットワークバイトオーダに変換します。
     *
     * @tparam        T         対象の値の型です。
     *
     * @param[in]     value     対象の値です。
     *
     * @return        バイトオーダを変換した値です。
     */
    template <typename T>
    inline T ConvertToNetworkByteOrder(T value) NN_NOEXCEPT
    {
        #if defined(NN_BUILD_CONFIG_ENDIAN_LITTLE)
            return ReverseByteOrder(value);
        #elif defined(NN_BUILD_CONFIG_ENDIAN_BIG)
            return value;
        #else
            #error "unknown byte order"
        #endif
    }

    /**
     * @brief         ネットワークバイトオーダからホストバイトオーダに変換します。
     *
     * @tparam        T         対象の値の型です。
     *
     * @param[in]     value     対象の値です。
     *
     * @return        バイトオーダを変換した値です。
     */
    template <typename T>
    inline T ConvertToHostByteOrder(T value) NN_NOEXCEPT
    {
        return ConvertToNetworkByteOrder(value);
    }

}}} // end of namespace nn::ldn::detail
