﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/ldn/detail/NetworkInterface/ldn_FrameQueue.h>
#include <nn/ldn/detail/NetworkInterface/ldn_NetworkInterface.h>
#include <nn/ldn/detail/NetworkInterface/ldn_ProtocolId.h>
#include <nn/os/os_EventTypes.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/os/os_Mutex.h>

namespace nn { namespace ldn { namespace detail
{
    /**
     * @brief       プロトコルの最大数です。
     */
    const int ProtocolCountMax = 4;

    /**
     * @brief       フレームの受信インタフェースです。
     */
    class IFrameReceiver
    {
    public:

        /**
         * @brief       デストラクタです。
         */
        virtual ~IFrameReceiver()
        {
        }

        /**
         * @brief       プロトコルを登録します。
         * @param[in]   protocol    対象のプロトコルです。
         * @param[in]   pEvent      受信の通知に使用するイベントです。
         * @param[in]   pQueue      データの受信に使用するキューです。
         */
        virtual void Register(
            ProtocolId protocol, nn::os::EventType* pEvent, FrameQueue* pQueue) NN_NOEXCEPT = 0;

        /**
         * @brief       プロトコルの登録を解除します。
         * @param[in]   protocol    対象のプロトコルです。
         */
        virtual void Unregister(ProtocolId protocol) NN_NOEXCEPT = 0;
    };

    /**
     * @brief       フレームの受信を担当するクラスです。
     */
    class FrameReceiver : public IFrameReceiver
    {
    public:

        /**
         * @brief       スコープ内限定で受信を有効化します。
         */
        class ScopedInitializer
        {
        public:

            ScopedInitializer(
                FrameReceiver* pReceiver, INetworkInterface* pInterface, void* buffer,
                size_t bufferSize, size_t frameSizeMax, int protocolCount) NN_NOEXCEPT
                : m_pReceiver(pReceiver)
            {
                m_pReceiver->Initialize(
                    pInterface, buffer, bufferSize, frameSizeMax, protocolCount);
            };

            ~ScopedInitializer() NN_NOEXCEPT
            {
                if (m_pReceiver != nullptr)
                {
                    m_pReceiver->Finalize();
                }
            }

            void Detach() NN_NOEXCEPT
            {
                m_pReceiver = nullptr;
            }

        private:

            FrameReceiver* m_pReceiver;
        };

        /**
         * @brief       コンストラクタです。
         */
        FrameReceiver() NN_NOEXCEPT;

        /**
         * @brief       デストラクタです。
         */
        ~FrameReceiver() NN_NOEXCEPT;

        /**
         * @brief       初期化に必要なバッファサイズを計算します。
         * @param[in]   frameSizeMax    受信フレームの最大サイズです。
         * @param[in]   protocolCount   プロトコルの数です。
         * @return      バッファサイズです。
         */
        static size_t GetRequiredBufferSize(size_t frameSizeMax, int protocolCount) NN_NOEXCEPT;

        /**
         * @brief       フレームの受信を開始します。
         * @param[in]   pInterface      ネットワークインタフェースです。
         * @param[in]   buffer          初期化に使用するバッファです。
         * @param[in]   bufferSize      初期化に使用するバッファのバイトサイズです。
         * @param[in]   frameSizeMax    受信フレームの最大サイズです。
         * @param[in]   protocolCount   プロトコルの数です。
         */
        void Initialize(
            INetworkInterface* pInterface, void* buffer, size_t bufferSize,
            size_t frameSizeMax, int protocolCount) NN_NOEXCEPT;

        /**
         * @brief       フレームの受信を停止します。
         */
        void Finalize() NN_NOEXCEPT;

        /**
         * @brief       プロトコルを登録します。
         * @param[in]   protocol    対象のプロトコルです。
         * @param[in]   pEvent      受信の通知に使用するイベントです。
         * @param[in]   pQueue      データの受信に使用するキューです。
         */
        virtual void Register(
            ProtocolId protocol, nn::os::EventType* pEvent,
            FrameQueue* pQueue) NN_NOEXCEPT NN_OVERRIDE;

        /**
         * @brief       プロトコルの登録を解除します。
         * @param[in]   protocol    対象のプロトコルです。
         */
        virtual void Unregister(ProtocolId protocol) NN_NOEXCEPT NN_OVERRIDE;

    private:

        static void ReceiveThread(void* pArg) NN_NOEXCEPT;

        struct QueueInfo
        {
            FrameQueue* pQueue;
            nn::os::EventType* pEvent;
            ProtocolId protocol;
        };

        QueueInfo m_QueueList[ProtocolCountMax];
        INetworkInterface* m_pNetworkInterface;
        void* m_ReceiveBuffer;
        size_t m_ReceiveBufferSize;
        bool m_IsRunning;
        nn::os::ThreadType m_ReceiveThread;
        nn::os::Mutex m_ProtocolMutex;
    };

}}} // end of namespace nn::ldn::detail
