﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/ldn/ldn_Types.h>

namespace nn { namespace ldn { namespace detail
{
    /**
     * @brief       認証サーバがクライアントに送信するデータです。
     */
    struct AuthenticationServerData
    {
        //  予約領域です。
        Bit8 reserved[64];
    };

    /**
     * @brief       認証クライアントがサーバに送信するデータです。
     */
    struct AuthenticationClientData
    {
        //! クライアントのユーザ名です。
        char userName[UserNameBytesMax];

        //! ローカル通信バージョンです。
        int16_t localCommunicationVersion;

        //  予約領域です。
        Bit8 reserved[30];
    };

    /**
     * @brief       認証サーバがクライアントに送信するデータを生成します。
     */
    class AuthenticationServerDataBuilder
    {
    public:

        /**
         * @brief       コンストラクタです。
         */
        AuthenticationServerDataBuilder() NN_NOEXCEPT;

        /**
         * @brief       ネットワーク情報を設定します。
         * @param[in]   ldn         ネットワーク情報です。
         */
        void SetLdnNetworkInfo(const LdnNetworkInfo& ldn) NN_NOEXCEPT;

        /**
         * @brief       生成されたデータを取得します。
         * @return      生成されたデータです。
         */
        const AuthenticationServerData& GetData() const NN_NOEXCEPT;

    private:

        AuthenticationServerData m_Data;
    };

    /**
     * @brief       認証クライアントがサーバに送信するデータを生成します。
     */
    class AuthenticationClientDataBuilder
    {
    public:

        /**
         * @brief       コンストラクタです。
         */
        AuthenticationClientDataBuilder() NN_NOEXCEPT;

        /**
         * @brief       ユーザ情報を設定します。
         * @param[in]   user        ユーザ情報です。
         */
        void SetUserConfig(const UserConfig& user) NN_NOEXCEPT;

        /**
         * @brief       ローカル通信バージョンを設定します。
         * @param[in]   version     ローカル通信バージョンです。
         */
        void SetLocalCommunicationVersion(int version) NN_NOEXCEPT;

        /**
         * @brief       生成されたデータを取得します。
         * @return      生成されたデータです。
         */
        const AuthenticationClientData& GetData() const NN_NOEXCEPT;

    private:

        AuthenticationClientData m_Data;
    };

    class AuthenticationClientDataParser
    {
    public:

        /**
         * @brief       コンストラクタです。
         * @param[in]   data        受信データです。
         * @param[in]   dataSize    受信データのバイトサイズです。
         */
        AuthenticationClientDataParser(const void* data, size_t dataSize) NN_NOEXCEPT;

        /**
         * @brief       ユーザ情報を取得します。
         * @return      ユーザ情報です。
         */
        const UserConfig& GetUserConfig() const NN_NOEXCEPT;

        /**
         * @brief       ローカル通信バージョンを取得します。
         * @return      ローカル通信バージョンです。
         */
        int GetLocalCommunicationVersion() const NN_NOEXCEPT;

        /**
         * @brief       受信データの正当性を確認します。
         * @return      受信データの内容に問題がなければ true です。
         */
        bool IsValid() const NN_NOEXCEPT;

    private:

        UserConfig m_UserConfig;
        int16_t m_LocalCommunicationVersion;
        bool m_IsValid;
    };

}}} // namespace nn::ldn::detail
