﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/ldn/detail/Advertise/ldn_Advertise.h>

namespace nn { namespace ldn { namespace detail
{
    /**
     * @brief         Advertise の解析結果です。
     */
    enum AdvertiseParserResult
    {
        //! 新しい Advertise データを受信しました。
        AdvertiseParserResult_Success,

        //! 前回と同じ Advertise データです。
        AdvertiseParserResult_NotUpdated,

        //! 無効な Advertise データです。
        AdvertiseParserResult_InvalidAdvertise,

        //! 互換性の無いプロトコル・バージョンです。
        AdvertiseParserResult_IncompatibleVersion,

        //! 前回の Advertise と矛盾する内容を含んでいます。
        AdvertiseParserResult_InconsistentUpdate,

        //! 署名が一致しません。
        AdvertiseParserResult_SignVerificationError,
    };

    /**
     * @brief         Advertise で受信したデータを解析するインタフェースです。
     */
    class IAdvertiseParser
    {
    public:

        /**
         * @brief         デストラクタです。
         */
        virtual ~IAdvertiseParser() NN_NOEXCEPT
        {
        }

        /**
         * @brief         Advertise データを解析します。
         *
         * @param[in]     data          受信した Advertise データです。
         * @param[in]     dataSize      受信した Advertise データのサイズです。
         *
         * @return        処理の結果です。
         *
         * @pre
         *                - data != nullptr
         */
        virtual AdvertiseParserResult Parse(const void* data, size_t dataSize) NN_NOEXCEPT = 0;

        /**
         * @brief         NetworkId を取得します。
         *
         * @return        NetworkId です。
         *
         * @pre
         *                - 一度でも Parse() に成功していること
         */
        virtual const NetworkId GetNetworkId() const NN_NOEXCEPT = 0;

        /**
         * @brief         プロトコル・バージョンを取得します。
         *
         * @return        プロトコル・バージョンです。
         *
         * @pre
         *                - 一度でも Parse() に成功していること
         */
        virtual const Version GetVersion() const NN_NOEXCEPT = 0;

        /**
         * @brief         Advertise に含まれていたデータのバイトサイズを取得します。
         *
         * @return        Advertise に含まれていたデータのバイトサイズです。
         *
         * @pre
         *                - 一度でも Parse() に成功していること
         */
        virtual size_t GetDataSize() const NN_NOEXCEPT = 0;

        /**
         * @brief         Advertise に含まれていたデータを取得します。
         *
         * @return        Advertise に含まれていたデータです。
         *
         * @pre
         *                - 一度でも Parse() に成功していること
         *
         * @details
         *                - ここで受信したデータは次に Parse() が呼ばれるまで有効です。
         */
        virtual const void* GetData() const NN_NOEXCEPT = 0;
    };

}}} // namespace nn::ldn::detail
