﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

 /**
 * @file
 * @brief       ローカルコンテンツ配信ライブラリの公開デバッグ API の宣言
 */

#include <nn/nn_Common.h>
#include <nn/err/err_ErrorContext.h>

namespace nn { namespace lcs
{
    /**
    * @brief       LCS ライブラリのメジャーバージョンを変更します。
    *
    * @param[in]   major                LCS ライブラリのメジャーバージョンです。
    *
    * @details
    * LCS ライブラリのメジャーバージョンを変更します。
    *
    * この API は LCS ライブラリの状態に依らず使用することができますが、
    * 変更した値が反映されるのは Initialize() を使用して LCS ライブラリが初期化されるときです。
    * 初期化後に変更しても即時反映されませんので注意してください。
    * 設定した値はアプリケーションが終了するまで保持されます。
    * メジャーバージョンを変更すると、LCS ライブラリの互換性がなくなり、
    * Scan() で発見できなくなります。
    * メジャーバージョンが異なるホストに JoinSession() を行ったときは、
    * ResultLowerVersion()、もしくは、ResultHigherVersion() が返ります。
    *
    * @a major には 0 から 15 までの値を与えてください。
    */
    void SetMajorVersion(int major) NN_NOEXCEPT;

    /**
    * @brief       LCS ライブラリのマイナーバージョンを変更します。
    *
    * @param[in]   minor                LCS ライブラリのマイナーバージョンです。
    *
    * @details
    * LCS ライブラリのマイナーバージョンを変更します。
    *
    * この API は LCS ライブラリの状態に依らず使用することができますが、
    * 変更した値が反映されるのは Initialize() を使用して LCS ライブラリが初期化されるときです。
    * 初期化後に変更しても即時反映されませんので注意してください。
    * 設定した値はアプリケーションが終了するまで保持されます。
    * マイナーバージョンを変更しても、LCS ライブラリの互換性は保たれます。
    *
    * @a minor には 0 から 15 までの値を与えてください。
    */
    void SetMinorVersion(int minor) NN_NOEXCEPT;

    /**
    * @brief       LCS ライブラリのメジャーバージョンを取得します。
    *
    * @return      LCS ライブラリのメジャーバージョンです。
    *
    * @details
    * LCS ライブラリのメジャーバージョンを取得します。
    *
    * この API は LCS ライブラリの状態に依らず使用することができます。
    * SetMajorVersion() で設定した値を取得することができます。
    * SetMajorVersion() で設定していないときは LCS ライブラリで設定されている値が取得できます。
    * GetMajorVersion() で取得できる値は現在設定されている値で、
    * 実際に LCS ライブラリで使用されている値とは異なる場合があることに注意してください。
    *
    */
    int GetMajorVersion() NN_NOEXCEPT;

    /**
    * @brief       LCS ライブラリのマイナーバージョンを取得します。
    *
    * @return      LCS ライブラリのマイナーバージョンです。
    *
    * @details
    * LCS ライブラリのマイナーバージョンを取得します。
    *
    * この API は LCS ライブラリの状態に依らず使用することができます。
    * SetMinorVersion() で設定した値を取得することができます。
    * SetMinorVersion() で設定していないときは LCS ライブラリで設定されている値が取得できます。
    * GetMinorVersion() で取得できる値は現在設定されている値で、
    * 実際に LCS ライブラリで使用されている値とは異なる場合があることに注意してください。
    *
    */
    int GetMinorVersion() NN_NOEXCEPT;

    /**
    * @brief       ErrorContext を保存します。
    *
    * @param[in]  errorContex               ErrorContext です。
    *
    * @details
    * nn::ns::AsyncResult から得られる ErrorContext を保存します。
    *
    * この API は LCS ライブラリの状態に依らず使用することができます。
    * 保存した値は Initialize() や ShowError() を使用するとクリアされます。
    */
    void SetErrorContext(const nn::err::ErrorContext& errorContex) NN_NOEXCEPT;

    /**
    * @brief       ErrorContext を取得します。
    *
    * @param[out]  pOutErrorContex          ErrorContext の出力先です。
    *
    * @pre
    * - @a pOutErrorContex != nullptr
    *
    * @details
    * nn::ns::AsyncResult から得られる ErrorContext を取得します。
    *
    * この API は LCS ライブラリの状態に依らず使用することができます。
    * ErrorContext は一度取得するとクリアされます。
    * また、Initialize() や ShowError() を使用した場合もクリアされます。
    * nn::ns::AsyncResult でエラーが発生していない状態で使用すると
    * 0 クリアされた値を取得します。
    */
    void GetErrorContext(nn::err::ErrorContext* pOutErrorContex) NN_NOEXCEPT;


}} // namespace nn::lcs
