﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/lcs/lcs_Types.h>
#include <nn/lcs/detail/lcs_NetworkInfo.h>
#include <nn/lcs/detail/lcs_Util.h>
#include <nn/lcs/detail/lcs_Version.h>
#include <nn/os/os_Mutex.h>

namespace nn { namespace lcs { namespace detail
{

    enum NodeInnerState
    {
        NodeInnerState_Temporary,
        NodeInnerState_Connected,
        NodeInnerState_Suspending,
        NodeInnerState_Transferring,
        NodeInnerState_Rebooting,
        NodeInnerState_Rejoining,
    };

    struct RequireDownload
    {
        bool isRequire;
        uint32_t index;
        Ipv4Address formAddress;
    };

    struct ShareApplicationInfo
    {
        bool hasApp;
        RequireDownload requireAppDownload;
        ApplicationInfo info;
    };

    struct NodeDetailInfo
    {
        NodeInfo nodeInfo;
        MacAddress macAddress;
        Ipv4Address ipAddress;
        Version version;
        RequireDownload requireSystemDownload;
        SystemDeliveryInfo sysInfo;
        ShareApplicationInfo shareApp[SharableContentsCountMax];
        bool isCanUpdate;
        bool isReceive;
        NodeInnerState state;
        int socket;
        int downloadedCount;
        int64_t leaveTime;
    };

    struct NodeDetailProgress
    {
        uint32_t index;
        NodeProgress progress;
    };

    struct RejectNodeInfo
    {
        uint32_t index;
        ContentsShareFailureReason reason;
    };


    class NodeManager
    {

    public:
        NodeManager() NN_NOEXCEPT;

        void Initialize(int clientCountMax) NN_NOEXCEPT;

        void Finalize() NN_NOEXCEPT;

        uint32_t AddNode(
            Ipv4Address address, const MacAddress& macAddress, uint32_t index) NN_NOEXCEPT;

        bool AddNodeSocket(int socket, Ipv4Address address) NN_NOEXCEPT;

        bool DeleteNodeSocket(uint32_t index) NN_NOEXCEPT;

        bool AddNodeDetail(const NodeDetailInfo& detail) NN_NOEXCEPT;

        void GetNodeInfo(int* pOutNodeCount, NodeInfo* outInfo, int infoCount) NN_NOEXCEPT;

        void GetConnectedNodeInfo(int* pOutNodeCount, NodeInfo* outInfo, int infoCount) NN_NOEXCEPT;

        Result GetNodeDetailInfo(NodeDetailInfo* pOutInfo, uint32_t index) NN_NOEXCEPT;

        uint32_t GetNodeIndex(Ipv4Address address) NN_NOEXCEPT;

        int GetNodeSocket(uint32_t index) NN_NOEXCEPT;

        bool SetLeaveTime(int64_t time, uint32_t index) NN_NOEXCEPT;

        int64_t GetLeaveTime(uint32_t index) NN_NOEXCEPT;

        Result RemoveNode(uint32_t index) NN_NOEXCEPT;

        void ClearNodes() NN_NOEXCEPT;

        int GetNodeCount() NN_NOEXCEPT;

        int GetConnectedNodeCount() NN_NOEXCEPT;

        bool IsNodeCountLimited() NN_NOEXCEPT;

        Result SetInnerState(NodeInnerState state, uint32_t index) NN_NOEXCEPT;

        bool IsInnerState(uint32_t index, NodeInnerState state) NN_NOEXCEPT;

        bool IsInnerStateAll(NodeInnerState state) NN_NOEXCEPT;

        bool IsExsistNode(uint32_t index) NN_NOEXCEPT;

        void FindLatestSystemNode(uint32_t* pOutIndex, Ipv4Address myAddress) NN_NOEXCEPT;

        void FindLatestAppDetail(ApplicationInfo* pOutAppInfo, int appIndex) NN_NOEXCEPT;

        Result ClearRequireDownload(uint32_t index) NN_NOEXCEPT;

        void ClearRequireDownload() NN_NOEXCEPT;

        Result CheckLatestApplication(const ApplicationInfo& appInfo, int appIndex) NN_NOEXCEPT;

        void CheckNodeContents() NN_NOEXCEPT;

        bool MakeSystemUpdateSchedule(int* pOutListCount, RejectNodeInfo* pOutList,
            int listCount, Ipv4Address hostAddress) NN_NOEXCEPT;

        Result MakeContentShareSchedule(int* pOutListCount, RejectNodeInfo* pOutList,
            int listCount, const ApplicationInfo& appInfo, int appIndex, Ipv4Address hostAddress) NN_NOEXCEPT;

        bool IsCanUpdateSystem(SystemDeliveryInfo receiverSysInfo) NN_NOEXCEPT;

        bool IsCanUpdateSystem(SystemDeliveryInfo receiverSysInfo,
            const ApplicationInfo& appInfo) NN_NOEXCEPT;

        Result CanReceivePatch(int nodeIndex,
            const ApplicationInfo& appInfo, int appIndex, Ipv4Address hostAddress) NN_NOEXCEPT;

        bool FindSystemSender(int nodeNum, Ipv4Address hostAddress) NN_NOEXCEPT;

        bool FindSystemSender(int nodeNum, Ipv4Address hostAddress,
            const ApplicationInfo &appInfo) NN_NOEXCEPT;

        bool FindPatchSender(int nodeNum, bool allowReceiverSystemUpdate, bool allowSenderSystemUpdate,
            const ApplicationInfo& appInfo, int appIndex, Ipv4Address hostAddress) NN_NOEXCEPT;

        void ClearDownloadedCount() NN_NOEXCEPT;

        void GetNodeProgress(int* pOutProgressCount,
            NodeDetailProgress* outProgress, int progressCount) NN_NOEXCEPT;

        bool IsCompletedSystemDownloadAll() NN_NOEXCEPT;

        bool IsCompletedContentDownloadAll() NN_NOEXCEPT;

    private:
        NodeDetailInfo m_NodeInfo[NodeCountMax];
        int m_NodeCountMax;
        int m_NodeCount;
        uint32_t m_Index;

    };

}}} // end of namespace nn::lcs::detail
