﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       6 軸センサーの高精度ユーザーキャリブレーション処理に関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/hid/system/hid_SixAxisSensorCalibration.h>

namespace nn { namespace hid { namespace system {

/**
 * @brief       6　軸センサーの高精度ユーザーキャリブレーションシーケンスの段階を示す定義です。
 */
enum SixAxisSensorAccurateUserCalibrationStage
{
    SixAxisSensorAccurateUserCalibrationStage_InitialStage,    //!< 初期状態です
    SixAxisSensorAccurateUserCalibrationStage_AtRest,          //!< 筐体を机上に置いて静止させます
    SixAxisSensorAccurateUserCalibrationStage_RotateX,         //!< X 軸に関して筐体を回します
    SixAxisSensorAccurateUserCalibrationStage_RotateY,         //!< Y 軸に関して筐体を回します
    SixAxisSensorAccurateUserCalibrationStage_RotateZ,         //!< Z 軸に関して筐体を回します
    SixAxisSensorAccurateUserCalibrationStage_Completed,       //!< キャリブレーションが完了した状態です
};

/**
 * @brief      6 軸センサーの高精度ユーザーキャリブレーションの状態を表す型です。
 */
struct SixAxisSensorAccurateUserCalibrationState
{
    SixAxisSensorAccurateUserCalibrationStage stage;  //!< 6　軸センサーの高精度ユーザーキャリブレーションシーケンスの段階を示す値です。
    float progress;                                   //!< 各段階におけるキャリブレーションの進捗状況を表す値です。0.f 以上 1.f 以下の値です。
};

//! @name 6 軸センサーの高精度ユーザーキャリブレーション関連 API
//! @{

/**
 * @brief       6 軸センサーの高精度ユーザーキャリブレーション機能のサポート有無を取得します。
 *
 * @details     6 軸センサーに対して、高精度ユーザーキャリブレーション機能がサポートされているかどうかを返します。
 *
 * @param[in]   handle                  6 軸センサーのハンドル
 *
 * @return      高精度ユーザーキャリブレーション機能がサポートされている場合は true, そうでない場合は false
 *
 */
bool IsSixAxisSensorAccurateUserCalibrationSupported(const UniqueSixAxisSensorHandle& handle) NN_NOEXCEPT;

/**
 * @brief       6 軸センサーの高精度ユーザーキャリブレーション処理を開始します。
 *
 * @details     6 軸センサーに対して高精度ユーザーキャリブレーション処理を開始します。
 *              進行中の高精度ユーザーキャリブレーション処理がある場合は、 CancelSixAxisSensorAccurateUserCalibration() で終了する必要があります。
 *
 *              成功した場合はユーザーキャリブレーション領域に結果が書き込まれます。
 *              失敗した場合はユーザーキャリブレーション領域は呼出し前の状態のまま書き換わりません。
 *              対象の 6 軸センサーの状態に依らず、高精度ユーザーキャリブレーション処理を行うことが可能です。
 *
 * @param[in]   handle                  6 軸センサーのハンドル
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                                 処理に成功しました。}
 *   @handleresult{nn::hid::system::ResultSixAxisSensorNotSupported,  指定の 6 軸センサーはユーザーキャリブレーション処理に対応していません。}
 * @endretresult
 *
 */
Result StartSixAxisSensorAccurateUserCalibration(const UniqueSixAxisSensorHandle& handle) NN_NOEXCEPT;

/**
 * @brief       6 軸センサーの高精度ユーザーキャリブレーション処理の経過状況を取得します。
 *
 * @details     6 軸センサーの高精度ユーザーキャリブレーション処理は以下のステージに分かれて順に実行されます。
 *
 *              1. SixAxisSensorAccurateUserCalibrationStage_InitialStage
 *                 初期状態です。
 *              2. SixAxisSensorAccurateUserCalibrationStage_AtRest
 *                 机上に置いて静止してください。
 *              3. SixAxisSensorAccurateUserCalibrationStage_RotateX
 *                 X 軸に関して回してください。
 *              4. SixAxisSensorAccurateUserCalibrationStage_RotateY
 *                 Y 軸に関して回してください。
 *              5. SixAxisSensorAccurateUserCalibrationStage_RotateZ
 *                 Z 軸に関して回してください。
 *              6. SixAxisSensorAccurateUserCalibrationStage_Completed
 *                 キャリブレーションが完了した状態です
 *
 *              6 軸センサーのユーザーキャリブレーション処理が問題なく進行している間は、nn::ResultSuccess が返されます。
 *              キャリブレーション中に何か問題があった場合は、失敗の理由が返されます。
 *              キャリブレーション処理が問題なく完了すると SixAxisSensorAccurateUserCalibrationStage_Completed が取得されます。
 *
 * @param[out]  pOutValue               6 軸センサーの高精度ユーザーキャリブレーション処理の経過状況
 * @param[in]   handle                  6 軸センサーのハンドル
 *
 * @param[out]  pOutValue               6 軸センサーの高精度ユーザーキャリブレーション処理の経過状況
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                                              処理に成功しました。}
 *   @handleresult{nn::hid::system::ResultSixAxisSensorNotSupported,               指定の 6 軸センサーはユーザーキャリブレーション処理に対応していません。}
 *   @handleresult{nn::hid::system::ResultSixAxisSensorWriteFailure,               書き込み処理に失敗しました。}
 *   @handleresult{nn::hid::system::ResultSixAxisSensorAccurateInvalidTermination, ユーザーキャリブレーション処理が異常終了しました。}
 * @endretresult
 *
 */
Result GetSixAxisSensorAccurateUserCalibrationState(SixAxisSensorAccurateUserCalibrationState* pOutValue,
                                                    const UniqueSixAxisSensorHandle& handle) NN_NOEXCEPT;

/**
 * @brief       進行中の 6 軸センサーの高精度ユーザーキャリブレーション処理をキャンセルします。
 *
 * @details     指定した UniqueSixAxisSensorHandle に該当する 6 軸センサーの高精度ユーザーキャリブレーション処理をキャンセルします。
 *              StartSixAxisSensorAccurateUserCalibration() が呼ばれておらず進行中の高精度キャリブレーション処理が存在しない場合は本関数は何も作用しません。
 *
 * @param[in]   handle                  6 軸センサーのハンドル
 *
 */
void CancelSixAxisSensorAccurateUserCalibration(const UniqueSixAxisSensorHandle& handle) NN_NOEXCEPT;

//! @}

}}} // namespace nn::hid::system
