﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       Npad に関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/bluetooth/bluetooth_AddressTypes.h>
#include <nn/hid/system/hid_CommonTypes.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/util/util_Color.h>

namespace nn { namespace hid { namespace system {

/**
 * @brief      互換性担保のための構造体 (削除予定)
 */
struct RegisteredDeviceOld
{
    DeviceTypeSet deviceType;                      //!< デバイスの種類    4バイト
    uint8_t subType;                               //!< サブタイプ        1バイト
    NN_PADDING3;
    ::nn::bluetooth::Address address;              //!< Bluetooth Address 6 バイト
    NN_PADDING2;
    char identificationCode[16];                   //!< シリアルナンバー 16 バイト
    nn::util::Color4u8Type mainColor;              //!< メイン色情報 4バイト
    nn::util::Color4u8Type subColor;               //!< サブ色情報 4バイト
    InterfaceType interfaceType;                   //!< ペアリングに用いられたインタフェース 1バイト
    NN_PADDING7;
};

/**
 * @brief      登録されているコントローラーの登録情報です
 */
struct RegisteredDevice
{
    DeviceTypeSet deviceType;                      //!< デバイスの種類    4バイト
    uint8_t subType;                               //!< サブタイプ        1バイト
    NN_PADDING3;
    ::nn::bluetooth::Address address;              //!< Bluetooth Address 6 バイト
    NN_PADDING2;
    struct
    {
        Bit8 raw[16];
    } linkKey;                                     //!< LinkKey           16 バイト
    struct
    {
        char raw[20];                              //NULL終端
    } bdName;                                      //!< Bluetooth Device Name 20 バイト
    struct
    {
        uint8_t raw[3];
    } classOfDevice;                               //!< Class Of Device 3 バイト
    uint8_t featureSet;                            //!< FeatureSet 1 バイト
    struct
    {
        uint16_t vid;                              //!< Vid
        uint16_t pid;                              //!< Pid
    } hid;                                         //!< Hid に関連する情報 4バイト
    NN_PADDING4;
    char identificationCode[16];                   //!< シリアルナンバー 16 バイト
    nn::util::Color4u8Type mainColor;              //!< メイン色情報 4バイト
    nn::util::Color4u8Type subColor;               //!< サブ色情報 4バイト
    nn::util::Color4u8Type thirdColor;             //!< 右グリップ色 4バイト
    nn::util::Color4u8Type forthColor;             //!< 左グリップ色 4バイト
    InterfaceType interfaceType;                   //!< ペアリングに用いられたインタフェース 1バイト
    NN_PADDING7;
};

static const size_t RegisteredDeviceCountMax = 10;    //!< コントローラーの登録情報の最大数です

//! @name 登録されているコントローラーに関する関数
//! @{

/**
 * @brief       コントローラーの登録情報を取得します
 *
 * @details     コントローラーの登録情報はデバイス毎に、 RegisteredDeviceCountMax 個まで取得できます。
 *              ひとつもコントローラーが登録状態にない場合は、 0 が返ります。
 *
 * @param[out]  pOutValues                   コントローラーの登録情報を読み出すバッファ（配列）
 * @param[in]   count                        pOutValues の数
 *
 * @return      登録されているコントローラーの数を返します。
 *
 * @pre
 *              - pOutValues != nullptr
 *              - count >= 0
 * @post
 *              - 戻り値 n について、n >= 0
 */
int GetRegisteredDevices(RegisteredDevice* pOutValues, int count) NN_NOEXCEPT;

/**
 * @brief       デバイスへの接続処理がタイムアウトした際に通知を受け取るシステムイベントを登録します
 *
 * @details     デバイスへの接続処理を試みている場合に処理がタイムアウトで終了したことを通知します。
 *
 * @param[in]   pSystemEvent                  接続処理がタイムアウトしたことを受け取るためのシステムイベント
 * @param[in]   clearMode                     イベントのクリアモード
 *
 * @pre         - pSystemEvent != nullptr
 */
void BindConnectionTriggerTimeoutEvent(nn::os::SystemEventType* pSystemEvent, nn::os::EventClearMode clearMode) NN_NOEXCEPT;

/**
 * @brief      指定したアドレスのデバイスに対して Bluetooth 接続を試みます
 *
 * @details    指定されたアドレスのデバイスに対して Bluetooth 接続を試みます
 *             接続を試みるデバイスの Address を指定することで接続トリガーを送信することができます
 *             本体に登録されているデバイスの一覧は GetRegisteredDevices() で取得することができます。
 *             本 API が呼ばれた段階では接続処理が開始されません。
 *
 *             デバイスがみつからないず処理がタイムアウトした場合は、
 *             SetConnectionTriggerTimeoutEvent() で設定したイベントで通知を受け取ることができます。
 *
 * @param[in]  Address                        接続を試みる
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                             処理に成功しました。
 * @retval     ResultBluetoothAlreadyConnected           指定されたデバイスはすでに接続済みです
 * @retval     ResultBluetoothDeviceNotRegistered        指定されたデバイスがデータベースに登録されていません
 * @retval     ResultBluetoothOtherTriggerEventOnProgreess 別の接続トリガーイベントが進行中です
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 */
Result SendConnectionTrigger(nn::bluetooth::Address address) NN_NOEXCEPT;

/**
 * @brief      コントローラーサポートアプレットのコントローラーの探す画面において、
 *             コントローラーが新たに登録された際に通知を受け取るシステムイベントを登録します
 *
 * @details    コントローラーが新たに本体に登録された場合に通知を発行します
 *             本通知はコントローラーの新規に登録された場合のみ発行されます。
 *             もともと登録済みのデバイスが再度登録された場合においては、イベントは発行されません。
  *
 * @param[in]   pSystemEvent                  接続処理がタイムアウトしたことを受け取るためのシステムイベント
 * @param[in]   clearMode                     イベントのクリアモード
 *
 * @pre         - pSystemEvent != nullptr
 */
void BindDeviceRegisteredEventForControllerSupport(nn::os::SystemEventType* pSystemEvent, nn::os::EventClearMode clearMode) NN_NOEXCEPT;

//! @}

}}} // namespace nn::hid::system
