﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 *  @file
 *  @brief   NFC 管理に関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/applet/applet_Types.h>
#include <nn/hid/system/hid_Common.h>
#include <nn/hid/hid_NpadCommonTypes.h>
#include <nn/xcd/xcd_Device.h>

namespace nn { namespace hid { namespace system {

/**
 * @brief       NFC を搭載した hid の状態変化の通知をバインドします
 *
 * @details     NFC を搭載した hid の状態変化の通知を指定のイベントオブジェクトにバインドします。
 *              同時に複数のイベントオブジェクトに対してバインドすることはできません。
 *
 * @param[in]   pEvent                      イベントオブジェクトを指すポインタ
 * @param[in]   clearMode                   イベントオブジェクトのクリアモード
 *
 * @pre
 *              - pEvent != nullptr
 *              - *pEvent は未初期化状態
 * @post
 *              - *pEvent は初期化状態
 *              - *pEvent はシグナル状態
 */
void BindNfcDeviceUpdateEvent(::nn::os::SystemEventType* pEvent,
                              ::nn::os::EventClearMode clearMode) NN_NOEXCEPT;

/**
 * @brief       NFC を搭載した hid に関連づけられた Npad の NpadIdType を取得します。
 *
 * @details     NFC を搭載した　hid に関連づけられた Npad の NpadIdType を取得します。
 *              本関数で列挙されるデバイスは NFC が使える状態にあるかどうかを問わずすべて列挙します。
 *              ここで列挙されるデバイスに変化があった場合は、 BindNfcDeviceUpdateEvent() で通知されます。
 *
 * @param[out]  pOutValue                  NpadIdType を格納するバッファ
 * @param[out]  length                     pOutValue に格納できる NpadIdType の数
 *
 * @return      取得された Npad の数。NFCを搭載したデバイスが1つも存在しない場合は 0 を返します。
 *
 * @pre
 *              - pOutValue != nullptr
 *              - length > 0
 */
int GetNpadsWithNfc(NpadIdType* pOutValue, int length) NN_NOEXCEPT;

/**
 * @brief       NFC を搭載した Npad に関連付けられた nn::xcd::DeviceHandle を取得します。
 *
 * @details     Xcd のハンドルを取得したい NpadIdType を渡して、 nn::xcd::DeviceHandle を取得します。
 *              Npad は NFC を搭載している必要があります。
 *
 * @param[out]  pOutValue                  nn::xcd::DeviceHandle を格納するポインタ
 * @param[in]   npadId                     xcd のデバイスハンドルを取得する Npad の NpadIdType
 *
 * @retresult
 *     @handleresult{ResultSuccess,                 Xcd のハンドルの取得に成功しました。}
 *     @handleresult{ResultNoNfcDeviceFoundOnNpad,  指定された Npad は NFC デバイスを搭載していません。}
 * @endretresult
 *
 * @pre
 *              - pOutValue != nullptr
 */
Result GetXcdHandleForNpadWithNfc(nn::xcd::DeviceHandle* pOutHandle, NpadIdType npadId) NN_NOEXCEPT;

/**
 * @brief       Npad の NFC のアクティブ状態に変化があったときに通知を指定のイベントオブジェクトにバインドします。
 *
 * @details     Npad に搭載された NFC のアクティブ状態は、 ActivateNfcOnNpad の呼び出しに対して非同期で変化します。
　*              またシステム内部で、 アクティブ状態であった NFC が自動的に非アクティブ状態に変化することがあります。
 *              本関数でイベントオブジェクトをバインドすることで、 Npad のアクティブ状態に変化した場合に、通知を受け取ることができます。
 *
 * @param[in]   npadId                      通知を受け取る Npad の NpadIdType
 * @param[in]   pEvent                      イベントオブジェクトを指すポインタ
 * @param[in]   clearMode                   イベントオブジェクトのクリアモード
 *
 * @pre
 *              - pEvent != nullptr
 *              - *pEvent は未初期化状態
 * @post
 *              - *pEvent は初期化状態
 *              - *pEvent はシグナル状態
 */
void BindNfcActivateEvent(NpadIdType npadId,
                          ::nn::os::SystemEventType* pEvent,
                          ::nn::os::EventClearMode clearMode) NN_NOEXCEPT;

/**
 * @brief       Npad に搭載された　NFC のアクティブ状態を取得します。
 *
 * @param[in]   npadId                      アクティブ状態を取得する Npad の NpadIdType
 *
 * @return      Npad に搭載された NFC がアクティブであれば、 true を返します。
 *
 */
bool IsNfcActivated(NpadIdType npadId) NN_NOEXCEPT;

/**
 * @brief       Npad に搭載された NFC をアクティブ化します。
 *
 * @details     Npad に搭載された NFC は使用を開始する前に、アクティブ化する必要があります。
 *              非アクティブな Npad をアクティブにする処理には時間がかかります。
 *              本関数は呼び出しが行われると、即時に処理が返ります。
 *              アクティブ化の処理が完了したかどうかを取得するためには、 BindNfcActivateEvent() でイベントオブジェクトをバインドし、
 *              IsNfcActivated() で変化した状態を確認します。
 *
 *              有効にできる NFC デバイスは 1つだけです。
 *              NFC を使用中に別の Npad の NFC をアクティブにするためには、　NFC がアクティブな　Npad に対して DeactivateNfc() を呼んでください。
 *
 * @param[in]   npadId                      アクティブにする Npad の NpadIdType
 *
 * @retresult
 *     @handleresult{ResultSuccess,                 非アクティブ化に成功しました。}
 *     @handleresult{ResultNoNfcDeviceFoundOnNpad,  指定された Npad は NFC デバイスを搭載していません。}
 *     @handleresult{ResultNfcActivatedOnOtherNpad, 既に別の Npad で NFC が有効になっています。}
 *     @handleresult{ResultNfcActivateFailureNpadBusy, Npad で NFC と排他で制御されるべき他の機能が有効なためアクティブ化できませんでした。}
 * @endretresult
 *
 */
Result ActivateNfc(NpadIdType npadId) NN_NOEXCEPT;

/**
 * @brief       Npad に搭載された NFC を非アクティブします。
 *
 * @details     NFC の使用を終了した際には、本関数を呼んで NFC を非アクティブにしてください。
 *              本関数は同期処理され、呼び出しが完了した時点で、NFC は非アクティブになります。
 *
 * @param[in]   npadId                      非アクティブにする Npad の NpadIdType
 *
 * @retresult
 *     @handleresult{ResultSuccess,                 非アクティブ化に成功しました。}
 *     @handleresult{ResultNoNfcDeviceFoundOnNpad,  指定された Npad は NFC デバイスを搭載していません。}
 * @endretresult
 *
 */
Result DeactivateNfc(NpadIdType npadId) NN_NOEXCEPT;

/**
 * @brief       Npad に搭載された NFC をアクティブ化します。
 *
 * @details     Npad に搭載された NFC は使用を開始する前に、アクティブ化する必要があります。
 *              非アクティブな Npad をアクティブにする処理には時間がかかります。
 *              本関数は呼び出しが行われると、即時に処理が返ります。
 *              アクティブ化の処理が完了したかどうかを取得するためには、 BindNfcActivateEvent() でイベントオブジェクトをバインドし、
 *              IsNfcActivated() で変化した状態を確認します。
 *
 *              有効にできる NFC デバイスは 1つだけです。
 *              NFC を使用中に別の Npad の NFC をアクティブにするためには、　NFC がアクティブな　Npad に対して DeactivateNfc() を呼んでください。
 *
 *              NFC を有効にする際は、クライアントの AppletResourceUserId を指定します。
 *
 * @param[in]   npadId                      アクティブにする Npad の NpadIdType
 * @param[in]   aruid                       NFC を使うクライアントの Aruid
 *
 * @retresult
 *     @handleresult{ResultSuccess,                 非アクティブ化に成功しました。}
 *     @handleresult{ResultNoNfcDeviceFoundOnNpad,  指定された Npad は NFC デバイスを搭載していません。}
 *     @handleresult{ResultNfcActivatedOnOtherNpad, 既に別の Npad で NFC が有効になっています。}
 *     @handleresult{ResultNfcActivateFailureNpadBusy, Npad で NFC と排他で制御されるべき他の機能が有効なためアクティブ化できませんでした。}
 * @endretresult
 *
 */
Result ActivateNfc(NpadIdType npadId, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

/**
 * @brief       Npad に搭載された NFC を非アクティブします。
 *
 * @details     NFC の使用を終了した際には、本関数を呼んで NFC を非アクティブにしてください。
 *              本関数は同期処理され、呼び出しが完了した時点で、NFC は非アクティブになります。
 *
 * @param[in]   npadId                      非アクティブにする Npad の NpadIdType
 * @param[in]   aruid                       NFC を使っていたクライアントの Aruid
 *
 * @retresult
 *     @handleresult{ResultSuccess,                 非アクティブ化に成功しました。}
 *     @handleresult{ResultNoNfcDeviceFoundOnNpad,  指定された Npad は NFC デバイスを搭載していません。}
 * @endretresult
 *
 */
Result DeactivateNfc(NpadIdType npadId, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;


/**
 * @brief       本関数は廃止されました。
 *
 */
bool GetXcdDeviceHandleForNfc(nn::xcd::DeviceHandle* pOutHandle) NN_NOEXCEPT;

}}} // namespace nn::hid::system
