﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 *  @file
 *  @brief   Keyboard に関するシステムアプリケーション向け API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/hid/system/hid_Common.h>
#include <nn/util/util_BitFlagSet.h>

namespace nn { namespace hid { namespace system {

/**
 * @brief       Keyboard のロックキーイベント定義です。
 */
struct KeyboardLockKeyEvent
{
    typedef ::nn::util::BitFlagSet<32, KeyboardLockKeyEvent>::Flag<0>
            NumLockOn;          //!< Num Lock の有効化イベント
    typedef ::nn::util::BitFlagSet<32, KeyboardLockKeyEvent>::Flag<1>
            NumLockOff;         //!< Num Lock の無効化イベント
    typedef ::nn::util::BitFlagSet<32, KeyboardLockKeyEvent>::Flag<2>
            NumLockToggle;      //!< Num Lock の切り替えイベント
    typedef ::nn::util::BitFlagSet<32, KeyboardLockKeyEvent>::Flag<3>
            CapsLockOn;         //!< Caps Lock の有効化イベント
    typedef ::nn::util::BitFlagSet<32, KeyboardLockKeyEvent>::Flag<4>
            CapsLockOff;        //!< Caps Lock の無効化イベント
    typedef ::nn::util::BitFlagSet<32, KeyboardLockKeyEvent>::Flag<5>
            CapsLockToggle;     //!< Caps Lock の切り替えイベント
    typedef ::nn::util::BitFlagSet<32, KeyboardLockKeyEvent>::Flag<6>
            ScrollLockOn;       //!< Scroll Lock の有効化イベント
    typedef ::nn::util::BitFlagSet<32, KeyboardLockKeyEvent>::Flag<7>
            ScrollLockOff;      //!< Scroll Lock の無効化イベント
    typedef ::nn::util::BitFlagSet<32, KeyboardLockKeyEvent>::Flag<8>
            ScrollLockToggle;   //!< Scroll Lock の切り替えイベント
};

/**
 * @brief       Keyboard のロックキーイベントの集合を扱う型です。
 */
typedef ::nn::util::BitFlagSet<32, KeyboardLockKeyEvent>
        KeyboardLockKeyEventSet;

//! @name Keyboard 関連 API (システムアプリケーション向け)
//! @{

/**
 * @brief       Keyboard のロックキーイベントを送信します。
 *
 * @details     イベントはフラグの添字の値が小さいものから順に処理されます。
 *              入力モードはキーボード操作からも変更される可能性があるため、
 *              当該関数呼び出し後の入力モードの状態には何の保証も無い点に
 *              注意してください。
 *
 * @param[in]   value                       ロックキーイベント
 *
 * @pre
 *              - InitializeKeyboard() の呼び出しが完了している
 */
void SendKeyboardLockKeyEvent(KeyboardLockKeyEventSet value) NN_NOEXCEPT;

//! @}

}}} // namespace nn::hid::system
