﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 *  @file
 *  @brief   IrSensor 管理に関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/applet/applet_Types.h>
#include <nn/hid/system/hid_Common.h>
#include <nn/hid/hid_NpadCommonTypes.h>
#include <nn/xcd/xcd_Device.h>

namespace nn { namespace hid { namespace system {

/**
 * @brief       IrSensor に対応している NpadId のリストです。
 */
const NpadIdType IrSensorSupportedNpadIds[] = {NpadId::No1,
                                               NpadId::No2,
                                               NpadId::No3,
                                               NpadId::No4,
                                               NpadId::No5,
                                               NpadId::No6,
                                               NpadId::No7,
                                               NpadId::No8,
                                               NpadId::Handheld};

/**
 * @brief       IrSensor に対応している NpadId の数です。
 */
const int IrSensorSupportedNpadIdsCount = sizeof(IrSensorSupportedNpadIds) / sizeof(IrSensorSupportedNpadIds[0]);

/**
 * @brief       IrSensor の状態です。
 */
enum IrSensorState
{
    IrSensorState_NotConnected,    //!< デバイスが接続されていません
    IrSensorState_NotSupported,    //!< IrSensor 非対応デバイスが接続されています。
    IrSensorState_Deactivated,     //!< IrSensor 対応デバイスが接続されており、非アクティベート状態にあります。
    IrSensorState_Activated,       //!< IrSensor 対応デバイスが接続されており、アクティベート状態にあります。
};


/**
 * @brief       IrSensor を搭載した Npad に関連付けられた nn::xcd::DeviceHandle を取得します。
 *
 * @details     Xcd のハンドルを取得したい NpadIdType を渡して、 nn::xcd::DeviceHandle を取得します。
 *              Npad は IrSensor を搭載している必要があります。
 *
 * @param[out]  pOutValue                  nn::xcd::DeviceHandle を格納するポインタ
 * @param[in]   npadId                     xcd のデバイスハンドルを取得する Npad の NpadIdType
 *
 * @retresult
 *     @handleresult{ResultSuccess,                 Xcd のハンドルの取得に成功しました。}
 *     @handleresult{ResultNoIrSensorDeviceFoundOnNpad,  指定された Npad は IrSensor デバイスを搭載していません。}
 * @endretresult
 *
 * @pre
 *              - pOutValue != nullptr
 */
Result GetXcdHandleForNpadWithIrSensor(nn::xcd::DeviceHandle* pOutHandle, NpadIdType npadId) NN_NOEXCEPT;

/**
 * @brief       Npad の IrSensor の状態に変化があったときに通知を指定のイベントオブジェクトにバインドします。
 *
 * @details     Npad に搭載された IrSensor の状態は、 ActivateIrSensor の呼び出しに対して非同期で変化します。
 *             またシステム内部で、 アクティブ状態であった IrSensor が自動的に非アクティブ状態に変化することがあります。
 *              本関数でイベントオブジェクトをバインドすることで、 Npad のアクティブ状態に変化した場合に、通知を受け取ることができます。
 *
 * @param[in]   npadId                      通知を受け取る Npad の NpadIdType
 * @param[in]   pEvent                      イベントオブジェクトを指すポインタ
 * @param[in]   clearMode                   イベントオブジェクトのクリアモード
 *
 * @pre
 *              - pEvent != nullptr
 *              - *pEvent は未初期化状態
 * @post
 *              - *pEvent は初期化状態
 *              - *pEvent はシグナル状態
 */
void BindIrSensorEvent(NpadIdType npadId,
                       ::nn::os::SystemEventType* pEvent,
                       ::nn::os::EventClearMode clearMode) NN_NOEXCEPT;

/**
 * @brief       Npad に搭載された　IrSensor の状態を取得します。
 *
 * @param[in]   npadId                      状態を取得する Npad の NpadIdType
 *
 * @return      IrSensor の状態
 *
 */
IrSensorState GetIrSensorState(NpadIdType npadId) NN_NOEXCEPT;

/**
 * @brief       Npad に搭載された IrSensor をアクティブ化します。
 *
 * @details     Npad に搭載された IrSensor は使用を開始する前に、アクティブ化する必要があります。
 *              非アクティブな Npad をアクティブにする処理には時間がかかります。
 *              本関数は呼び出しが行われると、即時に処理が返ります。
 *              アクティブ化の処理が完了したかどうかを取得するためには、 BindIrSensorActivateEvent() でイベントオブジェクトをバインドし、
 *              IsIrSensorActivated() で変化した状態を確認します。
 *
 *              有効にできる IrSensor デバイスは 1つだけです。
 *              IrSensor を使用中に別の Npad の IrSensor をアクティブにするためには、　IrSensor がアクティブな　Npad に対して DeactivateIrSensor() を呼んでください。
 *
 *              IrSensor を有効にする際は、クライアントの AppletResourceUserId を指定します。
 *
 * @param[in]   npadId                      アクティブにする Npad の NpadIdType
 * @param[in]   aruid                       IrSensor を使うクライアントの Aruid
 *
 * @retresult
 *     @handleresult{ResultSuccess,                 非アクティブ化に成功しました。}
 *     @handleresult{ResultNoIrSensorDeviceFoundOnNpad,  指定された Npad は IrSensor デバイスを搭載していません。}
 *     @handleresult{ResultIrSensorActivationLimitOver,  Npad で IrSensor が上限台数以上アクティベートされました。}
 *     @handleresult{ResultIrSensorActivateFailureNpadBusy, Npad で IrSensor と排他で制御されるべき他の機能が有効なためアクティブ化できませんでした。}
 * @endretresult
 *
 */
Result ActivateIrSensor(NpadIdType npadId, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

/**
 * @brief       Npad に搭載された IrSensor を非アクティブします。
 *
 * @details     IrSensor の使用を終了した際には、本関数を呼んで IrSensor を非アクティブにしてください。
 *              本関数は同期処理され、呼び出しが完了した時点で、IrSensor は非アクティブになります。
 *
 * @param[in]   npadId                      非アクティブにする Npad の NpadIdType
 * @param[in]   aruid                       IrSensor を使っていたクライアントの Aruid
 *
 * @retresult
 *     @handleresult{ResultSuccess,                 非アクティブ化に成功しました。}
 *     @handleresult{ResultNoIrSensorDeviceFoundOnNpad,  指定された Npad は IrSensor デバイスを搭載していません。}
 * @endretresult
 *
 */
Result DeactivateIrSensor(NpadIdType npadId, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;


}}} // namespace nn::hid::system
