﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 *  @file
 *  @brief   入力変化の検出処理に関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/util/util_BitFlagSet.h>

namespace nn { namespace hid { namespace system {

const int InputSourceIdCountMax = 16;   //!< 内部的に保持される InputSourceId の最大数

/**
 * @brief       InputSourceId の定義です。
 */
struct InputSourceId
{
    typedef ::nn::util::BitFlagSet<32, InputSourceId>::Flag<0>  Pad;                   //!< Pad (ボタン, アナログスティックボタン) の識別子
    typedef ::nn::util::BitFlagSet<32, InputSourceId>::Flag<1>  Sensor;                //!< センサー (加速度, 角速度)の識別子
    typedef ::nn::util::BitFlagSet<32, InputSourceId>::Flag<2>  DebugPad;              //!< デバッグコントローラーの識別子
    typedef ::nn::util::BitFlagSet<32, InputSourceId>::Flag<3>  TouchScreen;           //!< タッチスクリーンの識別子
    typedef ::nn::util::BitFlagSet<32, InputSourceId>::Flag<4>  Keyboard;              //!< キーボードの識別子
    typedef ::nn::util::BitFlagSet<32, InputSourceId>::Flag<5>  Mouse;                 //!< マウスの識別子
    typedef ::nn::util::BitFlagSet<32, InputSourceId>::Flag<6>  UsbPort;               //!< USB 接続端子の識別子
    typedef ::nn::util::BitFlagSet<32, InputSourceId>::Flag<7>  SdCardSlot;            //!< SD カードスロットの識別子
    typedef ::nn::util::BitFlagSet<32, InputSourceId>::Flag<8>  GameCardSlot;          //!< ゲームカードスロットの識別子
    typedef ::nn::util::BitFlagSet<32, InputSourceId>::Flag<9>  PadConnection;         //!< Pad 接続の識別子
    typedef ::nn::util::BitFlagSet<32, InputSourceId>::Flag<10> AnalogStick;           //!< アナログスティックの識別子
    typedef ::nn::util::BitFlagSet<32, InputSourceId>::Flag<11> ConsoleSixAxisSensor;  //!< 本体 6 軸センサーの識別子
};

/**
 * @brief       InputSourceId の集合を扱う型です。
 */
typedef ::nn::util::BitFlagSet<32, InputSourceId> InputSourceIdSet;

/**
 * @brief       InputSource の状態を表す構造体です。
 */
struct InputSourceState
{
    int64_t timestamp;        //!< 最後に入力変化があった時点でのチック値です。
};

/**
 * @brief       InputDetector を初期化します。
 */
void InitializeInputDetector() NN_NOEXCEPT;

/**
 * @brief       InputSource の状態を取得します。
 *
 * @details     指定の InputSourceId と対応する InputSource の状態を取得します。
 *              複数の InputSourceId が指定された場合は、
 *              最後に入力変化があった InputSource のタイムスタンプを取得します。
 *
 * @param[out]  pOutValue               状態を読み出すバッファ
 * @param[in]   id                      InputSource ID
 *
 * @pre
 *              - id.CountPopulation() >= 1
 *              - InitializeInputDetector() の呼び出しが完了している
 *              - pOutValue != nullptr
 */
void GetInputSourceState(InputSourceState* pOutValue,
                         const InputSourceIdSet& id) NN_NOEXCEPT;

/**
 * @brief       入力変化を通知します。
 *
 * @details     指定の InputSource の入力変化を通知します。
 *              複数指定が可能です。
 *
 * @param[in]   id                      InputSource ID
 *
 * @pre
 *              - id.CountPopulation() >= 1
 */
void NotifyInputDetector(const InputSourceIdSet& id) NN_NOEXCEPT;

}}} // namespace nn::hid::system
