﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       Npad に関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/bluetooth/bluetooth_AddressTypes.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/util/util_BitFlagSet.h>
#include <nn/util/util_Color.h>

#include "hid_RegisteredDevice.h"

namespace nn { namespace hid { namespace system {

//! @name 本体からコントローラーへの接続機能
//! @{


/**
 * @brief       本体からコントローラーに対して Bluetooth 接続が可能なコントローラーの登録情報の一覧を取得します
 *
 * @details     コントローラーの登録情報はデバイス毎に、 RegisteredDeviceCountMax 個まで取得できます。
 *              ひとつもコントローラーが登録状態にない場合は、 0 が返ります。
 *              本関数では、 SendConnectionTrigger() を用いて本体からコントローラーに対して接続をトリガーすることが可能なデバイスのみを列挙します
 *
 * @param[out]  pOutValues                   コントローラーの登録情報を読み出すバッファ（配列）
 * @param[in]   count                        pOutValues の数
 *
 * @return      本体からコントローラー接続可能な登録済コントローラーの数を返します。
 *
 * @pre
 *              - pOutValues != nullptr
 *              - count >= 0
 * @post
 *              - 戻り値 n について、n >= 0
 */
int GetConnectableRegisteredDevices(RegisteredDevice* pOutValues, int count) NN_NOEXCEPT;

/**
 * @brief       デバイスへの接続処理がタイムアウトした際に通知を受け取るシステムイベントを登録します
 *
 * @details     デバイスへの接続処理を試みている場合に処理がタイムアウトで終了したことを通知します。
  *
 * @param[in]   pSystemEvent                  接続処理がタイムアウトしたことを受け取るためのシステムイベント
 *
 * @pre         - pSystemEvent != nullptr
 */
void BindConnectionTriggerTimeoutEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

/**
 * @brief      指定したアドレスのデバイスに対して Bluetooth 接続を試みます
 *
 * @details    指定されたアドレスのデバイスに対して Bluetooth 接続を試みます
 *             接続を試みるデバイスの Address を指定することで接続トリガーを送信することができます
 *             本体に登録されているデバイスの一覧は GetRegisteredDevices() で取得することができます。
 *             本 API が呼ばれた段階では接続処理が開始されません。
 *
 *             デバイスがみつからないず処理がタイムアウトした場合は、
 *             SetConnectionTriggerTimeoutEvent() で設定したイベントで通知を受け取ることができます。
 *
 * @param[in]  Address                        接続を試みるデバイスの Bluetooth Address
 *
 * @return     実行結果を返します。
 * @retval     ResultSuccess                             処理に成功しました。
 * @retval     ResultBluetoothAlreadyConnected           指定されたデバイスはすでに接続済みです
 * @retval     ResultBluetoothDeviceNotRegistered        指定されたデバイスがデータベースに登録されていません
 *
 * @pre        - xcd ライブラリは初期化済である必要があります。
 */
Result SendConnectionTrigger(nn::bluetooth::Address address) NN_NOEXCEPT;

/**
 * @brief       Bluetooth 接続可能なコントローラーの台数を取得します
 *
 * @details     現在のシステムに対して追加で Bluetooth で接続が可能なコントローラーの数を取得します
 *              コントローラーが Bluetooth 接続できる数はその状況で都度変化します
 *              本関数では、現在接続されている台数は除いて追加で Bluetooth で接続可能な台数を取得します。
 *              0 が返る場合、これ以上 Bluetooth でのコントローラー接続ができません。
 *              0 が出る状況では、 SendConnectionTrigger() を発行してもコントローラーは接続できません。
 *
 */
int GetAllowedBluetoothLinksCount() NN_NOEXCEPT;

//! @}

}}} // namespace nn::hid::system
