﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   振動子に関する API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/hid/hid_VibrationDeviceApi.h>

namespace nn { namespace hid {

/**
 * @brief   Gc コントローラーに搭載された偏芯モーターを表す、振動子の物理的な種類です
 */
const VibrationDeviceType VibrationDeviceType_GcErm = static_cast<VibrationDeviceType>(2);

/**
 * @brief   Gc コントローラーに搭載された ERM を制御するためのコマンドの定義です
 */
enum VibrationGcErmCommand : int64_t
{
    VibrationGcErmCommand_Stop = 0,       //!< モーターを自然に停止します。
    VibrationGcErmCommand_Start = 1,      //!< モーターを振動させます
    VibrationGcErmCommand_StopHard = 2,   //!< モーターが振動している場合、モーターを急停止します。
};

//! @name Gc コントローラーに搭載された振動子の制御関連 API
//! @{

/**
 * @brief       Gc コントローラーに搭載された振動子に対して、振動コマンドを送信します。
 *
 * @param[in]   handle          振動子のハンドル
 * @param[in]   command         振動コマンド
 *
 * @pre
 *              - 対応する振動子について InitializeVibrationDevice() の呼び出しが完了している
 *
 * @details
 *  振動子に振動コマンドを送信します。
 *  送信した振動コマンドは別の値で上書きされるか Npad に割り当てられたコントローラーとの接続状態が切れるまで保持されます。
 *  そのため、振動を止めるためには明示的に VibrationGcErmCommand_Stop または VibrationGcErmCommand_StopHard を送信する必要があります。
 */
void SendVibrationGcErmCommand(const VibrationDeviceHandle& handle, const VibrationGcErmCommand& command) NN_NOEXCEPT;

/**
 * @brief       振動子で発生している実際の振動値を取得します。
 *
 * @param[out]  pOutValue       実際の振動値の格納先
 * @param[in]   handle          振動子のハンドル
 *
 * @pre
 *              - 対応する振動子について InitializeVibrationDevice() の呼び出しが完了している
 *
 * @details
 *  振動子で発生している実際の振動コマンドを取得します。
 *  Npad にコントローラーが割り当てられた直後の状態は VibrationGcErmCommand_Stop となります。
 *
 *  SendVibrationGcErmCommand 関数で指定した振動値が実際に振動子に送信されるまで、
 *  最大で数十ミリ秒程度の時間がかかります。
 *  一方この関数で取得される振動値は、実際に振動子に送信された最新の振動コマンドです。
 *  したがって、 SendVibrationGcErmCommand 関数で送信した振動コマンドが
 *  この関数で取得される振動コマンドに反映されるまでの間に、
 *  最大で数十ミリ秒程度のタイミングのずれが発生します。
 */
void GetActualVibrationGcErmCommand(VibrationGcErmCommand* pOutValue, const VibrationDeviceHandle& handle) NN_NOEXCEPT;

//! @}

}} // namespace nn::hid
