﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       キーコードに関する API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_Keyboard.h>
#include <nn/util/util_BitPack.h>
#include <nn/util/util_TypedStorage.h>

namespace nn { namespace hid {

/**
 * @brief       Keyboard のレイアウト定義です。
 */
struct KeyboardLayout
{
    static const ::nn::util::BitPack<uint32_t, KeyboardLayout
        > EnglishUnitedKingdom;             //!< 英語 (イギリス)
    static const ::nn::util::BitPack<uint32_t, KeyboardLayout
        > EnglishUnitedStates;              //!< 英語 (アメリカ)
    static const ::nn::util::BitPack<uint32_t, KeyboardLayout
        > EnglishUnitedStatesInternational; //!< 英語 (US-International)
    static const ::nn::util::BitPack<uint32_t, KeyboardLayout
        > French;                           //!< フランス語
    static const ::nn::util::BitPack<uint32_t, KeyboardLayout
        > FrenchCanada;                     //!< フランス語 (カナダ)
    static const ::nn::util::BitPack<uint32_t, KeyboardLayout
        > German;                           //!< ドイツ語
    static const ::nn::util::BitPack<uint32_t, KeyboardLayout
        > Italian;                          //!< イタリア語
    static const ::nn::util::BitPack<uint32_t, KeyboardLayout
        > Japanese;                         //!< 日本語
    static const ::nn::util::BitPack<uint32_t, KeyboardLayout
        > Portuguese;                       //!< ポルトガル語
    static const ::nn::util::BitPack<uint32_t, KeyboardLayout
        > RussianCyrillic;                  //!< ロシア語 (キリル)
    static const ::nn::util::BitPack<uint32_t, KeyboardLayout
        > RussianLatin;                     //!< ロシア語 (ラテン)
    static const ::nn::util::BitPack<uint32_t, KeyboardLayout
        > Spanish;                          //!< スペイン語
    static const ::nn::util::BitPack<uint32_t, KeyboardLayout
        > SpanishLatinAmerica;              //!< スペイン語 (ラテンアメリカ)
    static const ::nn::util::BitPack<uint32_t, KeyboardLayout
        > KoreanHangul;                     //!< 韓国語 (ハングル)
    static const ::nn::util::BitPack<uint32_t, KeyboardLayout
        > KoreanLatin;                      //!< 韓国語 (ラテン)
    static const ::nn::util::BitPack<uint32_t, KeyboardLayout
        > ChineseSimplified;                //!< 簡体字中国語
    static const ::nn::util::BitPack<uint32_t, KeyboardLayout
        > ChineseTraditionalZhuyin;         //!< 繁体字中国語 (注音)
    static const ::nn::util::BitPack<uint32_t, KeyboardLayout
        > ChineseTraditionalCangjie;        //!< 繁体字中国語 (倉頡)
    static const ::nn::util::BitPack<uint32_t, KeyboardLayout
        > ChineseTraditionalLatin;          //!< 繁体字中国語 (ラテン)
};

/**
 * @brief       Keyboard のレイアウトを表す型です。
 */
typedef ::nn::util::BitPack<uint32_t, KeyboardLayout> KeyboardLayoutType;

namespace detail {

struct KeyCodeComposerStorage;

} // namespace detail

/**
 * @brief   キーコードの組み立てを扱うクラスです。
 *
 * @details キーコードは UTF-16 符号で表現されます。
 */
class KeyCodeComposer
{
    NN_DISALLOW_COPY(KeyCodeComposer);
    NN_DISALLOW_MOVE(KeyCodeComposer);

private:
    ::nn::util::TypedStorage<detail::KeyCodeComposerStorage, 32, 8> m_Storage;

public:
    /**
     * @brief   コンストラクタです。
     *
     * @post
     *          - GetDeadKeyMode() == false
     *          - GetRomajiHiraganaMode() == false
     *          - GetRomajiKatakanaMode() == false
     *          - GetCompositionCount() == 0
     *          - GetOutputCount() == 0
     */
    KeyCodeComposer() NN_NOEXCEPT;

    /**
     * @brief   デッドキー処理を設定します。
     *
     * @param[in]       enables     デッドキー処理が有効か否かを表す値
     *
     * @post
     *          - GetDeadKeyMode() == enables
     *          - (!enables || !GetRomajiHiraganaMode()) == true
     *          - (!enables || !GetRomajiKatakanaMode()) == true
     */
    void SetDeadKeyMode(bool enables) NN_NOEXCEPT;

    /**
     * @brief   デッドキー処理が有効か否かを表す値を返します。
     *
     * @return  デッドキー処理が有効か否かを表す値です。
     */
    bool GetDeadKeyMode() const NN_NOEXCEPT;

    /**
     * @brief   ローマ字変換（ひらがな）を設定します。
     *
     * @param[in]       enables     ローマ字変換（ひらがな）が有効か否かを表す値
     *
     * @post
     *          - GetRomajiHiraganaMode() == enables
     *          - (!enables || !GetDeadKeyMode()) == true
     *          - (!enables || !GetRomajiKatakanaMode()) == true
     */
    void SetRomajiHiraganaMode(bool enables) NN_NOEXCEPT;

    /**
     * @brief   ローマ字変換（ひらがな）が有効か否かを表す値を返します。
     *
     * @return  ローマ字変換（ひらがな）が有効か否かを表す値です。
     */
    bool GetRomajiHiraganaMode() const NN_NOEXCEPT;

    /**
     * @brief   ローマ字変換（カタカナ）を設定します。
     *
     * @param[in]       enables     ローマ字変換（カタカナ）が有効か否かを表す値
     *
     * @post
     *          - GetRomajiKatakanaMode() == enables
     *          - (!enables || !GetDeadKeyMode()) == true
     *          - (!enables || !GetRomajiHiraganaMode()) == true
     */
    void SetRomajiKatakanaMode(bool enables) NN_NOEXCEPT;

    /**
     * @brief   ローマ字変換（カタカナ）が有効か否かを表す値を返します。
     *
     * @return  ローマ字変換（カタカナ）が有効か否かを表す値です。
     */
    bool GetRomajiKatakanaMode() const NN_NOEXCEPT;

    /**
     * @brief   組み立て中キーコード列の長さを返します。
     *
     * @return  組み立て中キーコード列の長さです。
     *
     * @post
     *          - 戻り値 n について、n >= 0
     */
    int GetCompositionCount() const NN_NOEXCEPT;

    /**
     * @brief   組み立て中キーコード列を取得します。
     *
     * @details 取得するキーコードの数が outBuffer の長さより小さい場合、
     *          余った領域に対しては何も変更は加えません。
     *
     * @param[out]      outBuffer   キーコード列の読み出し先（配列）
     * @param[in]       count       outBuffer の長さ
     *
     * @return  取得したキーコード列の長さです。
     *
     * @pre
     *          - outBuffer != nullptr
     * @post
     *          - 戻り値 n について、n >= 0
     */
    int GetComposition(uint16_t outBuffer[], int count) const NN_NOEXCEPT;

    /**
     * @brief   組み立て中キーコード列の確定部分の長さを返します。
     *
     * @return  組み立て中キーコード列の確定部分の長さです。
     */
    int GetOutputCount() const NN_NOEXCEPT;

    /**
     * @brief   組み立て中キーコード列の確定部分を取り出します。
     *
     * @details 取り出すキーコードの数が outBuffer の長さより小さい場合、
     *          余った領域に対しては何も変更は加えません。
     *
     * @param[out]      outBuffer   キーコード列の読み出し先（配列）
     * @param[in]       count       outBuffer の長さ
     *
     * @return  取り出したキーコード列の長さです。
     *
     * @pre
     *          - outBuffer != nullptr
     * @post
     *          - 戻り値 n について、n >= 0
     */
    int DequeueOutputs(uint16_t outBuffer[], int count) NN_NOEXCEPT;

    /**
     * @brief   組み立て中キーコード列の末尾にキーコードを追加します。
     *
     * @details キーコードの追加は 1 つずつ処理され、
     *          その度に組み立て中キーコード列の状態が更新されます。
     *          キーコード列に確定部分が発生すると、その時点でキーコードの追加は
     *          打ち切られます。
     *          既に確定部分が存在する場合はキーコードの追加は行われません。
     *          GetOutputCount() が 0 になるまで DequeueOutputs() を呼び出して
     *          確定部分を取り出すことで再びキーコードの追加が可能になります。
     *
     * @param[in]       buffer      キーコード列
     * @param[in]       count       buffer の長さ
     *
     * @return  追加したキーコードの数です。
     *
     * @pre
     *          - buffer != nullptr
     * @post
     *          - 戻り値 n について、n >= 0
     */
    int EnqueueKeyCodes(const uint16_t buffer[], int count) NN_NOEXCEPT;

    /**
     * @brief   組み立て中キーコード列の末尾から未確定部分を取り出します。
     *
     * @details 取り出すキーコードの数が outBuffer の長さより小さい場合、
     *          余った領域に対しては何も変更は加えません。
     *          キーコードは古いものから順に格納されます。
     *          未確定の Alt コードが存在する場合、これは破棄されます。
     *
     * @param[out]      outBuffer   キーコード列の読み出し先（配列）
     * @param[in]       count       outBuffer の長さ
     *
     * @return  取り出したキーコード列の長さです。
     *
     * @pre
     *          - outBuffer != nullptr
     * @post
     *          - 戻り値 n について、n >= 0
     */
    int RemoveKeyCodes(uint16_t outBuffer[], int count) NN_NOEXCEPT;

    /**
     * @brief   組み立て中キーコード列の未確定部分全てを明示的に確定させます。
     *
     * @details 追加中の Alt コードが存在する場合、これをキーコード列の
     *          末尾に追加した後、全てのキーコードを確定させます。
     *
     * @post
     *          - GetCompositionCount() == GetOutputCount()
     */
    void FlushKeyCodes() NN_NOEXCEPT;

    /**
     * @brief   組み立て中キーコード列の末尾に Alt コードを追加します。
     *
     * @details Alt コードは組み立て中キーコード列とは別に管理され、
     *          確定した時点で組み立て中キーコード列の末尾に追加されます。
     *          Alt コードは EnqueueKeyCodes() か FlushKeyCodes() か
     *          FlushAltCodes() が呼出された時点で確定します。
     *          組み立て中キーコード列に確定部分が存在する場合は Alt コードの
     *          追加は行われません。 GetOutputCount() が 0 になるまで
     *          DequeueOutputs() を呼び出して確定部分を取り出すことで
     *          Alt コードの追加が可能になります。
     *
     * @param[in]       buffer      Alt コード列
     * @param[in]       count       buffer の長さ
     *
     * @return  追加した Alt コードの数です。
     *
     * @pre
     *          - buffer != nullptr
     *          - 0 <= i && i < count を満たす i について、
     *            0 <= buffer[i] && buffer[i] <= 9
     * @post
     *          - 戻り値 n について、n >= 0
     */
    int EnqueueAltCodes(const int buffer[], int count) NN_NOEXCEPT;

    /**
     * @brief   Alt コードを明示的に確定させます。
     *
     * @details 追加中の Alt コードが存在するならば、これを確定させて
     *          組み立て中キーコード列の末尾に追加します。
     */
    void FlushAltCodes() NN_NOEXCEPT;

    /**
     * @brief   組み立て中キーコード列を破棄します。
     *
     * @details Alt コードを含む組み立て中キーコード列を破棄します。
     *
     * @post
     *          - GetCompositionCount() == 0
     *          - GetOutputCount() == 0
     */
    void Clear() NN_NOEXCEPT;
};

/**
 * @brief   Keyboard の HID Usage ID からキーコードを取得します。
 *
 * @details キーコードは UTF-16 符号で表現されます。
 *
 * @param[out]  pOutValue                   キーコードを読み出すバッファ
 * @param[in]   usageId                     Keyboard の HID Usage ID
 * @param[in]   modifiers                   Keyboard の修飾情報
 * @param[in]   layout                      Keyboard のレイアウト
 *
 * @pre
 *              - pOutValue != nullptr
 *              - 0x0000 <= usageId && usageId <= 0xffff
 *              - layout は有効なレイアウト
 */
void GetKeyCode(uint16_t* pOutValue,
                int usageId,
                KeyboardModifierSet modifiers,
                KeyboardLayoutType layout) NN_NOEXCEPT;

}} // namespace nn::hid
