﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       JoyXpad に関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/hid/hid_AnalogStickState.h>
#include <nn/util/util_BitFlagSet.h>

namespace nn { namespace hid {

/**
 * @brief       Xpad ID を表す構造体です。
 *
 * @details     Joy-Con を参照するための Xpad IDです。
 */
struct JoyXpadId
{
    int32_t _storage;
};

/**
 * @brief       Joy-Con のデジタルボタン定義です。 Joy-Con (R) / Joy-Con (L) 共通です。
 */
struct JoyXpadButton
{
    typedef ::nn::util::BitFlagSet<32, JoyXpadButton>::Flag<0> A;          //!< Joy-Con の a ボタン
    typedef ::nn::util::BitFlagSet<32, JoyXpadButton>::Flag<1> B;          //!< Joy-Con の b ボタン
    typedef ::nn::util::BitFlagSet<32, JoyXpadButton>::Flag<2> X;          //!< Joy-Con の x ボタン
    typedef ::nn::util::BitFlagSet<32, JoyXpadButton>::Flag<3> Y;          //!< Joy-Con の y ボタン
    typedef ::nn::util::BitFlagSet<32, JoyXpadButton>::Flag<4> StickL;     //!< Joy-Con の L スティックボタン
    typedef ::nn::util::BitFlagSet<32, JoyXpadButton>::Flag<5> StickR;     //!< Joy-Con の R スティックボタン
    typedef ::nn::util::BitFlagSet<32, JoyXpadButton>::Flag<6> L;          //!< Joy-Con の L ボタン
    typedef ::nn::util::BitFlagSet<32, JoyXpadButton>::Flag<7> R;          //!< Joy-Con の R ボタン
    typedef ::nn::util::BitFlagSet<32, JoyXpadButton>::Flag<8> ZL;         //!< Joy-Con の ZL ボタン
    typedef ::nn::util::BitFlagSet<32, JoyXpadButton>::Flag<9> ZR;         //!< Joy-Con の ZR ボタン
    typedef ::nn::util::BitFlagSet<32, JoyXpadButton>::Flag<10> Plus;      //!< Joy-Con の Start ボタン
    typedef ::nn::util::BitFlagSet<32, JoyXpadButton>::Flag<11> Minus;     //!< Joy-Con の Select ボタン
    typedef ::nn::util::BitFlagSet<32, JoyXpadButton>::Flag<12> Left;      //!< Joy-Con の十字ボタン 左
    typedef ::nn::util::BitFlagSet<32, JoyXpadButton>::Flag<13> Up;        //!< Joy-Con の十字ボタン 上
    typedef ::nn::util::BitFlagSet<32, JoyXpadButton>::Flag<14> Right;     //!< Joy-Con の十字ボタン 右
    typedef ::nn::util::BitFlagSet<32, JoyXpadButton>::Flag<15> Down;      //!< Joy-Con の十字ボタン 下
    typedef ::nn::util::BitFlagSet<32, JoyXpadButton>::Flag<16> LeftSL;    //!< Joy-Con (L) の SL ボタン
    typedef ::nn::util::BitFlagSet<32, JoyXpadButton>::Flag<17> LeftSR;    //!< Joy-Con (L) の SR ボタン
    typedef ::nn::util::BitFlagSet<32, JoyXpadButton>::Flag<18> RightSL;   //!< Joy-Con (R) の SL ボタン
    typedef ::nn::util::BitFlagSet<32, JoyXpadButton>::Flag<19> RightSR;   //!< Joy-Con (R) の SR ボタン
};

/**
 * @brief       Joy-Con のデジタルボタンの集合を扱う型です。 Joy-Con (R) / Joy-Con (L) 共通です。
 */
typedef ::nn::util::BitFlagSet<32, JoyXpadButton> JoyXpadButtonSet;

/**
 * @brief       Joy-Con の入力状態を表す構造体です。 Joy-Con (R) ・ Joy-Con (L) 共通です。
 */
struct JoyXpadState
{
    int64_t samplingNumber;         //!< Joy-Con の入力状態が更新される度に増加する値です。
    NN_PADDING4;
    JoyXpadButtonSet buttons;       //!< Joy-Con のデジタルボタンの状態です。
    AnalogStickState analogStick;   //!< Joy-Con のアナログスティックの状態です。
};

//! @name JoyXpad (Joy-Con) 関連 API
//! @{

/**
 * @brief       使用可能な Joy-Con の Xpad ID を取得します。
 *
 * @details     取得される Xpad ID の数、値、順番は常に一定です。
 *              使用可能な Joy-Con の数より大きなバッファ（配列）が指定された場合、余った領域に対しては何も行いません。
 *              任意のタイミングで呼び出しに成功します。
 *
 * @deprecated  Xpad は廃止されました。 Npad を利用してください。
 *
 * @param[out]  outXpadIds                  Xpad ID を読み出すバッファ（配列）
 * @param[in]   count                       読み出す Xpad ID の数
 *
 * @return      読み出した Xpad ID の数です。
 *
 * @pre
 *              - outXpadIds != nullptr
 *              - count >= 0
 * @post
 *              - 戻り値 n について、n >= 0
 *              - 読み出した outXpadIds[i] は有効な Xpad ID
 */
NN_DEPRECATED
int GetXpadIds(JoyXpadId* outXpadIds, int count) NN_NOEXCEPT;

/**
 * @brief       Xpad のプレイヤー番号を取得します。
 *
 * @deprecated  Xpad は廃止されました。 Npad を利用してください。
 *
 * @details     Xpad ID で指定した Xpad のプレイヤー番号を取得します。
 *              プレイヤー番号は 0 オリジンです。
 *              Xpad ID の異なる Xpad が同じ値を返すことはありません。
 *              同じ Xpad ID に対しては常に同じ結果となります。
 *
 * @param[in]   xpadId                      Xpad ID
 *
 * @return      Xpad のプレイヤー番号です。
 *
 * @post
 *              - 戻り値 n について、n >= 0
 */
NN_DEPRECATED
int GetXpadPlayerNumber(const JoyXpadId& xpadId) NN_NOEXCEPT;

/**
 * @brief       JoyXpad を初期化します。
 *
 * @deprecated  Xpad は廃止されました。 Npad を利用してください。
 *
 * @details     Xpad ID で指定した JoyXpad を初期化します。
 *
 * @param[in]   xpadId                      Xpad ID
 */
NN_DEPRECATED
void InitializeXpad(const JoyXpadId& xpadId) NN_NOEXCEPT;

/**
 * @brief       Joy-Con (R) の最新の入力状態を取得します。
 *
 * @details     指定の Xpad ID と対応する Joy-Con (R) から GetJoyXpadRightStates() で 1 つの入力状態を読み出した時と同じ値が返ります。
 *              最低 1 回の入力状態の更新が保証されるため、 InitializeXpad() の呼び出し完了直後から利用可能です。
 *
 * @deprecated  Xpad は廃止されました。 Npad を利用してください。
 *
 * @param[out]  pOutValue                   入力状態を読み出すバッファ
 * @param[in]   xpadId                      Xpad ID
 *
 * @pre
 *              - 指定の Xpad ID について InitializeXpad() の呼び出しが完了している
 *              - pOutValue != nullptr
 * @post
 *              - アナログスティックの入力状態 pOutValue->analogStick は下記の条件を満たす円の内側（境界含む）に位置
 *                  - 原点は (0, 0)
 *                  - 半径は AnalogStickMax
 */
NN_DEPRECATED
void GetJoyXpadRightState(JoyXpadState* pOutValue, const JoyXpadId& xpadId
                          ) NN_NOEXCEPT;

/**
 * @brief       Joy-Con (R) の入力状態を過去に遡って読み出します。
 *
 * @details     最新のものから過去に遡って利用可能な数だけ順に、指定の Xpad ID と対応する Joy-Con (R) から入力状態を読み出します。
 *              利用可能な入力状態の数より大きなバッファ（配列）が指定された場合、余った領域に対しては何も行いません。
 *              読み出し可能な入力状態の最大数は XpadStateCountMax 個です。
 *              利用可能な入力状態には読み出し済みのものも含まれます。
 *              差分だけを利用したい場合は JoyXpadState::samplingNumber を参照してください。
 *              再解決の発生によって Xpad ID が解決状態でなくなった場合であっても、
 *              Joy-Con (R) の入力状態は無入力（デジタルボタンの押下は無く、アナログスティックはニュートラル位置）となりますが、
 *              値の読み出し自体は継続して実施可能です。
 *
 * @deprecated  Xpad は廃止されました。 Npad を利用してください。
 *
 * @param[out]  pOutStates                  入力状態を読み出すバッファ（配列）
 * @param[in]   count                       読み出す入力状態の数
 * @param[in]   xpadId                      Xpad ID
 *
 * @return      読み出した入力状態の数を返します。
 *
 * @pre
 *              - 指定の Xpad ID について InitializeXpad() の呼び出しが完了している
 *              - pOutStates != nullptr
 *              - count >= 0
 * @post
 *              - 戻り値 n について、n >= 0
 *              - 読み出した入力状態 pOutStates[i] について
 *                  - アナログスティックの入力状態 pOutStates[i].analogStick は下記の条件を満たす円の内側（境界含む）に位置
 *                      - 原点は (0, 0)
 *                      - 半径は AnalogStickMax
 */
NN_DEPRECATED
int GetJoyXpadRightStates(JoyXpadState* pOutStates,
                         int count,
                         const JoyXpadId& xpadId) NN_NOEXCEPT;

/**
 * @brief       Joy-Con (L) の最新の入力状態を取得します。
 *
 * @details     指定の Xpad ID と対応する Joy-Con (L) から GetJoyXpadRightStates() で 1 つの入力状態を読み出した時と同じ値が返ります。
 *              最低 1 回の入力状態の更新が保証されるため、 InitializeXpad() の呼び出し完了直後から利用可能です。
 *
 * @deprecated  Xpad は廃止されました。 Npad を利用してください。
 *
 * @param[out]  pOutValue                   入力状態を読み出すバッファ
 * @param[in]   xpadId                      Xpad ID
 *
 * @pre
 *              - 指定の Xpad ID について InitializeXpad() の呼び出しが完了している
 *              - pOutValue != nullptr
 * @post
 *              - アナログスティックの入力状態 pOutValue->analogStick は下記の条件を満たす円の内側（境界含む）に位置
 *                  - 原点は (0, 0)
 *                  - 半径は AnalogStickMax
 */
NN_DEPRECATED
void GetJoyXpadLeftState(JoyXpadState* pOutValue, const JoyXpadId& xpadId
                         ) NN_NOEXCEPT;

/**
 * @brief       Joy-Con (L) の入力状態を過去に遡って読み出します。
 *
 * @details     最新のものから過去に遡って利用可能な数だけ順に、指定の Xpad ID と対応する Joy-Con (L) から入力状態を読み出します。
 *              利用可能な入力状態の数より大きなバッファ（配列）が指定された場合、余った領域に対しては何も行いません。
 *              読み出し可能な入力状態の最大数は XpadStateCountMax 個です。
 *              利用可能な入力状態には読み出し済みのものも含まれます。
 *              差分だけを利用したい場合は JoyXpadState::samplingNumber を参照してください。
 *              再解決の発生によって Xpad ID が解決状態でなくなった場合であっても、
 *              Joy-Con (L) の入力状態は無入力（デジタルボタンの押下は無く、アナログスティックはニュートラル位置）となりますが、
 *              値の読み出し自体は継続して実施可能です。
 *
 * @deprecated  Xpad は廃止されました。 Npad を利用してください。
 *
 * @param[out]  pOutStates                  入力状態を読み出すバッファ（配列）
 * @param[in]   count                       読み出す入力状態の数
 * @param[in]   xpadId                      Xpad ID
 *
 * @return      読み出した入力状態の数を返します。
 *
 * @pre
 *              - 指定の Xpad ID について InitializeXpad() の呼び出しが完了している
 *              - pOutStates != nullptr
 *              - count >= 0
 * @post
 *              - 戻り値 n について、n >= 0
 *              - 読み出した入力状態 pOutStates[i] について
 *                  - アナログスティックの入力状態 pOutStates[i].analogStick は下記の条件を満たす円の内側（境界含む）に位置
 *                      - 原点は (0, 0)
 *                      - 半径は AnalogStickMax
 */
NN_DEPRECATED
int GetJoyXpadLeftStates(JoyXpadState* pOutStates,
                        int count,
                        const JoyXpadId& xpadId) NN_NOEXCEPT;

//! @}

}} // namespace nn::hid
