﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/hid/hid_AnalogStickState.h>
#include <nn/hid/system/hid_CommonTypes.h>

namespace nn { namespace hid { namespace detail {

/**
 * @brief       抽象化 Pad の Id
 */
struct AbstractedPadId
{
    uint64_t value;

    AbstractedPadId() NN_NOEXCEPT
        : value(0)
    {
        // 何もしない
    }

    bool operator == (const AbstractedPadId& id) const NN_NOEXCEPT
    {
        return (value == id.value);
    }

    bool operator != (const AbstractedPadId& id) const NN_NOEXCEPT
    {
        return (value != id.value);
    }

    //!< アクティブ化回数をインクリメントします。
    uint64_t operator++() NN_NOEXCEPT
    {
        return ++value;
    }
};

/**
  * @brief     抽象化 Pad の種類
  */
enum AbstractedPadType
{
    AbstractedPadType_Xcd,              // Xcd に依存する抽象化 Pad
    AbstractedPadType_Usb,              // DirectInput 対応の Usb コントローラー に依存する抽象化 Pad
    AbstractedPadType_Generic,          // Win32 向け XInput 対応コントローラーに依存する抽象化 Pad
    AbstractedPadType_AutoPilot,        // 自動入力 (AutoPilot) に対応した抽象化 Pad
    AbstractedPadType_GcAdapter,        // Gc コントローラーアダプターに依存する抽象化 Pad
    AbstractedPadType_Palma,            // Xcd ベースの Palma に対応する抽象化 Pad
    AbstractedPadType_AutopilotVirtual, // DirectInput 対応の仮想コントローラーに依存する抽象化 Pad
    AbstractedPadType_Fifty,            // Fifty に対応した抽象化 Pad
};

/**
 * @brief       コントローラーに搭載される機能
 */
struct AbstractedPadFeature
{
    typedef ::nn::util::BitFlagSet<32, AbstractedPadFeature>::Flag<0>  AnalogStick;              //!< アナログスティック
    typedef ::nn::util::BitFlagSet<32, AbstractedPadFeature>::Flag<1>  SixAxisSensorCenter;      //!< 6軸センサー 中央
    typedef ::nn::util::BitFlagSet<32, AbstractedPadFeature>::Flag<2>  SixAxisSensorLeft;        //!< 6軸センサー 左
    typedef ::nn::util::BitFlagSet<32, AbstractedPadFeature>::Flag<3>  SixAxisSensorRight;       //!< 6軸センサー 右
    typedef ::nn::util::BitFlagSet<32, AbstractedPadFeature>::Flag<4>  LraLeft;                  //!< Lra 左
    typedef ::nn::util::BitFlagSet<32, AbstractedPadFeature>::Flag<5>  LraRight;                 //!< Lra 左
    typedef ::nn::util::BitFlagSet<32, AbstractedPadFeature>::Flag<6>  Nfc;                      //!< Nfc
    typedef ::nn::util::BitFlagSet<32, AbstractedPadFeature>::Flag<7>  IrSensor;                 //!< IrSensor
    typedef ::nn::util::BitFlagSet<32, AbstractedPadFeature>::Flag<8>  AnalogStickUserCal;       //!< AnalogStick の User Calibration
    typedef ::nn::util::BitFlagSet<32, AbstractedPadFeature>::Flag<9>  SixAxisSensorUserCal;     //!< 6軸センサーの User Calibration
    typedef ::nn::util::BitFlagSet<32, AbstractedPadFeature>::Flag<10> GcErm;                    //!< Gc コントローラー用の Erm
    typedef ::nn::util::BitFlagSet<32, AbstractedPadFeature>::Flag<11> RailAttachmentLeft;      //!< 左 Joy-Con レールのアタッチメント機能
    typedef ::nn::util::BitFlagSet<32, AbstractedPadFeature>::Flag<12> RailAttachmentRight;     //!< 右 Joy-Con レールのアタッチメント機能

    static ::nn::util::BitFlagSet<32, AbstractedPadFeature> GetSixAxisSensorMask()
    {
        return SixAxisSensorCenter::Mask | SixAxisSensorLeft::Mask | SixAxisSensorRight::Mask;
    }
};


/**
 * @brief       コントローラーに搭載された機能の集合
 */
typedef ::nn::util::BitFlagSet<32, AbstractedPadFeature> AbstractedPadFeatureSet;

/**
 * @brief       コントローラーのボタンの抽象化定義
 */
struct AbstractedPadButton
{
    typedef ::nn::util::BitFlagSet<32, AbstractedPadButton>::Flag<0> A;      //!< コントローラーの a ボタン
    typedef ::nn::util::BitFlagSet<32, AbstractedPadButton>::Flag<1> B;      //!< コントローラーの b ボタン
    typedef ::nn::util::BitFlagSet<32, AbstractedPadButton>::Flag<2> X;      //!< コントローラーの x ボタン
    typedef ::nn::util::BitFlagSet<32, AbstractedPadButton>::Flag<3> Y;      //!< コントローラーの y ボタン
    typedef ::nn::util::BitFlagSet<32, AbstractedPadButton>::Flag<4> StickL; //!< コントローラーの L スティックボタン
    typedef ::nn::util::BitFlagSet<32, AbstractedPadButton>::Flag<5> StickR; //!< コントローラーの R スティックボタン
    typedef ::nn::util::BitFlagSet<32, AbstractedPadButton>::Flag<6> L;      //!< コントローラーの L ボタン
    typedef ::nn::util::BitFlagSet<32, AbstractedPadButton>::Flag<7> R;      //!< コントローラーの R ボタン
    typedef ::nn::util::BitFlagSet<32, AbstractedPadButton>::Flag<8> ZL;     //!< コントローラーの ZL ボタン
    typedef ::nn::util::BitFlagSet<32, AbstractedPadButton>::Flag<9> ZR;     //!< コントローラーの ZR ボタン
    typedef ::nn::util::BitFlagSet<32, AbstractedPadButton>::Flag<10> Start; //!< コントローラーの Start ボタン
    typedef ::nn::util::BitFlagSet<32, AbstractedPadButton>::Flag<11> Select;//!< コントローラーの Select ボタン
    typedef ::nn::util::BitFlagSet<32, AbstractedPadButton>::Flag<12> Left;  //!< コントローラーの 方向ボタン 左
    typedef ::nn::util::BitFlagSet<32, AbstractedPadButton>::Flag<13> Up;    //!< コントローラーの 方向ボタン 上
    typedef ::nn::util::BitFlagSet<32, AbstractedPadButton>::Flag<14> Right; //!< コントローラーの 方向ボタン 右
    typedef ::nn::util::BitFlagSet<32, AbstractedPadButton>::Flag<15> Down;  //!< コントローラーの 方向ボタン 下
    typedef ::nn::util::BitFlagSet<32, AbstractedPadButton>::Flag<16> SL;    //!< コントローラーの SL ボタン
    typedef ::nn::util::BitFlagSet<32, AbstractedPadButton>::Flag<17> SR;    //!< コントローラーの SR ボタン
    typedef ::nn::util::BitFlagSet<32, AbstractedPadButton>::Flag<18> Home;  //!< コントローラーの Home ボタン
    typedef ::nn::util::BitFlagSet<32, AbstractedPadButton>::Flag<19> Shot;  //!< コントローラーの Shot ボタン
    typedef ::nn::util::BitFlagSet<32, AbstractedPadButton>::Flag<20> StickLLeft;   //!< 左スティックによる十字ボタンエミュレーション 左
    typedef ::nn::util::BitFlagSet<32, AbstractedPadButton>::Flag<21> StickLUp;     //!< 左スティックによる十字ボタンエミュレーション 上
    typedef ::nn::util::BitFlagSet<32, AbstractedPadButton>::Flag<22> StickLRight;  //!< 左スティックによる十字ボタンエミュレーション 右
    typedef ::nn::util::BitFlagSet<32, AbstractedPadButton>::Flag<23> StickLDown;   //!< 左スティックによる十字ボタンエミュレーション 下
    typedef ::nn::util::BitFlagSet<32, AbstractedPadButton>::Flag<24> StickRLeft;   //!< 右スティックによる十字ボタンエミュレーション 左
    typedef ::nn::util::BitFlagSet<32, AbstractedPadButton>::Flag<25> StickRUp;     //!< 右スティックによる十字ボタンエミュレーション 上
    typedef ::nn::util::BitFlagSet<32, AbstractedPadButton>::Flag<26> StickRRight;  //!< 右スティックによる十字ボタンエミュレーション 右
    typedef ::nn::util::BitFlagSet<32, AbstractedPadButton>::Flag<27> StickRDown;   //!< 右スティックによる十字ボタンエミュレーション 下
    typedef ::nn::util::BitFlagSet<32, AbstractedPadButton>::Flag<28> Palma;  //!< Palma 専用ボタン
};


/**
 * @brief       デジタルボタンの集合を扱う型です。
 */
typedef ::nn::util::BitFlagSet<32, AbstractedPadButton> AbstractedPadButtonSet;

/**
 * @brief       ボタン, アナログスティック
 */
struct AbstractedPadState
{
    int64_t sampleNumber;             //!< サンプル番号
    AbstractedPadButtonSet buttons;   //!< デジタルボタンの状態です。
    AnalogStickState analogStickL;    //!< 左アナログスティックの状態です。
    AnalogStickState analogStickR;    //!< 右アナログスティックの状態です。
};

}}} // namespace nn::hid::detail
