﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/audio/audio_SampleFormat.h>
#include <nn/capsrv/capsrv_AlbumFileDescription.h>

namespace nn { namespace grc {

struct ContinuousRecordingParameter
{
    ncm::ApplicationId applicationId = {};
    applet::AppletResourceUserId aruid = {};
    uint32_t videoBitRate = 8000000;
    uint32_t maxTimeInMilliSeconds = 20000;
    uint32_t minTimeInMilliSeconds = 1000;
    uint16_t frameCountBetweenIDR = 30;
    uint8_t fps = 30;
    bool forDebug = false;
    uint64_t recordBufferSize = 0;
};

struct ContinuousRecordingFlushParameter
{
    bool isCopyrightImageComposited = false;
    bool overrides = false;
    uint8_t imageOrientation = 0;
    Bit8 _padding0[1];

    struct OverrideParameter
    {
        TimeSpanType maxTime = {};
        TimeSpanType minTime = {};
        bool cutsPrevious = true;
        Bit8 _padding[3];
    };

    // valid only if overrides == true
    OverrideParameter overrideParameter;

    char _reserved[32];
};

NN_STATIC_ASSERT(sizeof(ContinuousRecordingFlushParameter) == 64);

struct OffscreenRecordingParameter
{
public:
    static OffscreenRecordingParameter GetDefaultValue() NN_NOEXCEPT
    {
        OffscreenRecordingParameter param = {};
        param.programId         = {0};
        param.applicationId     = {0};
        param.description       = nn::capsrv::AlbumFileDescription_MovieMakerSaved;
        //param.description       = nn::capsrv::AlbumFileDescription_MovieContinuous;
        param.videoBitRate      = 8000000;
        param.videoWidth        =    1280;
        param.videoHeight       =     720;
        param.videoFrameRate    =      30;
        param.videoFrameCountBetweenIdr = 30;
        param.videoImageOrientation = 0;
        param.audioBitRate      = 1536000;
        param.audioSampleRate   =   48000;
        param.audioChannelCount =       2;
        param.audioFormat       = nn::audio::SampleFormat_PcmInt16;
        return param;
    }

    // 設定可能なパラメータかチェックする。
    // 設定可能であれば true を返す。
    bool CheckParameterValid() const NN_NOEXCEPT
    {
        return CheckProgramIdValid()
            && CheckApplicationIdValid()
            && CheckDescriptionValid()
            && CheckVideoBitRateValid()
            && CheckVideoWidthAndHeightValid()
            && CheckVideoFrameRateValid()
            && CheckVideoFrameCountBetweenIdrValid()
            && CheckAudioBitRateValid()
            && CheckAudioSampleRateValid()
            && CheckAudioChannelCountValid()
            && CheckAudioFormatValid()
            && CheckReservedZero()
            ;
    }

    bool CheckProgramIdValid() const NN_NOEXCEPT
    {
        return true;
    }

    bool CheckApplicationIdValid() const NN_NOEXCEPT
    {
        return true;
    }

    bool CheckDescriptionValid() const NN_NOEXCEPT
    {
        return this->description == nn::capsrv::AlbumFileDescription_MovieMakerSaved;
    }

    bool CheckVideoBitRateValid() const NN_NOEXCEPT
    {
        return this->videoBitRate > 0;
    }

    bool CheckVideoWidthAndHeightValid() const NN_NOEXCEPT
    {
        return (this->videoWidth == 1280 && this->videoHeight == 720)
            || (this->videoWidth == 1920 && this->videoHeight == 1080)
            ;
    }

    bool CheckVideoFrameRateValid() const NN_NOEXCEPT
    {
        return (this->videoFrameRate == 30)
            || (this->videoFrameRate == 60)
            ;
    }

    bool CheckVideoFrameCountBetweenIdrValid() const NN_NOEXCEPT
    {
        return this->videoFrameCountBetweenIdr >= 1;
    }

    bool CheckAudioBitRateValid() const NN_NOEXCEPT
    {
        return true;
    }

    bool CheckAudioSampleRateValid() const NN_NOEXCEPT
    {
        return this->audioSampleRate > 0;
    }

    bool CheckAudioChannelCountValid() const NN_NOEXCEPT
    {
        return this->audioChannelCount == 2;
    }

    bool CheckAudioFormatValid() const NN_NOEXCEPT
    {
        return this->audioFormat == nn::audio::SampleFormat_PcmInt16;
    }

    bool CheckReservedZero() const NN_NOEXCEPT
    {
        for(size_t i = 0; i < sizeof(this->_reserved); i++)
        {
            if(this->_reserved[i] != 0)
            {
                return false;
            }
        }
        return true;
    }

public:
    ncm::ProgramId programId = {};
    ncm::ApplicationId applicationId = {};

    nn::capsrv::AlbumFileDescriptionType description = {};

    // エンコード後のビデオのビットレート[bps]
    uint32_t videoBitRate;

    // エンコード後のビデオの幅[px]
    uint32_t videoWidth;

    // エンコード後のビデオの高さ[px]
    uint32_t videoHeight;

    // エンコード後のビデオのフレームレート[fps]
    uint32_t videoFrameRate;

    // エンコード後のビデオのキーフレーム間隔[frame]。キーフレーム自体を含む。
    uint32_t videoFrameCountBetweenIdr;

    // エンコード後のオーディオのビットレート[bps]
    uint32_t audioBitRate;

    // エンコード前のオーディオのサンプルレート[sample/sec]
    uint32_t audioSampleRate;

    // オーディオのチャンネル数
    uint32_t audioChannelCount;

    // オーディオのサンプルのフォーマット。nn::audio::SampleFormet の値。
    uint32_t audioFormat;

    // ビデオの回転方向（album::ImageOrientation 型）
    uint32_t videoImageOrientation;

    // 予約領域。ゼロ埋めのこと。
    char _reserved[68];
};

}}
