﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/ens/ens_Types.h>
#include <nn/time/time_PosixTime.h>

namespace nn { namespace ens {

/**
 * @brief メッセージ 1 通を一意に識別する ID を表す型
 *
 * @details
 */
struct MessageId
{
    uint64_t value; //!< メッセージ ID 値

    /**
     * @brief   nn::ens::MessageId 同士の等価演算子のオーバーロードです。
     *
     * @param[in]   lhs 比較する一方を指定します。
     * @param[in]   rhs 比較する他方を指定します。
     *
     * @return  両者が同じなら true を、異なるなら false を返します。
     *
     * @details
     */
    friend bool operator==(const MessageId& lhs, const MessageId& rhs) NN_NOEXCEPT
    {
        return lhs.value == rhs.value;
    }

    /**
     * @brief   nn::ens::MessageId 同士の不等価演算子のオーバーロードです。
     *
     * @param[in]   lhs 比較する一方を指定します。
     * @param[in]   rhs 比較する他方を指定します。
     *
     * @return  両者が異なるなら true を、同じなら false を返します。
     *
     * @details
     */
    friend bool operator!=(const MessageId& lhs, const MessageId& rhs) NN_NOEXCEPT
    {
        return !(lhs == rhs);
    }
};

/**
 * @brief   無効なメッセージ ID
 *
 * @details
 */
static const MessageId InvalidMessageId = {0ull};

/**
 * @brief メッセージのヘッダーを表す型
 *
 * @details
 *
 * @see nn::ens::ReceiveMessageHeaderList()
 */
struct MessageHeader
{
    MessageId id;               //!< メッセージ ID
    UserId sender;              //!< メッセージ送信者のユーザー ID
    nn::time::PosixTime sentAt; //!< メッセージの送信時刻
    Digest digest;              //!< ダイジェスト
    ReceiveBuffer metadata;     //!< メタデータ
};

/**
 * @brief 通知データを表す型
 *
 * @details
 *
 * @see nn::ens::PopNotificationData()
 */
struct NotificationData
{
    static const size_t PayloadSizeMax = 2048; //!< 通知ペイロードの最大サイズ

    char type[16];                    //!< 通知の種類を表す文字列
    UserId receiver;                  //!< 通知受信者を表すユーザー ID
    nn::Bit8 payload[PayloadSizeMax]; //!< 通知ペイロードを格納するバッファ
    size_t payloadSize;               //!< 通知ペイロードのサイズ
};

/**
 * @brief   検索クエリの最大長
 *
 * @details
 *  検索機能を有する API に指定する検索クエリの最大長です。
 *
 * @see nn::ens::GetMyDesignHeaderList()
 */
static const size_t SearchQueryLengthMax = 1023;

/**
 * @brief マイデザインを一意に識別する ID を表す型
 *
 * @details
 */
struct MyDesignId
{
    uint64_t value; //!< マイデザイン ID 値

    /**
     * @brief   nn::ens::MyDesignId 同士の等価演算子のオーバーロードです。
     *
     * @param[in]   lhs 比較する一方を指定します。
     * @param[in]   rhs 比較する他方を指定します。
     *
     * @return  両者が同じなら true を、異なるなら false を返します。
     *
     * @details
     */
    friend bool operator==(const MyDesignId& lhs, const MyDesignId& rhs) NN_NOEXCEPT
    {
        return lhs.value == rhs.value;
    }

    /**
     * @brief   nn::ens::MyDesignId 同士の不等価演算子のオーバーロードです。
     *
     * @param[in]   lhs 比較する一方を指定します。
     * @param[in]   rhs 比較する他方を指定します。
     *
     * @return  両者が異なるなら true を、同じなら false を返します。
     *
     * @details
     */
    friend bool operator!=(const MyDesignId& lhs, const MyDesignId& rhs) NN_NOEXCEPT
    {
        return !(lhs == rhs);
    }
};

/**
 * @brief   無効なマイデザイン ID
 *
 * @details
 */
static const MyDesignId InvalidMyDesignId = {0ull};

/**
 * @brief マイデザインの作者を一意に識別する ID を表す型
 *
 * @details
 */
struct MyDesignAuthorId
{
    uint64_t value; //!< マイデザイン作者 ID 値

    /**
     * @brief   nn::ens::MyDesignAuthorId 同士の等価演算子のオーバーロードです。
     *
     * @param[in]   lhs 比較する一方を指定します。
     * @param[in]   rhs 比較する他方を指定します。
     *
     * @return  両者が同じなら true を、異なるなら false を返します。
     *
     * @details
     */
    friend bool operator==(const MyDesignAuthorId& lhs, const MyDesignAuthorId& rhs) NN_NOEXCEPT
    {
        return lhs.value == rhs.value;
    }

    /**
     * @brief   nn::ens::MyDesignAuthorId 同士の不等価演算子のオーバーロードです。
     *
     * @param[in]   lhs 比較する一方を指定します。
     * @param[in]   rhs 比較する他方を指定します。
     *
     * @return  両者が異なるなら true を、同じなら false を返します。
     *
     * @details
     */
    friend bool operator!=(const MyDesignAuthorId& lhs, const MyDesignAuthorId& rhs) NN_NOEXCEPT
    {
        return !(lhs == rhs);
    }
};

/**
 * @brief   無効なマイデザイン作者 ID
 *
 * @details
 */
static const MyDesignAuthorId InvalidMyDesignAuthorId = {0ull};

/**
 * @brief マイデザインの作者名を表す型
 *
 * @details
 *  作者名は UTF-8 でエンコードされます。
 */
struct MyDesignAuthorName
{
    char value[64]; //!< マイデザイン作者名
};

/**
 * @brief マイデザインのヘッダーを表す型
 *
 * @details
 *
 * @see nn::ens::GetMyDesignHeaderList()
 */
struct MyDesignHeader
{
    MyDesignId id;                 //!< マイデザイン ID
    MyDesignAuthorName authorName; //!< マイデザイン作者名
    nn::time::PosixTime updatedAt; //!< マイデザインの更新時刻
    Digest digest;                 //!< ダイジェスト
    ReceiveBuffer metadata;        //!< メタデータ
};

}}
