﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

namespace nn { namespace ens {

/**
 * @brief   ネットワーク機能の利用に必要なメモリの最小サイズ
 *
 * @details
 *
 * @see nn::ens::StartServiceLoop()
 */
static const size_t RequiredMemorySizeMin = 1024 * 1024;

/**
 * @brief   ネットワーク機能のスレッドに必要なスタックサイズ
 *
 * @details
 *
 * @see nn::ens::StartServiceLoop()
 */
static const size_t RequiredStackSize = 1024 * 64;

/**
 * @brief   サービス利用者を一意に特定するための ID を表す型
 *
 * @details
 */
struct UserId
{
    uint64_t value; //!< ユーザー ID 値

    /**
     * @brief   nn::ens::UserId 同士の等価演算子のオーバーロードです。
     *
     * @param[in]   lhs 比較する一方を指定します。
     * @param[in]   rhs 比較する他方を指定します。
     *
     * @return  両者が同じなら true を、異なるなら false を返します。
     *
     * @details
     */
    friend bool operator==(const UserId& lhs, const UserId& rhs) NN_NOEXCEPT
    {
        return lhs.value == rhs.value;
    }

    /**
     * @brief   nn::ens::UserId 同士の不等価演算子のオーバーロードです。
     *
     * @param[in]   lhs 比較する一方を指定します。
     * @param[in]   rhs 比較する他方を指定します。
     *
     * @return  両者が異なるなら true を、同じなら false を返します。
     *
     * @details
     */
    friend bool operator!=(const UserId& lhs, const UserId& rhs) NN_NOEXCEPT
    {
        return !(lhs == rhs);
    }
};

/**
 * @brief   無効なユーザー ID
 *
 * @details
 */
static const UserId InvalidUserId = {0ull};

/**
 * @brief   サービスを利用するための認証情報
 *
 * @details
 *  本構造体には、nn::ens のネットワークサービスを利用するユーザーを一意に決める nn::ens::UserId が含まれています。
 *
 *  同じユーザー ID に対する認証情報は変化することはありません。@n
 *  本構造体をセーブデータなどに永続化し、次回以降の起動へ引き継ぐことで、同じユーザーとしてサービスを利用することができます。
 *
 * @see nn::ens::GenerateCredential()
 */
struct Credential
{
    UserId userId;      //!< ユーザー ID
    char _password[72];

    /**
     * @brief   有効な認証情報かどうかを判定します。
     *
     * @return  有効な認証情報なら true を、そうでないなら false を返します。
     *
     * @details
     */
    bool IsValid() const
    {
        return this->userId != InvalidUserId;
    }
};

/**
 * @brief   無効な認証情報
 *
 * @details
 */
static const Credential InvalidCredential = {InvalidUserId, {}};

/**
 * @brief   ダイジェストを表す型
 *
 * @details
 */
struct Digest
{
    static const size_t Size = 20; //!< ダイジェスト値のサイズ

    static const size_t StringLength = Size * 2; //!< 文字列化した時の文字列長（NULL 終端を含まない）

    nn::Bit8 value[Size]; //!< ダイジェスト値

    /**
     * @brief   nn::ens::Digest 同士の等価演算子のオーバーロードです。
     *
     * @param[in]   lhs 比較する一方を指定します。
     * @param[in]   rhs 比較する他方を指定します。
     *
     * @return  両者が同じなら true を、異なるなら false を返します。
     *
     * @details
     */
    friend bool operator==(const Digest& lhs, const Digest& rhs) NN_NOEXCEPT
    {
        for (size_t i = 0; i < Size; i++)
        {
            if (lhs.value[i] != rhs.value[i])
            {
                return false;
            }
        }

        return true;
    }

    /**
     * @brief   nn::ens::Digest 同士の不等価演算子のオーバーロードです。
     *
     * @param[in]   lhs 比較する一方を指定します。
     * @param[in]   rhs 比較する他方を指定します。
     *
     * @return  両者が異なるなら true を、同じなら false を返します。
     *
     * @details
     */
    friend bool operator!=(const Digest& lhs, const Digest& rhs) NN_NOEXCEPT
    {
        return !(lhs == rhs);
    }

    /**
     * @brief   ダイジェスト値を文字列に変換します。
     *
     * @param[out]  pOut    文字列を格納するバッファ
     * @param[in]   size    pOut のサイズ
     *
     * @return  pOut に指定したバッファ
     *
     * @pre
     *  - pOut != nullptr
     *  - size > StringLength
     *
     * @details
     */
    const char* ToString(char* pOut, size_t size) const NN_NOEXCEPT;

    /**
     * @brief   文字列をダイジェスト値に変換します。
     *
     * @param[out]  pOut    ダイジェストを格納するバッファ
     * @param[in]   pValue  ダイジェスト値の文字列
     *
     * @return  変換に成功したかどうか
     *
     * @pre
     *  - pOut != nullptr
     *  - pValue != nullptr
     *
     * @details
     *  文字列長が StringLength 未満の場合、本関数は失敗します。
     */
    static bool FromString(Digest* pOut, const char* pValue) NN_NOEXCEPT;
};

/**
 * @brief   データの送信に扱うバッファを保持する型
 *
 * @details
 *  本構造体を引数にとる API を利用する前に、 pBuffer, bufferSize にデータを格納してください。@n
 *  bufferSize の上限は UINT32_MAX です。
 */
struct SendBuffer
{
    const void* pBuffer; //!< バッファの先頭ポインタ
    size_t bufferSize;   //!< pBuffer のサイズ

    /**
     * @brief   有効な送信バッファかどうかを判定します。
     *
     * @return  有効な送信バッファなら true を、そうでないなら false を返します。
     *
     * @details
     */
    bool IsValid() const
    {
        return (this->pBuffer == nullptr) ?
            (this->bufferSize == 0) : (this->bufferSize > 0 && this->bufferSize <= UINT32_MAX);
    }

    /**
     * @brief   空の送信バッファかどうかを判定します。
     *
     * @return  空の送信バッファなら true を、そうでないなら false を返します。
     *
     * @details
     */
    bool IsNull() const
    {
        return (this->pBuffer == nullptr && this->bufferSize == 0);
    }
};

/**
 * @brief   空の送信バッファ
 *
 * @details
 */
static const SendBuffer NullSendBuffer = {nullptr, 0};

/**
 * @brief   データの受信に扱うバッファを保持する型
 *
 * @details
 *  本構造体を引数にとる API は受信バッファの確保を行いません。@n
 *  API を利用する前に、 pBuffer, bufferSize に有効なバッファを設定してください。
 *
 *  receivedSize には API 成功時に値が格納されます。
 */
struct ReceiveBuffer
{
    void* pBuffer;       //!< バッファの先頭ポインタ
    size_t bufferSize;   //!< pBuffer のサイズ
    size_t receivedSize; //!< 受信したサイズ

    /**
     * @brief   有効な受信バッファかどうかを判定します。
     *
     * @return  有効な受信バッファなら true を、そうでないなら false を返します。
     *
     * @details
     */
    bool IsValid() const
    {
        return (this->pBuffer == nullptr) ?
            (this->bufferSize == 0) : (this->bufferSize > 0);
    }

    /**
     * @brief   空の受信バッファかどうかを判定します。
     *
     * @return  空の受信バッファなら true を、そうでないなら false を返します。
     *
     * @details
     */
    bool IsNull() const
    {
        return (this->pBuffer == nullptr && this->bufferSize == 0);
    }
};

/**
 * @brief   空の受信バッファ
 *
 * @details
 */
static const ReceiveBuffer NullReceiveBuffer = {nullptr, 0};

}}
