﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/ens/ens_Types.h>
#include <nn/ens/ens_ResultPublic.h>
#include <nn/ens/detail/ens_TypesDetail.h>
#include <nn/os/os_Event.h>

namespace nn { namespace ens {

/**
 * @brief   非同期処理の完了や結果を管理するクラス
 *
 * @details
 *  非同期処理の管理を行うためのクラスです。
 *
 *  1 つの非同期処理につき 1 つの AsyncContext クラスインスタンスが必要です。@n
 *  使いまわしはできません。
 *
 *  @link about_ens_async_api 非同期処理 API の利用について @endlink も参照してください。
 */
class AsyncContext
{
private:
    NN_DISALLOW_COPY(AsyncContext);
    NN_DISALLOW_MOVE(AsyncContext);

public:
    /**
     * @brief   コンストラクタ
     *
     * @details
     */
    AsyncContext() NN_NOEXCEPT;

    /**
     * @brief   デストラクタ
     *
     * @details
     *  非同期処理中に呼ばれた場合には Cancel() 相当の処理を行い、非同期処理の完了までブロックします。
     */
    ~AsyncContext() NN_NOEXCEPT;

    /**
     * @brief   非同期処理の結果を取得します。
     *
     * @retresult
     *  @handleresult{
     *      nn::ResultSuccess,
     *      処理に成功しました。
     *  }
     *  @handleresult{
     *      nn::ens::ResultCanceled,
     *      非同期処理がキャンセルされました。
     *      このエラーはユーザーに通知しないでください。
     *  }
     *  @handleresult{
     *      nn::ens::ResultApplicationSpecificError,
     *      アプリケーション専用のエラーが発生しました。
     *      GetErrorDetail() でエラーの詳細を取得し、エラーハンドリングを行ってください。
     *  }
     * @endretresult
     *
     * @pre
     *  - GetEvent() がシグナル状態
     *
     * @details
     *  非同期処理の結果を取得します。
     *
     *  戻り値に記載されていないものに関しては、適切なエラーハンドリングを行うか nn::err::ShowError() でユーザーに通知してください。@n
     *  なお、バックグラウンドで通信処理を行う等、アプリケーションの進行に影響がない場合、ユーザーに通知する必要はありません。
     *
     * @attention
     *  戻り値の種類は今後増減する可能性があります。
     */
    Result GetResult() const NN_NOEXCEPT;

    /**
     * @brief   エラーの詳細を取得します。
     *
     * @return  エラーの詳細
     *
     * @pre
     *  - GetEvent() がシグナル状態
     *
     * @details
     *  nn::Result だけでは表現できないエラーが発生した時、そのエラーの詳細を返します。
     *
     * @note
     *  エラーの詳細の意味は非同期処理の結果によって異なります。
     */
    int32_t GetErrorDetail() const NN_NOEXCEPT;

    /**
     * @brief   非同期処理の中止を要求します。
     *
     * @details
     *  非同期処理の中止を要求します。
     *
     *  本関数の呼び出し時点では、非同期処理はまだ完了していません。@n
     *  アプリケーションは非同期処理の完了イベントがシグナルされるまで待機する必要があります。
     *
     *  本関数を呼び出した後、非同期処理は可能な限り早く完了します。
     *
     *  非同期処理が既に完了していた場合、非同期処理の結果は変わりません。@n
     *  非同期処理を中止して完了した場合、 GetResult() は nn::ens::ResultCanceled を返します。
     */
    void Cancel() NN_NOEXCEPT;

    /**
     * @brief   非同期処理完了イベントを取得します。
     *
     * @return  非同期処理完了イベント
     *
     * @details
     *  非同期処理の完了時にシグナルされるイベントを取得します。
     *
     *  イベントのクリアモードは nn::os::EventClearMode_ManualClear です。
     */
    nn::os::Event& GetEvent() NN_NOEXCEPT;

public:
    // internal use only
    void Register(const detail::TaskId& taskId,
        detail::CancelObserver cancelObserver, detail::DestructObserver destructObserver) NN_NOEXCEPT;

    // internal use only
    void Complete(nn::Result result) NN_NOEXCEPT;
    // internal use only
    void Complete(nn::Result result, int32_t errorDetail) NN_NOEXCEPT;

private:
    //
    detail::TaskId m_TaskId;
    detail::CancelObserver m_CancelObserver;
    detail::DestructObserver m_DestructObserver;
    //
    nn::Result m_Result;
    int32_t m_ErrorDetail;
    //
    mutable nn::os::Event m_Event;
};

}}
