﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/ens/detail/ens_Common.h>

namespace nn { namespace ens { namespace detail { namespace util {

/**
 * @brief   MessagePack を書き込むクラス
 *
 * @details
 */
class MessagePackWriter
{
public:
    /**
     * @brief   コンストラクタ
     *
     * @details
     */
    MessagePackWriter() NN_NOEXCEPT;

    /**
     * @brief   データの書き込み先バッファを設定します。
     *
     * @param[in]   pBuffer   バッファ
     * @param[in]   size      バッファサイズ
     *
     * @pre
     *  - pBuffer != nullptr
     *  - size > 0
     *
     * @details
     */
    void SetBuffer(void* pBuffer, size_t size) NN_NOEXCEPT;

    /**
     * @brief   内部状態をクリアします。
     *
     * @details
     */
    void Clear() NN_NOEXCEPT;

    /**
     * @brief   連想配列を書き込みます。
     *
     * @param[in]   num 連想配列の要素数
     *
     * @pre
     *  - num <= UINT32_MAX
     *
     * @return  書き込めたかどうか
     *
     * @details
     */
    bool WriteMap(size_t num) NN_NOEXCEPT;

    /**
     * @brief   配列を書き込みます。
     *
     * @param[in]   num 配列の要素数
     *
     * @pre
     *  - num <= UINT32_MAX
     *
     * @return  書き込めたかどうか
     *
     * @details
     */
    bool WriteArray(size_t num) NN_NOEXCEPT;

    /**
     * @brief   キーを書き込みます。
     *
     * @param[in]   pKey    キー
     *
     * @pre
     *  - pKey != nullptr
     *
     * @return  書き込めたかどうか
     *
     * @details
     *
     * @attention
     *  文字列は必ず NULL 終端している必要があります。
     */
    bool WriteKey(const char* pKey) NN_NOEXCEPT;

    /**
     * @brief   値を書き込みます。（空データ）
     *
     * @return  書き込めたかどうか
     *
     * @details
     */
    bool WriteValue(nullptr_t) NN_NOEXCEPT;

    /**
     * @brief   値を書き込みます。（論理値）
     *
     * @param[in]   value   値
     *
     * @return  書き込めたかどうか
     *
     * @details
     */
    bool WriteValue(bool value) NN_NOEXCEPT;

    /**
     * @brief   値を書き込みます。（整数）
     *
     * @param[in]   value   値
     *
     * @return  書き込めたかどうか
     *
     * @details
     */
    bool WriteValue(int8_t value) NN_NOEXCEPT;

    /**
     * @brief   値を書き込みます。（整数）
     *
     * @param[in]   value   値
     *
     * @return  書き込めたかどうか
     *
     * @details
     */
    bool WriteValue(int16_t value) NN_NOEXCEPT;

    /**
     * @brief   値を書き込みます。（整数）
     *
     * @param[in]   value   値
     *
     * @return  書き込めたかどうか
     *
     * @details
     */
    bool WriteValue(int32_t value) NN_NOEXCEPT;

    /**
     * @brief   値を書き込みます。（整数）
     *
     * @param[in]   value   値
     *
     * @return  書き込めたかどうか
     *
     * @details
     */
    bool WriteValue(int64_t value) NN_NOEXCEPT;

    /**
     * @brief   値を書き込みます。（整数）
     *
     * @param[in]   value   値
     *
     * @return  書き込めたかどうか
     *
     * @details
     */
    bool WriteValue(uint8_t value) NN_NOEXCEPT;

    /**
     * @brief   値を書き込みます。（整数）
     *
     * @param[in]   value   値
     *
     * @return  書き込めたかどうか
     *
     * @details
     */
    bool WriteValue(uint16_t value) NN_NOEXCEPT;

    /**
     * @brief   値を書き込みます。（整数）
     *
     * @param[in]   value   値
     *
     * @return  書き込めたかどうか
     *
     * @details
     */
    bool WriteValue(uint32_t value) NN_NOEXCEPT;

    /**
     * @brief   値を書き込みます。（整数）
     *
     * @param[in]   value   値
     *
     * @return  書き込めたかどうか
     *
     * @details
     */
    bool WriteValue(uint64_t value) NN_NOEXCEPT;

    /**
     * @brief   値を書き込みます。（浮動小数点数）
     *
     * @param[in]   value   値
     *
     * @return  書き込めたかどうか
     *
     * @details
     */
    bool WriteValue(float value) NN_NOEXCEPT;

    /**
     * @brief   値を書き込みます。（浮動小数点数）
     *
     * @param[in]   value   値
     *
     * @return  書き込めたかどうか
     *
     * @details
     */
    bool WriteValue(double value) NN_NOEXCEPT;

    /**
     * @brief   値を書き込みます。（文字列）
     *
     * @param[in]   pValue  文字列
     *
     * @pre
     *  - pValue != nullptr
     *
     * @return  書き込めたかどうか
     *
     * @details
     *
     * @attention
     *  文字列は必ず NULL 終端している必要があります。
     */
    bool WriteValue(const char* pValue) NN_NOEXCEPT;

    /**
     * @brief   値を書き込みます。（文字列）
     *
     * @param[in]   pValue  文字列
     * @param[in]   length  文字列の長さ
     *
     * @pre
     *  - pValue != nullptr
     *  - length <= UINT32_MAX
     *
     * @return  書き込めたかどうか
     *
     * @details
     */
    bool WriteValue(const char* pValue, size_t length) NN_NOEXCEPT;

    /**
     * @brief   値を書き込みます。（バイナリ）
     *
     * @param[in]   pValue  バイナリ
     * @param[in]   size    バイナリのサイズ
     *
     * @pre
     *  - pValue != nullptr
     *  - size <= UINT32_MAX
     *
     * @return  書き込めたかどうか
     *
     * @details
     */
    bool WriteValue(const void* pValue, size_t size) NN_NOEXCEPT;

    /**
     * @brief   値を書き込みます。（バイナリ）
     *
     * @param[in]   value   送信バッファ
     *
     * @pre
     *  - value.pBuffer != nullptr
     *  - value.size <= UINT32_MAX
     *
     * @return  書き込めたかどうか
     *
     * @details
     */
    bool WriteValue(const SendBuffer& value) NN_NOEXCEPT;

    /**
     * @brief   値を書き込みます。（拡張データ）
     *
     * @param[in]   type    拡張データの種類
     * @param[in]   pValue  拡張データ
     * @param[in]   size    拡張データのサイズ
     *
     * @pre
     *  - pValue != nullptr
     *  - size <= UINT32_MAX
     *
     * @return  書き込めたかどうか
     *
     * @details
     */
    bool WriteValue(int8_t type, const void* pValue, size_t size) NN_NOEXCEPT;

    /**
     * @brief   キーと値を書き込みます。
     *
     * @tparam      Value   値の型
     *
     * @param[in]   pKey    キー
     * @param[in]   value   値
     *
     * @pre
     *  - pKey != nullptr
     *
     * @return  書き込めたかどうか
     *
     * @details
     */
    template <class... Value>
    bool WriteKeyValue(const char* pKey, Value... value) NN_NOEXCEPT
    {
        NN_DETAIL_ENS_RETURN_IF_FALSE(WriteKey(pKey));
        NN_DETAIL_ENS_RETURN_IF_FALSE(WriteValue(value...));

        return true;
    }

    /**
     * @brief   バッファを取得します。
     *
     * @return  バッファ
     *
     * @details
     */
    void* GetBuffer() const NN_NOEXCEPT;

    /**
     * @brief   書き込んだサイズを取得します。
     *
     * @return  書き込んだサイズ
     *
     * @details
     */
    size_t GetWrittenSize() const NN_NOEXCEPT;

public:
    // internal use only
    struct Stream
    {
        nn::Bit8* pBuffer;
        size_t size;
        size_t written;

        //
        void operator++(int) NN_NOEXCEPT
        {
            NN_SDK_ASSERT_GREATER_EQUAL(size, 1u);

            pBuffer++;
            size--;
            written++;
        }

        //
        void operator+=(size_t step) NN_NOEXCEPT
        {
            NN_SDK_ASSERT_GREATER_EQUAL(size, step);

            pBuffer += step;
            size -= step;
            written += step;
        }
    };

private:
    //
    nn::Bit8* m_pBuffer;
    size_t m_Size;
    //
    Stream m_Stream;

private:
    //
    bool WriteIntegerValue(int64_t value) NN_NOEXCEPT;
    bool WriteIntegerValue(uint64_t value) NN_NOEXCEPT;
};

}}}}
