﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/ens/detail/ens_Common.h>
#include <nn/ens/detail/util/ens_MessagePackInputStream.h>
#include <nn/ens/detail/util/ens_ResponseReader.h>

namespace nn { namespace ens { namespace detail { namespace util {

/**
 * @brief   MessagePack を読み込むクラス
 *
 * @details
 */
class MessagePackReader : public ResponseReader
{
public:
    /**
     * @brief   コンストラクタ
     *
     * @details
     */
    MessagePackReader() NN_NOEXCEPT;

    /**
     * @brief   データの読み込みコールバックを設定します。
     *
     * @param[in]   pCallback   コールバック
     * @param[in]   pParam      コールバックパラメータ
     *
     * @details
     */
    virtual void SetCallback(ReadCallback pCallback, void* pParam) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief   MessagePack を読み込みます。
     *
     * @param[in]   pData   MessagePack データ
     * @param[in]   size    MessagePack データのサイズ
     *
     * @return  妥当な MessagePack を読み込んだかどうか
     *
     * @pre
     *  - pData != nullptr
     *
     * @details
     */
    virtual bool Read(const void* pData, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief   MessagePack を読み込みます。
     *
     * @param[in]   pStream         入力ストリーム
     * @param[in]   pDataBuffer     データ展開用バッファ
     * @param[in]   dataBufferSize  データ展開用バッファのサイズ
     *
     * @return  妥当な MessagePack を読み込んだかどうか
     *
     * @pre
     *  - pStream != nullptr
     *  - (pDataBuffer != nullptr && dataBufferSize > 0) || (pDataBuffer == nullptr && dataBufferSize == 0)
     *
     * @details
     *  データ展開用バッファは、サイズの大きいデータ（文字列やバイナリデータ等）を展開するための一時領域です。@n
     *  MessagePack 内にバッファより大きいデータが存在する場合、MessagePack の読み込みは失敗します。
     *
     *  妥当な MessagePack を読み込む時に MessagePackInputStream::GetDataPointerFromMemoryCache() が常に成功することが期待できる場合、
     *  データ展開用バッファに nullptr を指定しても問題ありません。
     */
    bool Read(MessagePackInputStream* pStream, void* pDataBuffer, size_t dataBufferSize) NN_NOEXCEPT;

private:
    //
    JsonPath m_JsonPath;
    //
    nn::Bit8* m_pDataBuffer;
    size_t m_DataBufferSize;
    //
    ReadCallback m_pCallback;
    void* m_pCallbackParam;

private:
    //
    bool ReadRecursively(MessagePackInputStream* pStream) NN_NOEXCEPT;
    //
    bool ReadString(String* outValue, MessagePackInputStream* pStream, size_t length) NN_NOEXCEPT;
    //
    bool ReadBinary(Binary* outValue, MessagePackInputStream* pStream, size_t length) NN_NOEXCEPT;
    //
    bool ReadExtension(Extension* outValue, MessagePackInputStream* pStream, size_t length) NN_NOEXCEPT;
    //
    bool ReadMap(MessagePackInputStream* pStream, size_t num) NN_NOEXCEPT;
    //
    bool ReadArray(MessagePackInputStream* pStream, size_t num) NN_NOEXCEPT;

    //
    bool DoCallbackForNode(const DataHolder& holder) NN_NOEXCEPT;
    //
    bool DoCallbackForLeaf(const DataHolder& holder) NN_NOEXCEPT;
};

}}}}
