﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/ens/detail/util/ens_MessagePackInputStream.h>

namespace nn { namespace ens { namespace detail { namespace util {

/**
 * @brief   MessagePack 読み込み用の入力ストリーム（メモリ）
 *
 * @details
 */
class MessagePackMemoryInputStream : public MessagePackInputStream
{
public:
    /**
     * @brief   コンストラクタ
     *
     * @details
     */
    MessagePackMemoryInputStream() NN_NOEXCEPT;

    /**
     * @brief   デストラクタ
     *
     * @details
     */
    virtual ~MessagePackMemoryInputStream() NN_NOEXCEPT;

    /**
     * @brief   入力ストリームを開きます。
     *
     * @param[in]   pData   データ
     * @param[in]   size    データのサイズ
     *
     * @details
     */
    void Open(const void* pData, size_t size) NN_NOEXCEPT;

    /**
     * @brief   入力ストリームを閉じます。
     *
     * @details
     */
    void Close() NN_NOEXCEPT;

    /**
     * @brief   データを読み込みます。
     *
     * @param[in]   pBuffer バッファ
     * @param[in]   size    バッファサイズ
     *
     * @return  成功したかどうか
     *
     * @pre
     *  - pBuffer != nullptr
     *  - size > 0
     *
     * @details
     */
    virtual bool Read(void* pBuffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief   メモリ上に展開されているデータのポインタを取得します。
     *
     * @param[out]  ppOutData   データのポインタ
     * @param[in]   size        読み込みたいサイズ
     *
     * @return  読み込みたいサイズのデータがメモリ上に展開されているかどうか
     *
     * @pre
     *  - ppOutData != nullptr
     *  - size > 0
     *
     * @details
     *  妥当な MessagePack を読み込んでいる限り、本関数は常に成功します。
     */
    virtual bool GetDataPointerFromMemoryCache(const void** ppOutData, size_t size) NN_NOEXCEPT NN_OVERRIDE;

private:
    //
    const nn::Bit8* m_pData;
    size_t m_Size;
    //
    size_t m_Position;
};

}}}}
