﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @brief   タスクを作成するマクロ
 *
 * @param[out]  OutContext  非同期コンテキスト
 * @param[in]   Variable    detail::core::Task* の変数名
 * @param[in]   ClassName   タスクのクラス型
 *
 * @details
 *  内部アロケータからメモリを確保し、タスクを配置 new で作成します。@n
 *  メモリを確保できなかった場合、非同期コンテキストに nn::ens::ResultOutOfMemory を設定した後 return します。
 */
#define NN_DETAIL_ENS_CREATE_TASK(OutContext, Variable, ClassName, ...) \
    nn::ens::detail::core::WaitUntilAllocatorCreation();                        \
                                                                                \
    ClassName* Variable = nullptr;                                              \
    do                                                                          \
    {                                                                           \
        void* _p = nn::ens::detail::core::AllocateMemory(sizeof (ClassName));   \
                                                                                \
        if (!_p)                                                                \
        {                                                                       \
            (OutContext)->Complete(nn::ens::ResultOutOfMemory());               \
            return;                                                             \
        }                                                                       \
                                                                                \
        Variable = new (_p) ClassName(__VA_ARGS__);                             \
    }                                                                           \
    while (NN_STATIC_CONDITION(false))

/**
 * @brief   タスクを破棄するマクロ
 *
 * @param[in]   Variable    detail::core::Task* の変数名
 *
 * @details
 *  タスクのデストラクタを呼び出し、内部アロケータにメモリを返却します。
 */
#define NN_DETAIL_ENS_DELETE_TASK(Variable) \
    do                                                  \
    {                                                   \
        (Variable)->~Task();                            \
        nn::ens::detail::core::FreeMemory(Variable);    \
    }                                                   \
    while (NN_STATIC_CONDITION(false))

/**
 * @brief   HTTP コネクションを局所的に確保するマクロ
 *
 * @param[in]   Variable    CURL* の変数名
 *
 * @details
 *  detail::core::HttpTask::GetHttpConnectionProvider() で HTTP コネクションオブジェクトを確保します。@n
 *  オブジェクトを確保できなかった場合、 nn::ens::ResultHttpErrorOutOfMemory を return します。
 *
 *  また、 detail::core::HttpTask::SetHttpCommonOptions() も自動的に呼ばれます。
 */
#define NN_DETAIL_ENS_SCOPED_ACQUIRE_HTTP_CONNECTION(Variable) \
    GetHttpConnectionProvider()->Acquire();                                     \
                                                                                \
    CURL* Variable = GetHttpConnectionProvider()->GetConnection().GetHandle();  \
                                                                                \
    NN_RESULT_THROW_UNLESS(Variable, nn::ens::ResultHttpErrorOutOfMemory());    \
                                                                                \
    NN_UTIL_SCOPE_EXIT                                                          \
    {                                                                           \
        GetHttpConnectionProvider()->Release();                                 \
    };                                                                          \
                                                                                \
    NN_RESULT_DO(SetHttpCommonOptions(pCurl))

/**
 * @brief   リクエストライターのエラーハンドリングを行うマクロ
 *
 * @param[in]   Expression  リクエストライターの実行式
 *
 * @details
 */
#define NN_DETAIL_ENS_REQUEST_WRITER_DO(Expression) \
    NN_RESULT_THROW_UNLESS(Expression, nn::ens::ResultOutOfMemory())

/**
 * @brief   curl API のエラーハンドリングを行うマクロ
 *
 * @param[in]   Expression  CURLcode を返す式
 *
 * @details
 */
#define NN_DETAIL_ENS_CURL_EASY_DO(Expression) \
    do                                                              \
    {                                                               \
        CURLcode _code = (Expression);                              \
                                                                    \
        if (_code != CURLE_OK)                                      \
        {                                                           \
            return nn::ens::detail::util::HandleHttpError(_code);   \
        }                                                           \
    }                                                               \
    while (NN_STATIC_CONDITION(false))

/**
 * @brief   論理値を返す式のエラーハンドリングを行うマクロ
 *
 * @param[in]   Expression  論理値を返す式
 *
 * @details
 */
#define NN_DETAIL_ENS_RETURN_IF_FALSE(Expression) \
    do                                  \
    {                                   \
        bool _result = (Expression);    \
                                        \
        if (!_result)                   \
        {                               \
            return false;               \
        }                               \
    }                                   \
    while (NN_STATIC_CONDITION(false))

#if defined(NN_SDK_BUILD_RELEASE)

#define NN_DETAIL_ENS_DUMP_RESPONSE(Reader, Response, ResponseSize)

#else

/**
 * @brief   レスポンスのダンプを行うマクロ
 *
 * @param[in]   Reader          レスポンスのリーダー
 * @param[in]   Response        レスポンス
 * @param[in]   ResponseSize    レスポンスのサイズ
 *
 * @details
 *  ライブラリ設定で有効化された場合のみ、レスポンスをダンプします。@n
 *  ただし、 Release ビルドの場合、ライブラリ設定に関わらずレスポンスはダンプされません。
 */
#define NN_DETAIL_ENS_DUMP_RESPONSE(Reader, Response, ResponseSize) \
    do                                                                                      \
    {                                                                                       \
        if (nn::ens::detail::LibrarySettings::IsResponseDumpEnabled())                      \
        {                                                                                   \
            nn::ens::detail::util::ResponseReader::Dump<Reader>(Response, ResponseSize);    \
        }                                                                                   \
    }                                                                                       \
    while (NN_STATIC_CONDITION(false))

#endif
