﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/ens/detail/core/ens_HttpTask.h>
#include <nn/ens/detail/core/task/ens_TaskGetAuthToken.h>

namespace nn { namespace ens { namespace detail { namespace core {

/**
 * @brief   認証を行うタスク
 *
 * @details
 *  本クラスを継承して ExecuteSteps() を実装することで、有効な認証トークンが設定された状態でタスクを実行することができます。
 *
 *  タスクが認証トークンの有効期限エラーを返した場合、本クラスは認証トークンの再取得を行い、 ExecuteSteps() を再実行します。@n
 *  ExecuteSteps() は複数回呼ばれる可能性があるため、メンバー変数の破壊的な変更に注意してください。
 *
 *  なお、 Main() には final 属性が設定されるため、 Main() をオーバーライドすることはできません。
 */
class AuthenticationTask : public HttpTask
{
public:
    NN_DETAIL_ENS_DEFINE_RUNTIME_TYPEINFO(AuthenticationTask, HttpTask);

public:
    /**
     * @brief   コンストラクタ
     *
     * @details
     */
    AuthenticationTask() NN_NOEXCEPT;

    /**
     * @brief   デストラクタ
     *
     * @details
     */
    ~AuthenticationTask() NN_NOEXCEPT;

    /**
     * @brief   非同期コンテキストを登録解除します。
     *
     * @details
     */
    virtual void UnregisterAsyncContext() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief   タスクをキャンセルします。
     *
     * @details
     *  別スレッド上で動いているタスクにキャンセル要求を通知します。
     */
    virtual void Cancel() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief   認証情報を取得します。
     *
     * @return  認証情報
     *
     * @details
     */
    const Credential& GetCredential() const NN_NOEXCEPT;

    /**
     * @brief   認証トークンを取得します。
     *
     * @return  認証トークン
     *
     * @details
     */
    const AuthToken& GetAuthToken() const NN_NOEXCEPT;

    /**
     * @brief   認証情報を設定します。
     *
     * @param[in]   credential  認証情報
     *
     * @details
     */
    void SetCredential(const Credential& credential) NN_NOEXCEPT;

    /**
     * @brief   認証トークンを設定します。
     *
     * @param[in]   authToken   認証トークン
     *
     * @details
     */
    void SetAuthToken(const AuthToken& authToken) NN_NOEXCEPT;

protected:
    //
    virtual nn::Result Main() NN_NOEXCEPT NN_OVERRIDE final;

protected:
    /**
     * @brief   認証トークンを必要とするタスクのエントリーポイントです。
     *
     * @return  処理結果
     *
     * @details
     */
    virtual nn::Result ExecuteSteps() NN_NOEXCEPT = 0;

private:
    //
    detail::core::task::TaskGetAuthToken m_TaskGetAuthToken;
    AsyncContext m_ContextForGetAuthToken;
    //
    Credential m_Credential;
    //
    AuthToken m_AuthToken;

private:
    //
    nn::Result StepGetAuthToken() NN_NOEXCEPT;
};

}}}}
