﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fs_Mount.h>
#include <nn/lmem/lmem_ExpHeap.h>

#include "capsrv_MovieConfig.h"
#include "capsrv_MovieFilePath.h"
#include "capsrv_CachedMovieStream.h"
#include "capsrv_AlbumMovieReadStreamDataSectionAccessor.h"
#include "capsrv_MovieReaderCacheStrategy.h"
#include "../capsrv_AlbumFileId.h"

namespace nn{ namespace capsrv{ namespace movie{

    namespace detail{
        class MovieReaderFileImpl;
        class MovieReaderFileSystemImpl;

        class MovieReaderFileProxy
            : public nn::fs::fsa::IFile
        {
        public:
            explicit MovieReaderFileProxy(MovieReaderFileImpl* pImpl) NN_NOEXCEPT;
            virtual ~MovieReaderFileProxy() NN_NOEXCEPT NN_OVERRIDE;

            void* operator new(size_t size) NN_NOEXCEPT;
            void operator delete(void* p) NN_NOEXCEPT;

            virtual nn::Result DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoWrite(int64_t offset, const void* buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoFlush() NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoGetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoOperateRange(
                void* outBuffer,
                size_t outBufferSize,
                fs::OperationId operationId,
                int64_t offset,
                int64_t size,
                const void* inBuffer,
                size_t inBufferSize
            ) NN_NOEXCEPT NN_OVERRIDE;

        private:
            MovieReaderFileImpl* m_pImpl;
        };

        class MovieReaderFileSystemProxy
            : public nn::fs::fsa::IFileSystem
        {
        public:
            explicit MovieReaderFileSystemProxy(MovieReaderFileSystemImpl* pImpl) NN_NOEXCEPT;
            virtual ~MovieReaderFileSystemProxy() NN_NOEXCEPT NN_OVERRIDE;

            void* operator new(size_t size) NN_NOEXCEPT;
            void operator delete(void* p) NN_NOEXCEPT;

            virtual nn::Result DoCreateFile(const char* path, int64_t size, int option) NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoDeleteFile(const char* path) NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoCreateDirectory(const char* path) NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoDeleteDirectory(const char* path) NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoDeleteDirectoryRecursively(const char* path) NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoRenameFile(const char* currentPath, const char* newPath) NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoRenameDirectory(const char* currentPath, const char* newPath) NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoGetEntryType(nn::fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE;
            //virtual nn::Result DoGetFreeSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE;
            //virtual nn::Result DoGetTotalSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoOpenFile(std::unique_ptr<nn::fs::fsa::IFile>* outValue, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoOpenDirectory(std::unique_ptr<nn::fs::fsa::IDirectory>* outValue, const char* path, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT NN_OVERRIDE;
            virtual nn::Result DoCommit() NN_NOEXCEPT NN_OVERRIDE;
            //virtual nn::Result DoFlush() NN_NOEXCEPT NN_OVERRIDE;
            //virtual nn::Result DoGetFileTimeStampRaw(nn::fs::FileTimeStampRaw* outTimestamp, const char* path) NN_NOEXCEPT NN_OVERRIDE;

        private:
            MovieReaderFileSystemImpl* m_pImpl;
        };


        class MovieReaderFileImpl
        {
            NN_DISALLOW_COPY(MovieReaderFileImpl);
            NN_DISALLOW_MOVE(MovieReaderFileImpl);
        public:
            typedef MovieFilePath FilePath;

        public:
            static size_t GetRequiredMemorySize(int64_t cacheChunkSize, int64_t cacheChunkCount) NN_NOEXCEPT;
            static size_t GetRequiredMemoryAlignment() NN_NOEXCEPT;

            MovieReaderFileImpl() NN_NOEXCEPT;

            bool IsInitialized() const NN_NOEXCEPT;

            void Initialize(
                const FilePath& filename,
                AlbumMovieReadStreamHandle handle,
                int64_t fileSize,
                int64_t cacheChunkSize,
                int64_t cacheChunkCount,
                void* memory,
                size_t memorySize
            ) NN_NOEXCEPT;
            void Finalize() NN_NOEXCEPT;

            bool IsMatchFilename(const FilePath& filename) const NN_NOEXCEPT;
            AlbumMovieReadStreamHandle GetHandle() const NN_NOEXCEPT;

            bool IsOpened() const NN_NOEXCEPT;
            void NotifyOpened() NN_NOEXCEPT;
            void NotifyClosed() NN_NOEXCEPT;

        private:
            nn::Result DoReadImpl(size_t* outValue, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT;
            nn::Result DoWriteImpl(int64_t offset, const void* buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT;
            nn::Result DoFlushImpl() NN_NOEXCEPT;
            nn::Result DoSetSizeImpl(int64_t size) NN_NOEXCEPT;
            nn::Result DoGetSizeImpl(int64_t* outValue) NN_NOEXCEPT;
            nn::Result DoOperateRangeImpl(
                void* outBuffer,
                size_t outBufferSize,
                fs::OperationId operationId,
                int64_t offset,
                int64_t size,
                const void* inBuffer,
                size_t inBufferSize
            ) NN_NOEXCEPT;

        private:
            friend class MovieReaderFileProxy;
            nn::os::SdkMutexType                    m_Mutex;
            CachedMovieStream                       m_Stream;
            AlbumMovieReadStreamDataSectionAccessor m_Accessor;
            MovieReaderCacheStrategy                m_Strategy;

            FilePath m_Filename;
            int64_t m_FileSize;
            std::atomic_int m_OpenCount;
        };


        class MovieReaderFileSystemImpl
        {
            NN_DISALLOW_COPY(MovieReaderFileSystemImpl);
            NN_DISALLOW_MOVE(MovieReaderFileSystemImpl);
            friend class MovieReaderFileSystemProxy;
            friend class MovieReaderFileProxy;

        public:
            static const int FileCountMax = MovieReaderFileSystemFileCountMax;
            static const size_t ProxyHeapSize = 512; // >= sizeof(FileSystemProxy) + sizeof(FileProxy) * FileCountMax + sizeof(HeapCommonHead)
            typedef MovieFilePath FilePath;

        public:
            // @pre fileCount > 0
            // @pre fileCount >= FileCountMax
            static size_t GetRequiredMemorySizeForFile(int64_t cacheChunkSize, int64_t cacheChunkCount) NN_NOEXCEPT;
            static size_t GetRequiredMemoryAlignmentForFile() NN_NOEXCEPT;

            MovieReaderFileSystemImpl() NN_NOEXCEPT;

            void Initialize(const char* mountName) NN_NOEXCEPT;

            void Finalize() NN_NOEXCEPT;

            nn::Result RegisterMovieReadStreamDataSection(
                const char* filename,
                AlbumMovieReadStreamHandle streamHandle,
                int64_t cacheChunkSize,
                int64_t cacheChunkCount,
                void* memory,
                size_t memorySize
            ) NN_NOEXCEPT;

            void UnregisterMovieReadStreamDataSection(const char* filename) NN_NOEXCEPT;
            void UnregisterMovieReadStreamDataSection(AlbumMovieReadStreamHandle streamHandle) NN_NOEXCEPT;

        private:
            int FindEmptyFileSlot() const NN_NOEXCEPT;
            int FindFileSlotByStreamHandle(AlbumMovieReadStreamHandle h) const NN_NOEXCEPT;
            int FindFileSlotByFilename(const char* filename) const NN_NOEXCEPT;

            void* AllocateProxyMemory(size_t size, size_t alignment) NN_NOEXCEPT;
            void FreeProxyMemory(void* ptr) NN_NOEXCEPT;

            nn::Result DoGetEntryTypeImpl(nn::fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT;
            nn::Result DoOpenFileImpl(std::unique_ptr<nn::fs::fsa::IFile>* outValue, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT;

        private:
            FilePath m_MountName;
            MovieReaderFileImpl m_File[FileCountMax];
            nn::os::SdkMutexType m_Mutex;

            nn::lmem::HeapHandle m_ProxyHeapHandle;
            char m_ProxyHeapMemory[ProxyHeapSize];
            std::atomic_int m_ProxyAllocateCount;
        };
    }

}}}
