﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include "../capsrv_AlbumFileId.h"
#include "../capsrv_AlbumMovieReadStreamHandle.h"

namespace nn{ namespace capsrv{ namespace movie{

    namespace detail{
        class MovieReaderFileSystemImpl;
    }

    //! @brief 動画読込ストリームを nn::fs の API からアクセスする機能を提供します。
    //! @details
    //! 動画ファイルのデータ部分を nn::fs の API を通じて読込む機能を提供します。
    //!
    //! このクラスは内部で動画データをキャッシュします。
    //! このクラスを利用して nn::fs の API を通じてアクセスする場合、サイズとオフセットは任意の値を指定できます。
    class MovieReaderFileSystem
    {
        NN_DISALLOW_COPY(MovieReaderFileSystem);
        NN_DISALLOW_MOVE(MovieReaderFileSystem);
    public:
        //! @brief 同時に登録できるファイル数です。
        static const int FileCountMax = 4;

    public:

        //! @brief ファイルごとに必要なワークメモリの大きさを取得します。
        //! @param[in] cacheChunkSize  データキャッシュのチャンクサイズ（バイト）です。
        //! @param[in] cacheChunkCount データキャッシュのチャンク数です。
        //! @pre cacheChunkSize > 0
        //! @pre cacheChunkSize % AlbumMovieDataUnitSize == 0
        //! @pre cacheChunkCount >= 1
        //! @details
        //! 動画データのキャッシュに必要なワークメモリの大きさを取得します。
        //!
        //! @see RegisterMovieReadStreamDataSection()
        static size_t GetRequiredMemorySizeForFile(int64_t cacheChunkSize, int64_t cacheChunkCount) NN_NOEXCEPT;

        //! @brief ファイルごとに必要なワークメモリのアライメントを取得します。
        //! @details
        //! 動画データのキャッシュに必要なワークメモリのアライメントを取得します。
        //!
        //! @see RegisterMovieReadStreamDataSection()
        static size_t GetRequiredMemoryAlignmentForFile() NN_NOEXCEPT;

        //! @brief デフォルトコンストラクタです。
        //! @post IsInitialized() == false
        //! @details
        //! このオブジェクトを利用するためには Initialize() を呼び出す必要があります。
        MovieReaderFileSystem() NN_NOEXCEPT;

        //! @brief このオブジェクトが初期化済かを返します。
        bool IsInitialized() const NN_NOEXCEPT;

        //! @brief このオブジェクトを初期化して指定したマウント名で nn::fs に登録します。
        //! @param[in] mountName fs に登録するマウント名
        //! @pre IsInitialized() == false
        //! @post IsInitialized() == true
        //! @details
        //! このオブジェクトを指定したマウント名で nn::fs に登録します。
        //!
        //! この関数はスレッド安全ではありません。
        //! MovieReaderFileSystem のその他の関数と同時に呼び出した場合の動作は未定義です。
        //!
        //! @see Finalize()
        void Initialize(const char* mountName) NN_NOEXCEPT;

        //! @brief このオブジェクトを nn::fs から登録解除して終了処理を行います。
        //! @pre IsInitialized() == true
        //! @pre 動画読込ストリームが登録されていない
        //! @pre nn::fs からファイルオープンされていない
        //! @post IsInitialized() == false;
        //! @details
        //! このオブジェクトを nn::fs から登録解除して終了処理を行います。
        //!
        //! この関数はスレッド安全ではありません。
        //! MovieReaderFileSystem のその他の関数と同時に呼び出した場合の動作は未定義です。
        void Finalize() NN_NOEXCEPT;

        //! @brief 動画読込ストリームをファイルシステムに登録します。
        //! @param[in] filename        ファイルシステム上でのファイル名です。
        //! @param[in] streamHandle    登録する動画読込ストリームのハンドルです。
        //! @param[in] cacheChunkSize  データキャッシュのチャンクサイズ（バイト）です。
        //! @param[in] cacheChunkCount データキャッシュのチャンク数です。
        //! @param[in] memory          ワークメモリです。
        //! @param[in] memorySize      ワークメモリのサイズ（バイト）です。
        //! @pre IsInitialized() == true
        //! @pre filename != nullptr
        //! @pre strlen(filename) <= MovieFilenameLengthMax
        //! @pre streamHandle は有効な動画読込ストリームハンドル
        //! @pre cacheChunkSize > 0
        //! @pre cacheChunkSize % AlbumMovieDataUnitSize == 0
        //! @pre cacheChunkCount >= 1
        //! @pre memory != nullptr
        //! @pre memory % GetRequiredMemoryAlignmentForFile(cacheChunkSize, cacheChunkCount) == 0
        //! @pre memorySize >= GetRequiredMemorySizeForFile(cacheChunkSize, cacheChunkCount)
        //! @retval nn::fs::ResultFileEntryFull 登録できるストリーム数の上限に達しています。
        //! @retval nn::fs::ResultAlreadyExists 既に同じファイル名でストリームが登録されています。
        //! @details
        //! 動画読込ストリームを指定したファイル名でファイルシステムに登録します。
        //! 例えばマウント名を "MOVIE"、ファイル名を "data.mp4" として登録した場合、
        //! "MOVIE:/data.mp4" というファイルパスで nn::fs からアクセス可能です。
        //!
        //! この API を利用して登録されたファイルに対して可能な操作は以下のものに限られます。
        //!
        //! - nn::fs::OpenFile()
        //! - nn::fs::CloseFile()
        //! - nn::fs::ReadFile()
        //! - nn::fs::GetFileSize()
        //!
        //! memory で渡されたメモリ領域は UnregisterMovieReadStreamDataSection() により登録解除するまでこのオブジェクトにより使用されます。
        nn::Result RegisterMovieReadStreamDataSection(
            const char* filename,
            AlbumMovieReadStreamHandle streamHandle,
            int64_t cacheChunkSize,
            int64_t cacheChunkCount,
            void* memory,
            size_t memorySize
        ) NN_NOEXCEPT;

        //! @brief 動画読込ストリームをファイルシステムから登録解除します。
        //! @param[in] filename 登録解除する動画読込ストリームのファイルシステム上のファイル名です。
        //! @pre IsInitialized() == true
        //! @pre このオブジェクトが初期化済
        //! @pre streamHandle が登録済
        //! @pre 登録解除するファイルが nn::fs からオープン中でない
        //! @details
        //! 動画読込ストリームをファイルシステムから登録解除します。
        //! この関数の呼出以降、 filename の指すファイルは nn::fs からオープンできなくなります。
        //!
        //! nn::fs::Open() によりファイルとして開かれているストリームに対してこの API を呼び出した場合の動作は未定義です。
        //! この API を呼び出す前にストリームに対応するすべてのハンドルを nn::fs::Close() により閉じてください。
        //!
        //! filename が登録されていない場合、この API は何もせずに返ります。
        //!
        //! この関数の呼出により、登録時に渡した memory がこのオブジェクトから返却されます。
        void UnregisterMovieReadStreamDataSection(const char* filename) NN_NOEXCEPT;

        //! @brief 動画読込ストリームをファイルシステムから登録解除します。
        //! @param[in] streamHandle 登録解除する動画読込ストリームのハンドルです。
        //! @pre IsInitialized() == true
        //! @pre このオブジェクトが初期化済
        //! @pre streamHandle が登録済
        //! @pre 登録解除するファイルが nn::fs からオープン中でない
        //! @details
        //! 動画読込ストリームをファイルシステムから登録解除します。
        //! この関数の呼出以降、 streamHandle の指す動画読込ストリームは nn::fs からオープンできなくなります。
        //!
        //! nn::fs::Open() によりファイルとして開かれているストリームに対してこの API を呼び出した場合の動作は未定義です。
        //! この API を呼び出す前にストリームに対応するすべてのハンドルを nn::fs::Close() により閉じてください。
        //!
        //! streamHandle が複数の filename で登録されていた場合、すべての filename が無効化されます。
        //! streamHandle がいずれの filename にも登録されていない場合、この API は何もせずに返ります。
        //!
        //! この関数の呼出により、登録時に渡した memory がこのオブジェクトから返却されます。
        void UnregisterMovieReadStreamDataSection(AlbumMovieReadStreamHandle streamHandle) NN_NOEXCEPT;


    private:
        NN_ALIGNAS(8) char m_Storage[2040];
        detail::MovieReaderFileSystemImpl* m_pImpl;
    };

}}}
