﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/capsrv/movie/capsrv_MovieMetaData.h>
#include <nn/capsrv/capsrv_ScreenShotAttribute.h>
#include <nn/capsrv/capsrv_AlbumMovieWriteStreamHandle.h>
#include <nn/capsrv/capsrv_AlbumFileId.h>
#include <nn/capsrv/capsrv_ScreenShotControl.h>
#include <nn/capsrv/capsrv_AlbumControl.h>

namespace nn{ namespace capsrv{ namespace movie{

    enum MovieMetaDataVersion
    {
        MovieMetaDataVersion_1 = 1,
    };

    class MovieMetaDataBuilder
    {
    public:
        MovieMetaDataBuilder() NN_NOEXCEPT;

        bool IsInitialized() const NN_NOEXCEPT;

        //! @brief 初期化します。
        //! @pre !IsInitialized()
        //! @pre version == MovieMetaDataVersion_1
        //! @pre buffer != nullptr
        //! @pre size >= MovieMetaDataSize
        //! @post IsInitialized()
        void Initialize(MovieMetaDataVersion version, void* buffer, size_t size, const AlbumFileId& fileId) NN_NOEXCEPT;

        //! @brief 終了します。
        //! @pre IsInitialized()
        //! @post !IsInitialized()
        void Finalize() NN_NOEXCEPT;

        //! @breif 動画データ部分のサイズを設定します。
        //! @pre IsInitialized()
        void SetMovieDataSize(size_t size) NN_NOEXCEPT;

        //! @brief 動画の画像データを設定します（ RGBA フォーマット）
        //! @pre IsInitialized()
        //! @pre nn::capsrv::InitializeScreenShotControl() 済
        //! @pre rawImage != nullptr
        //! @pre rawImageSize >= 4 * width * height
        //! @pre width == 1280
        //! @pre height == 720
        //! @details
        //!   画像は RGBA8888 フォーマット形式です。
        void SetImageDataRgba(
            const void* rawImage,
            size_t rawImageSize,
            int width,
            int height
        ) NN_NOEXCEPT;

        //! @brief 動画の画像データを設定します（ NV12 フォーマット）
        //! @pre IsInitialized()
        //! @pre nn::capsrv::InitializeScreenShotControl() 済
        //! @pre rawImageNv12Y != nullptr
        //! @pre rawImageNv12Y % OverlayMovieImageBufferMemoryAlignment == 0
        //! @pre rawImageNv12YSize % OverlayMovieImageBufferMemoryUnitSize == 0
        //! @pre rawImageNv12YSize >= width * height
        //! @pre rawImageNv12Uv != nullptr
        //! @pre rawImageNv12Uv % OverlayMovieImageBufferMemoryAlignment == 0
        //! @pre rawImageNv12UvSize % OverlayMovieImageBufferMemoryUnitSize == 0
        //! @pre rawImageNv12UvSize >= width * height / 2
        //! @pre width == 1280
        //! @pre height == 720
        //! @details
        //!   画像は NV12 フォーマット（YUV420 semi-planar）形式です。
        void SetImageDataNv12(
            const void* rawImageNv12Y,
            size_t rawImageNv12YSize,
            const void* rawImageNv12Uv,
            size_t rawImageNv12UvSize,
            int width,
            int height
        ) NN_NOEXCEPT;

        //! @brief 作成するファイルの種類を設定します。
        void SetDescription(
            AlbumFileDescriptionType description
        ) NN_NOEXCEPT;

        //! @brief 動画情報を設定します。
        //! @pre IsInitialized()
        void SetAttribute(
            int frameCount,
            int frameRateNumerator,
            int frameRateDenominator,
            int dataDurationMilliseconds,
            int keyFrameIntervalFrames,
            bool isCopyrightImageComposited,
            ScreenShotSizeType movieSize,
            ScreenShotOrientationType orientation
        ) NN_NOEXCEPT;

        //! @brief アプレット用領域を設定します。
        //! @pre IsInitialized()
        //! @details
        //! 省略可能。省略した場合は 0 埋め。
        void SetAppletData(
            const AppletData& appletData
        ) NN_NOEXCEPT;

        //! @brief アプリケーション任意データ用領域を設定します。
        //! @pre IsInitialized()
        //! @details
        //! 省略可能。省略した場合は 0 埋め。
        void SetApplicationData(
            const ApplicationData& applicationData
        ) NN_NOEXCEPT;

        //! @brief 設定した情報からメタ情報を構築します。
        //! @pre 各種パラメータが設定済
        //! @pre nn::capsrv::InitializeScreenShotControl() 済
        nn::Result Build() NN_NOEXCEPT;

        //! @brief ストリームに書き出します。
        //! @pre IsInitialized()
        //! @pre nn::capsrv::InitializeAlbumControl() 済
        //! @pre 動画データサイズを設定済
        //! @pre 画像データを設定済
        //! @pre streamHandle はメタ書込み中の有効な動画書込みストリームのハンドル
        nn::Result WriteToStream(AlbumMovieWriteStreamHandle streamHandle) const NN_NOEXCEPT;

        uint64_t GetMakerNoteVersion() const NN_NOEXCEPT;
        int64_t GetMakerNoteOffset() const NN_NOEXCEPT;
        int64_t GetMakerNoteSize() const NN_NOEXCEPT;

    private:
        bool m_IsInitialzed;
        bool m_IsMovieDataSizeReady;
        bool m_IsImageDataReady;
        bool m_IsDescriptionReady;
        bool m_IsAttributeReady;
        bool m_IsBuilt;

        MovieMetaDataVersion m_MovieMetaDataVersion;
        AlbumFileId m_FileId;
        MovieMetaData* m_pMetaData;
        AppletData* m_pAppletData;
        ApplicationData* m_pApplicationData;

        size_t m_MovieDataSize;

        const void* m_pRawImageRgba;
        size_t m_RawImageRgbaSize;
        const void* m_pRawImageNv12Y;
        size_t m_RawImageNv12YSize;
        const void* m_pRawImageNv12Uv;
        size_t m_RawImageNv12UvSize;
        int m_RawImageWidth;
        int m_RawImageHeight;

        AlbumFileDescriptionType m_Description;
        ScreenShotAttribute m_Attribute;
        int64_t m_MakerNoteOffset;
        int64_t m_MakerNoteSize;
    };

}}}

