﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

namespace nn{ namespace capsrv{ namespace movie{

    struct MovieStreamAccessor
    {
        void* userData;
        nn::Result (*read)(size_t* pOutReadSize, void* buffer, size_t size, int64_t offset, void* userData);
        nn::Result (*write)(int64_t offset, const void* buffer, size_t size, void* userData);
        nn::Result (*shrink)(int64_t size, void* userData);
    };

    struct MovieStreamCacheStrategy
    {
        void* userData;
        void (*notifyFetched)(int64_t chunkIndex, void* userData);
        void (*notifyInvalidated)(int64_t chunkIndex, void* userData);
        void (*notifyRead)(int64_t chunkIndex, int64_t offset, int64_t size, void* userData);
        void (*notifyWritten)(int64_t chunkIndex, int64_t offset, int64_t size, void* userData);
        int64_t (*queryChunkIndexToInvalidate)(void* userData);
    };

    class CachedMovieStream
    {
    public:
        static const int64_t InvalidChunkIndex = -1;

    private:
        enum CacheChunkFlag : uint64_t
        {
            CacheChunkFlag_None  = 0,
            CacheChunkFlag_Dirty = 1 << 0,
        };

        struct CacheChunkHead
        {
            int64_t chunkIndex;
            uint64_t flag;
            char* data;
        };
    public:
        CachedMovieStream() NN_NOEXCEPT;

        //! @pre cacheChunkSize > 0
        //! @pre cacheChunkCount >= 1
        static size_t GetRequiredWorkMemorySize(int64_t cacheChunkSize, int64_t cacheChunkCount) NN_NOEXCEPT;
        static size_t GetRequiredWorkMemoryAlignment() NN_NOEXCEPT;

        //! @pre cacheChunkSize > 0
        //! @pre cacheChunkCount >= 1
        //! @pre workMemory != nullptr
        //! @pre accessor.read != nullptr
        //! @pre strategy.notifyFetched != nullptr
        //! @pre strategy.notifyInvalidated != nullptr
        //! @pre strategy.queryChunkIndexToInvalidate != nullptr
        //! @pre workMemory % GetRequiredWorkMemoryAlignment() == 0
        //! @pre workMemorySize >= GetRequiredWorkMemorySize(cacheChunkCount)
        //! @pre initialChunkCount >= 0
        void Initialize(
            int64_t cacheChunkSize,
            int64_t cacheChunkCount,
            const MovieStreamAccessor& accessor,
            const MovieStreamCacheStrategy& strategy,
            void* workMemory,
            size_t workMemorySize,
            int64_t initialChunkCount
        ) NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;

        nn::Result Read(size_t* pOutReadSize, void* buffer, size_t size, int64_t offset) NN_NOEXCEPT;
        nn::Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT;
        nn::Result Resize(int64_t size) NN_NOEXCEPT;
        nn::Result Flush() NN_NOEXCEPT;

    private:
        int64_t CalculateAvailableChunkCount() const NN_NOEXCEPT;

        // @pre chunkIndex != InvalidChunkIndex
        CacheChunkHead* FindCacheChunk(int64_t chunkIndex) NN_NOEXCEPT;
        CacheChunkHead* FindEmptyCacheChunk() NN_NOEXCEPT;

        // @pre pChunk->chunkIndex != InvalidChunkIndex
        // @post pChunk->chunkIndex == InvalidChunkIndex
        // @details
        //   pChunk が Dirty であれば、 pChunk をファイルに書き出す。
        //   この際ファイルは必要な大きさまで拡大される。
        //   拡大量が複数チャンクになる場合、ファイル末尾から pChunk の先頭までの領域は 0 埋めされる。
        //
        //   pChunk のキャッシュを破棄する。
        nn::Result FlushCacheChunk(CacheChunkHead* pChunk) NN_NOEXCEPT;

        // @pre pChunk->chunkIndex != InvalidChunkIndex
        // @post pChunk->chunkIndex == InvalidChunkIndex
        // @details
        //   pChunk のキャッシュを破棄する。
        //   pChunk が Dirty でない場合は Flush と同等。
        //
        //   サイズの縮小時に不要になったチャンクを破棄するために使用する。
        void InvalidateCacheChunk(CacheChunkHead* pChunk) NN_NOEXCEPT;

        // @pre pChunk->chunkIndex == InvalidChunkIndex
        // @pre chunkIndex >= 0
        // @post pChunk->chunkIndex == chunkIndex
        // @details
        //   指定したインデックスのチャンクをファイルから読み込んで pChunk にキャッシュする。
        //   ファイルに書き出されていないチャンクを読もうとした場合には 0 埋めする。
        nn::Result FetchCacheChunk(CacheChunkHead* pChunk, int64_t chunkIndex) NN_NOEXCEPT;

        // @pre chunkIndexBeg >= 0
        // @pre pChunk->chunkIndex == InvalidChunkIndex
        // @details
        //   ファイル上の [chunkIndexBeg, chunkIndexEnd) の範囲のチャンクに 0 を書き出す。
        //   キャッシュしているチャンクに対しても 0 を書き出し、キャッシュは維持されるため、ゼロ埋めする場合には
        //   先にキャッシュを Flush または Invalidate しておくこと。
        //
        //   キャッシュの Flush 時に中間領域を 0 埋めする際に使用する。
        nn::Result WriteOutZeroChunkRange(int64_t chunkIndexBeg, int64_t chunkIndexEnd, CacheChunkHead* pWorkChunk) NN_NOEXCEPT;

        // @details
        //   chunkIndex のチャンクのキャッシュを取得する。
        //   既にキャッシュしている場合、キャッシュが取得される。
        //   キャッシュ中でない場合、新しく Fetch して作成したキャッシュが取得される。
        //   この際、キャッシュが一杯であれば 1 つキャッシュを Flush する。
        nn::Result RequestCacheChunk(CacheChunkHead** pOutChunk, int64_t chunkIndex) NN_NOEXCEPT;

        // @details
        //   chunkIndex >= chunkIndexBegin のキャッシュを Invalidate する。
        //
        //   サイズの縮小時に不要になったチャンクを破棄するために使用する。
        void InvalidateCacheChunkGreaterEqualIndex(int64_t chunkIndexBegin) NN_NOEXCEPT;

        nn::Result ReadImpl(size_t* pOutReadSize, void* buffer, size_t size, int64_t offset) NN_NOEXCEPT;
        nn::Result WriteImpl(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT;
        nn::Result ResizeImpl(int64_t size) NN_NOEXCEPT;
        nn::Result FlushImpl() NN_NOEXCEPT;


    private:
        MovieStreamAccessor      m_Accessor;
        MovieStreamCacheStrategy m_Strategy;

        void*  m_pWorkMemory;
        size_t m_WorkMemorySize;

        int64_t m_CacheChunkSize;
        int64_t m_CacheChunkCountMax;
        CacheChunkHead* m_pCacheChunkList;

        // キャッシュ中のチャンク数
        int64_t m_CacheChunkCount;
        // ファイルに書き出しているチャンク数
        int64_t m_FileChunkCount;
        // Resize で明示的に指定されたチャンク数
        // Flush 時に m_FileChunkCount がこれより少ない場合、末尾をゼロ埋めする
        int64_t m_ExplicitChunkCount;
    };

}}}
