﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/ncm/ncm_ProgramId.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/applet/applet_Types.h>
#include <nn/vi/vi_LayerStack.h>
#include <nn/vi/fbshare/vi_SharedBufferHandle.h>
#include <nn/sf/sf_Buffers.h>
#include <nn/sf/sf_Out.h>
#include <nn/capsrv/capsrv_AlbumFileId.h>
#include <nn/capsrv/capsrv_ApplicationAlbumEntry.h>
#include <nn/capsrv/capsrv_AppletData.h>
#include <nn/capsrv/capsrv_ApplicationData.h>
#include <nn/capsrv/capsrv_ScreenShotSize.h>
#include <nn/capsrv/capsrv_ScreenShotOrientation.h>
#include <nn/capsrv/capsrv_ScreenShotAttribute.h>
#include <nn/capsrv/capsrv_SystemReservedInfo.h>
#include <nn/capsrv/capsrv_OverlayNotificationRequest.h>
#include <nn/capsrv/movie/capsrv_MovieConfig.h>

namespace nn{ namespace capsrv{

    //! @brief 画面キャプチャのタイムアウトのデフォルト値です。
    static const int DefaultCaptureTimeoutMilliseconds = 100;

//! @name 初期化・終了処理
//! @{

    //! @brief 静止画撮影機能の制御を初期化します。
    //! @pre 静止画撮影機能の制御が初期化されていない
    //! @post 静止画撮影機能の制御が初期化されている
    //! @details
    //! 静止画撮影機能の制御を行う関数を呼び出す前にこの関数を呼び出す必要があります。
    nn::Result InitializeScreenShotControl() NN_NOEXCEPT;

    //! @brief 静止画撮影機能の制御の終了処理を行います。
    //! @pre 静止画撮影機能の制御が初期化されている
    //! @post 静止画撮影機能の制御が初期化されていない
    //! @details
    //! この関数を呼び出した後に静止画撮影機能の制御を行う関数を呼び出してはいけません。
    void FinalizeScreenShotControl() NN_NOEXCEPT;

    //! @brief 共有バッファを画面キャプチャに接続します。
    nn::Result AttachSharedBufferToCaptureModule(nn::vi::fbshare::SharedBufferHandle hBuffer) NN_NOEXCEPT;
//! @}

//! @name 撮影処理
//! @{

    //! @brief 画面写真の撮影を要求します。
    //! @param[in] seqNo        撮影シーケンスの番号です。
    //! @param[in] programId    撮影対象のプログラム ID です。
    //! @param[in] aruid        撮影対象のレイヤーの ID です。
    //! @param[in] attribute    画面写真に付与する属性です。
    //! @param[in] userIdList   ユーザ識別子のリストです。
    //! @param[in] timeout      画面キャプチャのタイムアウトです。
    //! @pre 静止画撮影機能の制御が初期化されている
    //! @pre seqNo >= 1
    //! @details
    //! 現在のディスプレイの出力を撮影して画面写真として保存するように要求します。
    //! 撮影対象となるレイヤーは nn::vi::LayerStack_ScreenShot です（デバッグのために fwdbg で変更される場合があります）
    //! この関数は撮影のリクエストを発行後、撮影の完了を待たずに直ちに返ります。
    //! 撮影の結果はオーバーレイに通知されます。
    //!
    //! キャプチャーボタンのハンドリングを担当するモジュールは、ボタン押下に対するリアクションを返すため、以下のうちのいずれかを呼び出す必要があります。
    //! - RequestTakingScreenShot()
    //! - NotifyTakingScreenShotRefused()
    //! - NotifyTakingScreenShotFailed()
    nn::Result RequestTakingScreenShot(uint64_t seqNo, nn::ncm::ProgramId programId, nn::applet::AppletResourceUserId aruid, const ScreenShotAttribute& attribute, const UserIdList& userIdList, nn::TimeSpan timeout) NN_NOEXCEPT;

    //! @brief 画面写真の撮影を取り消します。
    //! @param[in] seqNo キャンセルする撮影シーケンスの番号です。
    //! @param[in] reason キャンセル理由です。
    //! @details
    //! 指定したシーケンス番号以下の撮影要求を取り消します。
    //! この API から返ったあとのディスプレイの出画内容が画面写真として保存されないことが保証されます。
    //!
    //! より正確には、要求した撮影処理が進行中だった場合、以下のようになります。
    //! - 画面キャプチャが未完了だった場合、画面写真は作成されません。
    //! - 画面キャプチャが完了していた場合、画面写真は作成されます。
    //!
    //! reason には以下のいずれかを指定します。
    //! - nn::capsrv::ResultInternalScreenShotCanceledAsNoTarget()
    //! - nn::capsrv::ResultInternalScreenShotCanceledAsProhibited()
    //!
    nn::Result CancelTakingScreenShot(uint64_t seqNo, nn::Result reason) NN_NOEXCEPT;

    //! テスト用。通常は CancelTakingScreenShot を使用します。
    nn::Result SetTakingScreenShotCancelState(uint64_t seqNo, nn::Result reason) NN_NOEXCEPT;

    //! @breif 画面写真の撮影がアプリケーションによって拒否されたことを通知します。
    //! @param[in] programId 撮影を拒否したプログラムの ID です。
    //! @pre 静止画撮影機能の制御が初期化されている
    //! @details
    //! 画面写真の撮影がアプリケーションによって拒否されたことを通知します。
    //! RequestTakingScreenShot() の呼出以前に拒否された場合、この関数を呼び出します。
    //! この関数は通知後、直ちに返ります。
    //! 撮影が拒否されたことはオーバーレイに通知されます。
    //!
    //! キャプチャーボタンのハンドリングを担当するモジュールは、ボタン押下に対するリアクションを返すため、以下のうちのいずれかを呼び出す必要があります。
    //! - RequestTakingScreenShot()
    //! - NotifyTakingScreenShotRefused()
    //! - NotifyTakingScreenShotFailed()
    nn::Result NotifyTakingScreenShotRefused(nn::ncm::ProgramId programId) NN_NOEXCEPT;

    //! @breif 画面写真の撮影が失敗したことを通知します。
    //! @param[in] programId 撮影対象のプログラム ID です
    //! @pre 静止画撮影機能の制御が初期化されている
    //! @details
    //! 画面写真の撮影が失敗したことを通知します。
    //! RequestTakingScreenShot() の呼出以前に失敗した場合、この関数を呼び出します。
    //! この関数は通知後、直ちに返ります。
    //! 撮影が拒否されたことはオーバーレイに通知されます。
    //!
    //! キャプチャーボタンのハンドリングを担当するモジュールは、ボタン押下に対するリアクションを返すため、以下のうちのいずれかを呼び出す必要があります。
    //! - RequestTakingScreenShot()
    //! - NotifyTakingScreenShotRefused()
    //! - NotifyTakingScreenShotFailed()
    nn::Result NotifyTakingScreenShotFailed(nn::ncm::ProgramId programId) NN_NOEXCEPT;

    //! @brief アタッチ中の共有バッファに画面キャプチャを行います。
    nn::Result CaptureRawImageToAttachedSharedBuffer(
        int dstIndex,
        vi::LayerStack layerStack,
        nn::TimeSpan timeout
    ) NN_NOEXCEPT;

    //! @brief アプリからの SaveScreenShot 要求
    //! @details
    //! アプリ -> am -> vi というパスで UserIdList 情報を埋め込む用
    nn::Result SaveScreenShotEx2ViaAm(
        nn::sf::Out<ApplicationAlbumEntry> pOutEntry,
        const nn::sf::InBuffer& imageBuffer,
        const capsrv::detail::ScreenShotAttributeEx0& attribute,
        const UserIdList& userIdList,
        applet::AppletResourceUserId aruid,
        OverlayNotificationRequestType overlayRequest
    ) NN_NOEXCEPT;

//! @}

//! @name 動画サムネイル
//! @{

    //! @brief 動画の画像バッファに要求されるメモリの単位サイズ（バイト）です。
    static const size_t OverlayMovieImageBufferMemoryUnitSize = 128 * 1024;

    //! @brief 動画の画像バッファに要求されるメモリのアライメント（バイト）です。
    static const size_t OverlayMovieImageBufferMemoryAlignment = 4 * 1024;

    //! @brief 画像をオーバーレイ用のサムネイルに変換して動画のサムネイルとして保存します。
    //! @param[in] pImageBufferY     Y 平面
    //! @param[in] imageBufferYSize  pImageBufferY の大きさ（バイト）
    //! @param[in] pImageBufferUv    UV 平面
    //! @param[in] imageBufferUvSize pImageBufferUv の大きさ（バイト）
    //! @param[in] width             画像の幅（px）
    //! @param[in] height            画像の高さ（px）
    //! @param[in] fileId            動画のファイル ID
    //! @pre pImageBufferY != nullptr
    //! @pre pImageBufferY % OverlayMovieImageBufferMemoryAlignment == 0
    //! @pre imageBufferYSize % OverlayMovieImageBufferMemoryUnitSize == 0
    //! @pre imageBufferYSize >= width * height
    //! @pre pImageBufferUv != nullptr
    //! @pre pImageBufferUv % OverlayMovieImageBufferMemoryAlignment == 0
    //! @pre imageBufferUvSize % OverlayMovieImageBufferMemoryUnitSize == 0
    //! @pre imageBufferUvSize >= width * height / 2
    //! @pre width == 1280
    //! @pre height == 720
    //! @details
    //!   入力画像は NV12 フォーマット（YUV420 semi-planar）形式です。
    //! この API を複数回呼び出した場合、動画用サムネイルが上書きされます。
    //! nn::capsrv::GetLastOverlayMovieThumbnail() を使って最後に設定したサムネイル画像が取得できるようになります。
    nn::Result SetupOverlayMovieThumbnail(
        void* pImageBufferY,
        size_t imageBufferYSize,
        void* pImageBufferUv,
        size_t imageBufferUvSize,
        int width,
        int height,
        const nn::capsrv::AlbumFileId& fileId
    ) NN_NOEXCEPT;

    //! @brief 動画のメタデータを作成します（ RGBA 版）
    //! @param[out] pOutSize         作成したメタデータの大きさ（バイト）を受け取る変数のポインタ
    //! @param[out] pOutMakerNoteOffset 作成したメタデータの MakerNote のオフセットを受け取る変数のポインタ
    //! @param[out] pOutMakerNoteSize   作成したメタデータの MakerNote のサイズを受け取る変数のポインタ
    //! @param[out] pOutBuffer       メタデータを受け取るバッファ
    //! @param[in] outBufferSize     pOutBuffer の大きさ（バイト）
    //! @param[in] pImageBuffer      RGBA 画像データ
    //! @param[in] imageBufferSize   pImageBuffer の大きさ（バイト）
    //! @param[in] width             画像の幅（px）
    //! @param[in] height            画像の高さ（px）
    //! @param[in] fileId            動画のファイル ID
    //! @param[in] description       動画ファイルの詳細
    //! @param[in] attribute         動画の属性情報
    //! @param[in] appletData        アプレット用データ領域
    //! @param[in] applicationData   アプリケーション任意データ領域
    //! @pre pOutSize != nullptr
    //! @pre pOutBuffer != nullptr
    //! @pre outBufferSize >= MovieMetaDataSize
    //! @pre pImageBuffer != nullptr
    //! @pre pImageBuffer % 4 * width * height == 0
    //! @pre width == 1280
    //! @pre height == 720
    //! @details
    //!   入力画像は RGBA8888 フォーマットです。
    nn::Result CreateProtoMovieMetaDataRgba(
        size_t* pOutSize,
        int64_t* pOutMakerNoteOffset,
        int64_t* pOutMakerNoteSize,
        void* pOutBuffer,
        size_t outBufferSize,
        const void* pImageBuffer,
        size_t imageBufferSize,
        int width,
        int height,
        const nn::capsrv::AlbumFileId& fileId,
        const ScreenShotAttribute& attribute,
        const AppletData& appletData,
        const ApplicationData& applicationData
    ) NN_NOEXCEPT;

    //! @brief 動画のメタデータを作成します（暫定版）
    //! @param[out] pOutSize         作成したメタデータの大きさ（バイト）を受け取る変数のポインタ
    //! @param[out] pOutMakerNoteOffset 作成したメタデータの MakerNote のオフセットを受け取る変数のポインタ
    //! @param[out] pOutMakerNoteSize   作成したメタデータの MakerNote のサイズを受け取る変数のポインタ
    //! @param[out] pOutBuffer       メタデータを受け取るバッファ
    //! @param[in] outBufferSize     pOutBuffer の大きさ（バイト）
    //! @param[in] pImageBufferY     Y 平面
    //! @param[in] imageBufferYSize  pImageBufferY の大きさ（バイト）
    //! @param[in] pImageBufferUv    UV 平面
    //! @param[in] imageBufferUvSize pImageBufferUv の大きさ（バイト）
    //! @param[in] width             画像の幅（px）
    //! @param[in] height            画像の高さ（px）
    //! @param[in] fileId            動画のファイル ID
    //! @param[in] description       動画ファイルの詳細
    //! @param[in] attribute         動画の属性情報
    //! @param[in] appletData        アプレット用データ領域
    //! @param[in] applicationData   アプリケーション任意データ領域
    //! @pre pOutSize != nullptr
    //! @pre pOutBuffer != nullptr
    //! @pre outBufferSize >= MovieMetaDataSize
    //! @pre pImageBufferY != nullptr
    //! @pre pImageBufferY % OverlayMovieImageBufferMemoryAlignment == 0
    //! @pre imageBufferYSize % OverlayMovieImageBufferMemoryUnitSize == 0
    //! @pre imageBufferYSize >= width * height
    //! @pre pImageBufferUv != nullptr
    //! @pre pImageBufferUv % OverlayMovieImageBufferMemoryAlignment == 0
    //! @pre imageBufferUvSize % OverlayMovieImageBufferMemoryUnitSize == 0
    //! @pre imageBufferUvSize >= width * height / 2
    //! @pre width == 1280
    //! @pre height == 720
    //! @details
    //!   入力画像は NV12 フォーマット（YUV420 semi-planar）形式です。
    nn::Result CreateProtoMovieMetaDataNv12(
        size_t* pOutSize,
        int64_t* pOutMakerNoteOffset,
        int64_t* pOutMakerNoteSize,
        void* pOutBuffer,
        size_t outBufferSize,
        void* pImageBufferY,
        size_t imageBufferYSize,
        void* pImageBufferUv,
        size_t imageBufferUvSize,
        int width,
        int height,
        const nn::capsrv::AlbumFileId& fileId,
        const ScreenShotAttribute& attribute,
        const AppletData& appletData,
        const ApplicationData& applicationData
    ) NN_NOEXCEPT;

//! @}

//! @name 画像ストリーム（開発環境専用）
//! @{

    nn::Result OpenRawScreenShotReadStreamForDevelop(
        size_t* pOutImageDataSize,
        int* pOutImageWidth,
        int* pOutImageHeight,
        vi::LayerStack layerStack,
        nn::TimeSpan timeout
    ) NN_NOEXCEPT;

    void CloseRawScreenShotReadStreamForDevelop() NN_NOEXCEPT;

    nn::Result ReadRawScreenShotReadStreamForDevelop(
        size_t* pOutReadSize,
        void* pBuffer,
        size_t bufferSize,
        ptrdiff_t startOffset
    ) NN_NOEXCEPT;

//! @}

}}
