﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include "capsrv_AlbumFileContents.h"

namespace nn{ namespace capsrv{

    //! @brief AlbumContentsUsageFlag の値を格納するための型です。
    typedef uint32_t AlbumContentsUsageFlagType;

    //! @brief AlbumContentsUsage のフラグです。
    enum AlbumContentsUsageFlag
    {

        //! @brief 取得した値以上のファイル数・容量です。
        AlbumContentsUsageFlag_HasGreaterUsage   = 1 << 0,

        //! @brief 未知のファイルです。
        AlbumContentsUsageFlag_IsUnknownContents = 1 << 1,

    };

    //! @brief アルバム内のファイルの種類ごとの個数と容量の合計です。
    struct AlbumContentsUsage
    {
    public:
        //! @brief この種類のファイルの個数です。
        //! @details
        //!   カウントしたファイルの個数が記録されます。
        //!   カウントされなかったファイルが存在する可能性がある場合、 HasGreaterUsage() が true を返します。@n
        int64_t count;

        //! @brief この種類のファイルのサイズの合計です。
        //! @details
        //!   カウントしたファイルのサイズの合計が記録されます。
        //!   カウントされなかったファイルが存在する可能性がある場合、 HasGreaterUsage() が true を返します。@n
        int64_t size;

        //! @brief フラグです。
        //! @details
        //!   AlbumContentsUsageFlag で定義されるフラグの値が記録されます。
        //!   フラグの値を取得するためにアクセサ関数を使用することができます。
        //! @see HasGreaterUsage()
        //! @see IsUnknownContents()
        AlbumContentsUsageFlagType flags;

        //! @brief ファイルの種類です。
        //! @details
        //!   IsUnknownContents() が false を返す場合、この値がファイルの種類を表します。
        //!   IsUnknownContents() が true を返す場合、この値は無効です。
        AlbumFileContentsType contents;

        NN_PADDING3;

    public:
        //! @brief この種類のファイルが count, size に記録された以上のファイル数・容量を占めているかを返します。
        bool HasGreaterUsage() const NN_NOEXCEPT
        {
            return (this->flags & AlbumContentsUsageFlag_HasGreaterUsage) != 0;
        }

        //! @brief ファイルの種類が未知であるかを返します。
        //! @details
        //!   この関数が false を返す場合、ファイルの種類は contents の値です。
        //!   この関数が true を返す場合、 contents の値は無効です。
        bool IsUnknownContents() const NN_NOEXCEPT
        {
            return (this->flags & AlbumContentsUsageFlag_IsUnknownContents) != 0;
        }

    };

    namespace detail{
        struct AlbumUsage2
        {
            static const int Count = 2;
            AlbumContentsUsage usages[Count];
        };

        struct AlbumUsage3
        {
            static const int Count = 3;
            AlbumContentsUsage usages[Count];
        };

        struct AlbumUsage16
        {
            static const int Count = 16;
            AlbumContentsUsage usages[Count];
        };
    }

    //! @brief アルバム内のファイルのストレージ使用状況です。
    struct AlbumUsage : public detail::AlbumUsage16
    {
    public:
        typedef detail::AlbumUsage16 BaseType;
        static const int ContentsCount = BaseType::Count - 1;
        static const int UnknownContentsIndex = BaseType::Count - 1;

    public:

        //! @brief 指定した種類のファイルのストレージ使用状況を取得します。
        //! @param[in] contents ファイルの種類
        //! @pre contents は AlbumFileContents のいずれかの値でなければなりません。
        //! @return contents で指定された種類のファイルのストレージ使用状況へのポインタを返します。
        const AlbumContentsUsage* GetContentsUsage(AlbumFileContentsType contents) const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_RANGE(contents, 0, static_cast<int>(ContentsCount));
            return &this->usages[contents];
        }

        //! @brief アルバムの管理対象外のファイルによるストレージ使用状況を取得します。
        //! @return アルバムの管理対象外のファイルのストレージ使用状況へのポインタを返します。
        const AlbumContentsUsage* GetUnknownUsage() const NN_NOEXCEPT
        {
            return &this->usages[UnknownContentsIndex];
        }

    };

}}
