﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ncm/ncm_ProgramId.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/applet/applet_Types.h>
#include <nn/capsrv/capsrv_AlbumEntry.h>
#include <nn/capsrv/capsrv_AlbumStorage.h>
#include <nn/capsrv/capsrv_AlbumFileContents.h>
#include <nn/capsrv/capsrv_AlbumFileId.h>
#include <nn/capsrv/capsrv_ApplicationAlbumEntry.h>
#include <nn/capsrv/capsrv_AlbumMovieWriteStreamHandle.h>

namespace nn{ namespace capsrv{

//! @name 初期化・終了処理
//! @{

    //! @brief アルバム機能の制御を初期化します。
    //! @pre アルバム機能の制御が初期化されていない
    //! @post アルバム機能の制御が初期化されている
    //! @details
    //! アルバム機能の制御を行う関数を呼び出す前にこの関数を呼び出す必要があります。
    nn::Result InitializeAlbumControl() NN_NOEXCEPT;

    //! @brief アルバム機能制御の拡張機能を初期化します。
    //! @pre アルバム機能の制御が初期化されている
    //! @details
    //! 以下の機能を利用する場合にはこの関数を呼び出す必要があります。
    //!
    //! - MovieReadStream
    //! - MovieWriteStream
    //!
    //! FinalizeAlbumControl() により拡張機能も終了処理が行われます。
    nn::Result InitializeAlbumControlExtension() NN_NOEXCEPT;

    //! @brief アルバム機能の制御の終了処理を行います。
    //! @pre アルバム機能の制御が初期化されている
    //! @post アルバム機能の制御が初期化されていない
    //! @details
    //! この関数を呼び出した後にアルバム機能の制御を行う関数を呼び出してはいけません。
    void FinalizeAlbumControl() NN_NOEXCEPT;

//! @}

//! @name アプリの起動・終了を am プロセスから通知
//! @{

    //! @brief アプリ起動時に ApplicationId と ARUID を登録する。
    //! @param[in] aruid 起動したアプリの AppletResourceUserId です。
    //! @param[in] applicationId 起動したアプリの ApplicationId です。
    //! @pre アルバム機能の制御が初期化されている。
    //! @details
    //! アプリ起動時に ApplicationId と ARUID を登録します。
    //! この関数は am プロセスが使用します。
    nn::Result RegisterAppletResourceUserId(nn::applet::AppletResourceUserId aruid, nn::ncm::ApplicationId applicationId) NN_NOEXCEPT;

    //! @brief アプリ終了時に ApplicationId と ARUID の登録を解除する。
    //! @param[in] aruid 終了したアプリの AppletResourceUserId です。
    //! @param[in] applicationId 終了したアプリの ApplicationId です。
    //! @pre アルバム機能の制御が初期化されている。
    //! @details
    //! アプリ終了時に ApplicationId と ARUID の登録を解除します。
    //! この関数は am プロセスが使用します。
    nn::Result UnregisterAppletResourceUserId(nn::applet::AppletResourceUserId aruid, nn::ncm::ApplicationId applicationId) NN_NOEXCEPT;

//! @}

//! @name ストレージ管理
//! @{

    //! @brief ストレージが利用可能になったことを capsrv に通知します。
    //! @param[in] storage 利用可能になったストレージ
    //! @pre アルバム機能の制御が初期化されている
    //! @pre storage が nn::capsrv::AlbumStorage のいずれかの値
    //! @post capsrv が storage で指定したストレージを利用するようになります。
    //! @details
    //!   capsrv が指定したストレージを利用するようになります。
    //!   既にストレージを利用中だった場合、何もせずに nn::ResultSuccess を返します。
    nn::Result NotifyAlbumStorageIsAvailable(AlbumStorageType storage) NN_NOEXCEPT;

    //! @brief ストレージが利用不能になったことを capsrv に通知します。
    //! @param[in] storage 利用不能になったストレージ
    //! @pre アルバム機能の制御が初期化されている
    //! @pre storage が nn::capsrv::AlbumStorage のいずれかの値
    //! @post capsrv が storage で指定したストレージを利用しなくなります。
    //! @details
    //!   capsrv が指定したストレージを利用しなくなります。
    //!   既にストレージを利用していなかった場合、何もせずに nn::ResultSuccess を返します。
    nn::Result NotifyAlbumStorageIsUnavailable(AlbumStorageType storage) NN_NOEXCEPT;
//! @}

//! @name ファイル生成
//! @{

    nn::Result GetApplicationIdFromAruid(nn::ncm::ApplicationId* pOutValue, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;
    nn::Result CheckApplicationIdRegistered(nn::ncm::ApplicationId applicationId) NN_NOEXCEPT;
    nn::Result GenerateCurrentAlbumFileId(AlbumFileId* pOutValue, nn::ncm::ApplicationId applicationId, AlbumFileContentsType contents) NN_NOEXCEPT;
    nn::Result GenerateApplicationAlbumEntry(ApplicationAlbumEntry* pOutValue, const AlbumEntry& srcEntry, nn::ncm::ApplicationId applicationId) NN_NOEXCEPT;
    nn::Result SetOverlayScreenShotThumbnailData(const AlbumFileId& fileId, const void* pData, size_t dataSize) NN_NOEXCEPT;
    nn::Result SetOverlayMovieThumbnailData(const AlbumFileId& fileId, const void* pData, size_t dataSize) NN_NOEXCEPT;
    nn::Result SaveAlbumScreenShotFile(const AlbumFileId& fileId, const void* pData, size_t fileSize, uint64_t makerNoteVersion, int64_t makerNoteOffset, int64_t makerNoteSize) NN_NOEXCEPT;

//! @}

//! @name 動画書き出し
//! @{

    nn::Result OpenAlbumMovieWriteStream(AlbumMovieWriteStreamHandle* pOutHandle, const AlbumFileId& fileId) NN_NOEXCEPT;
    nn::Result FinishAlbumMovieWriteStream(AlbumMovieWriteStreamHandle handle) NN_NOEXCEPT;
    nn::Result CommitAlbumMovieWriteStream(AlbumMovieWriteStreamHandle handle) NN_NOEXCEPT;
    void DiscardAlbumMovieWriteStream(AlbumMovieWriteStreamHandle handle) NN_NOEXCEPT;
    void DiscardAlbumMovieWriteStreamNoDelete(AlbumMovieWriteStreamHandle handle) NN_NOEXCEPT;

    nn::Result StartAlbumMovieWriteStreamDataSection(AlbumMovieWriteStreamHandle handle) NN_NOEXCEPT;
    nn::Result EndAlbumMovieWriteStreamDataSection(AlbumMovieWriteStreamHandle handle) NN_NOEXCEPT;
    nn::Result StartAlbumMovieWriteStreamMetaSection(AlbumMovieWriteStreamHandle handle) NN_NOEXCEPT;
    nn::Result EndAlbumMovieWriteStreamMetaSection(AlbumMovieWriteStreamHandle handle) NN_NOEXCEPT;

    nn::Result ReadDataFromAlbumMovieWriteStream(size_t* pOutReadSize, void* buffer, size_t size, AlbumMovieWriteStreamHandle handle, int64_t offset) NN_NOEXCEPT;
    nn::Result WriteDataToAlbumMovieWriteStream(AlbumMovieWriteStreamHandle handle, int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT;
    nn::Result WriteMetaToAlbumMovieWriteStream(AlbumMovieWriteStreamHandle handle, const void* buffer, size_t size, uint64_t makerNoteVersion, int64_t makerNoteOffset, int64_t makerNoteSize) NN_NOEXCEPT;
    nn::Result GetAlbumMovieWriteStreamBrokenReason(AlbumMovieWriteStreamHandle handle) NN_NOEXCEPT;
    nn::Result GetAlbumMovieWriteStreamDataSize(int64_t* pOutValue, AlbumMovieWriteStreamHandle handle) NN_NOEXCEPT;
    nn::Result SetAlbumMovieWriteStreamDataSize(AlbumMovieWriteStreamHandle handle, int64_t value) NN_NOEXCEPT;

//! @}

}}
